 /**
  *    \file    dice/src/be/BEDeclarator.cpp
  *  \brief   contains the implementation of the class CBEDeclarator
  *
  *    \date    01/15/2002
  *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
  */
 /*
  * Copyright (C) 2001-2004
  * Dresden University of Technology, Operating Systems Research Group
  *
  * This file contains free software, you can redistribute it and/or modify
  * it under the terms of the GNU General Public License, Version 2 as
  * published by the Free Software Foundation (see the file COPYING).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  * GNU General Public License for more details.
  *
  * You should have received a copy of the GNU General Public License
  * along with this program; if not, write to the Free Software
  * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  *
  * For different licensing schemes please contact
  * <contact@os.inf.tu-dresden.de>.
  */

#include "BEDeclarator.h"
#include "BEContext.h"
#include "BEExpression.h"
#include "BEFile.h"
#include "BEFunction.h"
#include "BETarget.h"
#include "BEComponent.h"
#include "BEClient.h"
#include "BETypedDeclarator.h"
#include "BEType.h"
#include "BEAttribute.h"
#include "BESwitchCase.h"
#include "BEStructType.h"
#include "BEUnionType.h"
#include "Compiler.h"
#include "Attribute-Type.h"
#include "fe/FEDeclarator.h"
#include "fe/FEEnumDeclarator.h"
#include "fe/FEArrayDeclarator.h"
#include "fe/FEBinaryExpression.h"
#include <sstream>
#include <cassert>
using namespace std;

/** \brief writes the declarator stack to a file
 *  \param pFile the file to write to
 *  \param pStack the declarator stack to write
 *  \param bUsePointer true if one star should be ignored
 */
void CDeclaratorStackLocation::Write(CBEFile *pFile,
    vector<CDeclaratorStackLocation*> *pStack,
    bool bUsePointer)
{
    string sOut;
    CDeclaratorStackLocation::WriteToString(sOut, pStack, bUsePointer);
    *pFile << sOut;
}

/** \brief writes the declarator stack to a string
 *  \param sResult the string to append the decl stack to
 *  \param pStack the declarator stack to write
 *  \param bUsePointer true if one star should be ignored
 */
void CDeclaratorStackLocation::WriteToString(string &sResult,
    vector<CDeclaratorStackLocation*> *pStack,
    bool bUsePointer)
{
    assert(pStack);

    CBEFunction *pFunction = 0;
    CDeclaratorStackLocation *pLast = pStack->front();
    if (pLast && pLast->pDeclarator)
        pFunction = pLast->pDeclarator->GetSpecificParent<CBEFunction>();

    vector<CDeclaratorStackLocation*>::iterator iter;
    for (iter = pStack->begin(); iter != pStack->end(); iter++)
    {
        CBEDeclarator *pDecl = (*iter)->pDeclarator;
        assert(pDecl);
        int nStars = pDecl->GetStars();
        // test for empty array dimensions, which are treated as
        // stars
	DTRACE("%s for %s with %d stars (1)\n", __FUNCTION__,
	    pDecl->GetName().c_str(), nStars);
        if (pDecl->GetArrayDimensionCount())
        {
            int nLevel = 0;
            vector<CBEExpression*>::iterator iterB = 
		pDecl->GetFirstArrayBound();
            CBEExpression *pExpr;
            while ((pExpr = pDecl->GetNextArrayBound(iterB)) != 0)
            {
                if (pExpr->IsOfType(EXPR_NONE))
                {
                    if (!(*iter)->HasIndex(nLevel++))
                        nStars++;
                }
            }
        }
	DTRACE("%s for %s with %d stars (2)\n", __FUNCTION__,
	    pDecl->GetName().c_str(), nStars);
        if ((iter + 1)  == pStack->end())
        {
             // only apply to last element in row
            if (bUsePointer)
                nStars--;
            else
            {
                // test for an array, which only has stars as array dimensions
                if ((*iter)->HasIndex() && 
		    (pDecl->GetArrayDimensionCount() == 0) && 
		    (nStars > 0))
                    nStars--;
            }
        }
	DTRACE("%s for %s with %d stars (3)\n", __FUNCTION__,
	    pDecl->GetName().c_str(), nStars);
        if (pFunction && ((*iter)->nIndex[0] != -3))
        {
	    // this only works if the declarator really belongs to the
	    // parameter
	    CBETypedDeclarator *pTrueParameter = 
		pFunction->FindParameter(pDecl->GetName());
	    // now we do an additional check if we found a parameter
	    if (pTrueParameter &&
		pFunction->HasAdditionalReference(
	    	    pTrueParameter->GetDeclarator()))
    		nStars++;
        }
	DTRACE("%s for %s with %d stars func at %p (4)\n", __FUNCTION__,
	    pDecl->GetName().c_str(), nStars, pFunction);
        // check if the type is a pointer type
        if (pFunction)
        {
            CBETypedDeclarator *pParameter = 
		pFunction->FindParameter(pDecl->GetName(), false);
	    if (!pParameter)
		pParameter = pFunction->FindLocalVariable(pDecl->GetName());
	    CBEType *pType = pParameter ? pParameter->GetType() : 0;
	    // check transmit-as
	    CBEAttribute *pAttr = pParameter ?
		pParameter->FindAttribute(ATTR_TRANSMIT_AS) : 0;
	    if (pAttr)
		pType = pAttr->GetAttrType();
	    DTRACE("%s for %s: pType %p, pointer? %s, string: %s\n",
		__FUNCTION__, pDecl->GetName().c_str(), pType,
		pType ? (pType->IsPointerType() ? "yes" : "no") : "(no type)",
		pParameter ? (pParameter->IsString() ? "yes" : "no") : 
		"(no param)");
	    // XXX removed test for string 
            if (pType && pType->IsPointerType())
                nStars++;
        }
	DTRACE("%s for %s with %d stars (5)\n", __FUNCTION__,
	    pDecl->GetName().c_str(), nStars);
	if (nStars > 0)
	    sResult += "(";
	for (int i=0; i<nStars; i++)
	    sResult += "*";
	sResult += pDecl->GetName();
	if (nStars > 0)
	    sResult += ")";
        for (int i=0; i<(*iter)->GetUsedIndexCount(); i++)
        {
            if ((*iter)->nIndex[i] >= 0)
	    {
		std::ostringstream os;
		os << (*iter)->nIndex[i];
		sResult += string("[") + os.str() + "]";
	    }
            if ((*iter)->nIndex[i] == -2)
		sResult += string("[") + (*iter)->sIndex[i] + "]";
        }
        if ((iter + 1) != pStack->end())
	    sResult += ".";
    }
}

/*********************************************************************/
/* 
 * Declarator 
 */

CBEDeclarator::CBEDeclarator()
: m_sName()
{
    m_nStars = 0;
    m_nBitfields = 0;
    m_nType = DECL_NONE;
    m_nOldType = DECL_NONE;
    m_pInitialValue = 0;
    m_vBounds.clear();
}

CBEDeclarator::CBEDeclarator(CBEDeclarator & src)
: CBEObject(src)
{
    m_sName = src.m_sName;
    m_nStars = src.m_nStars;
    m_nBitfields = src.m_nBitfields;
    m_nType = src.m_nType;
    m_nOldType = src.m_nOldType;
    COPY_VECTOR(CBEExpression, m_vBounds, iterB);
    CLONE_MEM(CBEExpression, m_pInitialValue);
}

/** \brief destructor of this instance */
CBEDeclarator::~CBEDeclarator()
{
    DEL_VECTOR(m_vBounds);
    if (m_pInitialValue)
        delete m_pInitialValue;
}

/** \brief prepares this instance for the code generation
 *  \param pFEDeclarator the corresponding front-end declarator
 *  \return true if the code generation was successful
 *
 * This implementation extracts the name, stars and bitfields from the
 * front-end declarator.
 */
void 
CBEDeclarator::CreateBackEnd(CFEDeclarator * pFEDeclarator)
    throw (CBECreateException*)
{
    // call CBEObject's CreateBackEnd method
    CBEObject::CreateBackEnd(pFEDeclarator);

    CCompiler::VerboseI(0, "CBEDeclarator::%s called\n", __FUNCTION__);
    
    m_sName = pFEDeclarator->GetName();
    m_nBitfields = pFEDeclarator->GetBitfields();
    m_nStars = pFEDeclarator->GetStars();
    m_nType = pFEDeclarator->GetType();
    switch (m_nType)
    {
    case DECL_ARRAY:
        CreateBackEndArray((CFEArrayDeclarator *) pFEDeclarator);
        break;
    case DECL_ENUM:
        CreateBackEndEnum((CFEEnumDeclarator *) pFEDeclarator);
        break;
    }
    
    CCompiler::VerboseD(0, "CBEDeclarator::%s done\n", __FUNCTION__);
}

/** \brief create a simple declarator using name and number of stars
 *  \param sName the name of the declarator
 *  \param nStars the number of asterisks
 *  \return true if successful
 */
void 
CBEDeclarator::CreateBackEnd(string sName, 
    int nStars)
    throw (CBECreateException*)
{
    CCompiler::VerboseI(0, "CBEDeclarator::%s(string: %s)\n", __FUNCTION__,
	sName.c_str());
    
    m_sName = sName;
    m_nStars = nStars;
    
    CCompiler::VerboseD(0, "CBEDeclarator::%s(string) done\n", __FUNCTION__);
}

/** \brief creates the back-end representation for the enum declarator
 *  \param pFEEnumDeclarator the front-end declarator
 *  \return true if code generation was successful
 */
void 
CBEDeclarator::CreateBackEndEnum(CFEEnumDeclarator * pFEEnumDeclarator)
    throw (CBECreateException*)
{
    CCompiler::VerboseI(0, "CBEDeclarator::%s(enum)\n", __FUNCTION__);
    
    if (pFEEnumDeclarator->GetInitialValue())
    {
	CBEClassFactory *pCF = CCompiler::GetClassFactory();
        m_pInitialValue = pCF->GetNewExpression();
        m_pInitialValue->SetParent(this);
	try
	{
	    m_pInitialValue->CreateBackEnd(pFEEnumDeclarator->GetInitialValue());
	}
	catch (CBECreateException *e)
        {
            delete m_pInitialValue;
            m_pInitialValue = 0;
            throw;
        }
    }
    
    CCompiler::VerboseD(0, "CBEDeclarator::%s(enum) done\n", __FUNCTION__);
}

/** \brief creates the back-end representation for the array declarator
 *  \param pFEArrayDeclarator the respective front-end declarator
 *  \return true if code generation was successful
 */
void 
CBEDeclarator::CreateBackEndArray(CFEArrayDeclarator * pFEArrayDeclarator)
    throw (CBECreateException*)
{
    CCompiler::VerboseI(0, "CBEDeclarator::%s(array)\n", __FUNCTION__);
    // iterate over front-end bounds
    int nMax = pFEArrayDeclarator->GetDimensionCount();
    for (int i = 0; i < nMax; i++)
    {
        CFEExpression *pLower = pFEArrayDeclarator->GetLowerBound(i);
        CFEExpression *pUpper = pFEArrayDeclarator->GetUpperBound(i);
        CBEExpression *pBound = GetArrayDimension(pLower, pUpper);
        AddArrayBound(pBound);
    }
    CCompiler::VerboseD(0, "CBEDeclarator::%s(array) done\n", __FUNCTION__);
}

/** \brief writes the declarator to the target file
 *  \param pFile the file to write to
 *
 * This implementation writes the declarator as is (all stars, bitfields,
 * etc.).
 */
void 
CBEDeclarator::WriteDeclaration(CBEFile * pFile)
{
    CCompiler::VerboseI(0, "CBEDeclarator::%s called for %s\n", __FUNCTION__,
        m_sName.c_str());
    if (m_nType == DECL_ENUM)
    {
        WriteEnum(pFile);
	CCompiler::VerboseD(0, "CBEDeclarator::%s returns\n", __FUNCTION__);
        return;
    }
    // write stars
    for (int i = 0; i < m_nStars; i++)
        *pFile << "*";
    // write name
    *pFile << m_sName;
    // write bitfields
    if (m_nBitfields > 0)
        *pFile << ":" << m_nBitfields;
    // array dimensions
    if (IsArray())
        WriteArray(pFile);
    
    CCompiler::VerboseD(0, "CBEDeclarator::%s returned\n", __FUNCTION__);
}

/** \brief writes an array declarator
 *  \param pFile the file to write to
 *
 * This implementation has to write the array dimensions only. This is done by
 * iterating over them and writing them into brackets ('[]').
 */
void
CBEDeclarator::WriteArray(CBEFile * pFile)
{
    vector<CBEExpression*>::iterator iterB = GetFirstArrayBound();
    CBEExpression *pBound;
    while ((pBound = GetNextArrayBound(iterB)) != 0)
    {
	*pFile << "[";
        pBound->Write(pFile);
	*pFile << "]";
    }
}

/** \brief write an array declarator
 *  \param pFile the file to write to
 *
 * The differene to WriteArray is to skip unbound array dimensions.
 * We can determine an unbound array dimension by checking its integer value.
 * It it is 0 then its an unbound dimension.
 */
void 
CBEDeclarator::WriteArrayIndirect(CBEFile * pFile)
{
    vector<CBEExpression*>::iterator iterB = GetFirstArrayBound();
    CBEExpression *pBound;
    while ((pBound = GetNextArrayBound(iterB)) != 0)
    {
        if (pBound->GetIntValue() == 0)
            continue;
	*pFile << "[";
        pBound->Write(pFile);
	*pFile << "]";
    }
}

/** \brief writes an enum declarator
 *  \param pFile the file to write to
 */
void 
CBEDeclarator::WriteEnum(CBEFile * pFile)
{
    assert(false);
}

/** \brief creates a new back-end array bound using front-end array bounds
 *  \param pLower the lower array bound
 *  \param pUpper the upper array bound
 *  \return the new back-end array bound
 *
 * The expression may have the following values:
 * -# pLower == 0 && pUpper == 0
 * -# pLower == 0 && pUpper != 0
 * -# pLower != 0 && pUpper != 0
 * There is not such case, that pLower != 0 and pUpper == 0.
 *
 * To let the CreateBE function calculate the value of the expression, we
 * simply define the following resulting front-end expression used by the
 * CreateBE function (respective to the above cases):
 * -# pNew = '*'
 * -# pNew = pUpper
 * -# pNew = (pUpper)-(pLower)
 * This behaviour is also specified in the DCE Specification.
 *
 * For the latter case this implementation creates two primary expression,
 * which are the parenthesis and a binary expression used to subract them.
 */
CBEExpression*
CBEDeclarator::GetArrayDimension(CFEExpression * pLower, 
    CFEExpression * pUpper)
{
    CFEExpression *pNew = 0;
    // first get new front-end expression
    if ((!pLower) && (!pUpper))
    {
        pNew = new CFEExpression(EXPR_NONE);
    }
    else if ((!pLower) && (pUpper))
    {
        if ((pUpper->GetType() == EXPR_CHAR) && (pUpper->GetChar() == '*'))
            pNew = new CFEExpression(EXPR_NONE);
        else
            pNew = pUpper;
    }
    else if ((pLower) && (pUpper))
    {
        // if lower is '*' than there is no lower bound
        if ((pLower->GetType() == EXPR_CHAR) && (pLower->GetChar() == '*'))
        {
            return GetArrayDimension((CFEExpression *) 0, pUpper);
        }
        // if upper is '*' than array has no bound
        if ((pUpper->GetType() == EXPR_CHAR) && (pUpper->GetChar() == '*'))
        {
            pNew = new CFEExpression(EXPR_NONE);
        }
        else
        {
            CFEExpression *pLowerP = new CFEPrimaryExpression(EXPR_PAREN, 
		pLower);
            CFEExpression *pUpperP = new CFEPrimaryExpression(EXPR_PAREN, 
		pUpper);
            pNew = new CFEBinaryExpression(EXPR_BINARY, pUpperP, EXPR_MINUS, 
		pLowerP);
        }
    }
    // create new back-end expression
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEExpression *pReturn = pCF->GetNewExpression();
    pReturn->SetParent(this);
    try
    {
	pReturn->CreateBackEnd(pNew);
    }
    catch (CBECreateException *e)
    {
        delete pReturn;
	e->Print();
	delete e;
        return 0;
    }

    return pReturn;
}

/** \brief adds another array bound to the bounds vector
 *  \param pBound the bound to add
 *
 * This implementation creates the vector if not existing
 */
void 
CBEDeclarator::AddArrayBound(CBEExpression * pBound)
{
    if (!pBound)
        return;
    if (m_vBounds.empty())
    {
        m_nOldType = m_nType;
        m_nType = DECL_ARRAY;
    }
    m_vBounds.push_back(pBound);
    pBound->SetParent(this);
}

/** \brief removes an array bound from the bounds vector
 *  \param pBound the bound to remove
 */
void 
CBEDeclarator::RemoveArrayBound(CBEExpression *pBound)
{
    if (!pBound)
        return;
    vector<CBEExpression*>::iterator iter;
    for (iter = m_vBounds.begin(); iter != m_vBounds.end(); iter++)
    {
        if (*iter == pBound)
        {
            m_vBounds.erase(iter);
            break;
        }
    }
    if (m_vBounds.empty())
        m_nType = m_nOldType;
}

/** \brief retrieves a pointer to the first array bound
 *  \return pointer to first array bound
 */
vector<CBEExpression*>::iterator 
CBEDeclarator::GetFirstArrayBound()
{
    return m_vBounds.begin();
}

/** \brief retrieves a reference to the next array bound
 *  \param iter the pointer to the next array boundary
 *  \return a reference to the next array boundary
 */
CBEExpression*
CBEDeclarator::GetNextArrayBound(vector<CBEExpression*>::iterator &iter)
{
    if (iter == m_vBounds.end())
        return 0;
    return *iter++;
}

/** \brief calculates the size of the declarator
 *  \return size of the declarator (or -x if pointer, where x is the number of stars)
 *
 * The size of a declarator is 1. An exception is the array declarator,
 * which's size is calculated from the array dimensions. If the declarator has
 * any stars in front of it, it may have a undefined size, since the asterisk
 * indicates pointers.
 *
 * If the integer value of an array bound is not determinable, we return the
 * number of unbound dimension plus the number of asterisks. We add these two
 * values, so we can later determine if the value returned by GetSize were
 * only asterisks or also unbound array dimensions.
 *
 * Another exception is the usage of bitfields. If the declarator has
 * bitfields asscoiated with it the function returns zero. The caller has to
 * know that the size of zero means possible bitfields and ask for them
 * seperately.
 */
int CBEDeclarator::GetSize()
{
    if (!IsArray())
    {
        if (m_nStars > 0)
            return -(m_nStars);
        // if bitfields: return 0
        if (m_nBitfields)
            return 0;
        return 1;
    }

    int nSize = 1;
    int nFakeStars = 0;

    vector<CBEExpression*>::iterator iterB = GetFirstArrayBound();
    CBEExpression *pBound;
    while ((pBound = GetNextArrayBound(iterB)) != 0)
    {
        int nVal = pBound->GetIntValue();
        if (nVal == 0)    // no integer value
            nFakeStars++;
        else
            nSize *= ((nVal < 0) ? -nVal : nVal);
    }
    if (nFakeStars > 0)
        return -(nFakeStars + m_nStars);
    return nSize;
}

/** \brief return the maximum size of the declarator
 *  \return maximum size in bytes
 */
int CBEDeclarator::GetMaxSize()
{
    int nStars = m_nStars;
    // deduct from stars one if this is an out reference
    CBETypedDeclarator *pParameter = GetSpecificParent<CBETypedDeclarator>();
    if (pParameter && pParameter->FindAttribute(ATTR_OUT))
	nStars--;
    
    if (!IsArray())
    {
        if (nStars > 0)
            return -(nStars);
        // if bitfields: return 0
        if (m_nBitfields)
            return 0;
        return 1;
    }
    else
    {
        if ((GetArrayDimensionCount() == 0) &&
            (nStars > 0))
        {
            // this is a weird situation:
            // we have an unbound array, but express it
            // using '*' instead of '[]'
            // Nonetheless the DECL_ARRAY is set...
            //
            // To make the MAX algorithms work, this has to
            // return a negative value
            return -(nStars);
        }
    }

    int nSize = 1;
    int nFakeStars = 0;

    vector<CBEExpression*>::iterator iterB = GetFirstArrayBound();
    CBEExpression *pBound;
    while ((pBound = GetNextArrayBound(iterB)) != 0)
    {
        int nVal = pBound->GetIntValue();
        if (nVal == 0)    // no integer value
            nFakeStars++;
        else
            nSize *= ((nVal < 0) ? -nVal : nVal);
    }

    if (nFakeStars > 0)
        return -(nFakeStars + nStars);
    return nSize;
}

/** \brief returns the number of stars
 *  \return the value of m_nStars
 */
int CBEDeclarator::GetStars()
{
    return m_nStars;
}

/** \brief simply prints the name of the declarator
 *  \param pFile the file to write to
 */
void CBEDeclarator::WriteName(CBEFile * pFile)
{
    *pFile << m_sName;
}

/** \brief simply prints the name of the declarator
 *  \param str the string to write to
 */
void CBEDeclarator::WriteNameToStr(string& str)
{
    str += m_sName;
}

/** \brief writes for every pointer an extra declarator without this pointer
 *  \param pFile the file to write to
 *  \param bUsePointer true if the variable is intended to be used as a pointer
 *  \param bHasPointerType true if the decl's type is a pointer type
 *
 * If we have an array declarator with unbound array dimensions, we
 * have to write them as pointers as well. We can determine these "fake"
 * pointers by checking GetSize.
 *
 * We do not need dereferenced declarators for these unbound arrays.
 *
 * \todo indirect var by underscore hard coded => replace with configurable
 */
void
CBEDeclarator::WriteIndirect(CBEFile * pFile,
    bool bUsePointer,
    bool bHasPointerType)
{
    //     if (m_nType == DECL_ENUM)
    //     {
    //             WriteEnum(pFile);
    //             return;
    //     }

    int nFakeStars = GetFakeStars();
    int nStartStars = m_nStars + nFakeStars;
    // for something like:
    // 'char *buf' we want a declaration of 'char *buf'
    //   (m_nStars:1 nFakeStars:0 bUsePointer:true -> nStartStars:1 nFakeStars:1)
    // 'char buf[]' we want a declaration of 'char *buf'
    //   (m_nStars:0 nFakeStars:1 bUsePointer:true -> nStartStars:1 nFakeStars:1)
    // 'char *buf[]' we want a declaration of 'char **buf, *_buf'
    //   (m_nStars:1 nFakeStars:1 bUsePointer:true -> nStartStars:2 nFakeStars:1)
    // 'char **buf' we want a declaration of 'char **buf, *_buf'
    //   (m_nStars:2 nFakeStars:0 bUsePointer:true -> nStartStars:2 nFakeStars:1)
    if (bUsePointer && (nFakeStars == 0) && (m_nStars > 0))
        nFakeStars++;
    DTRACE("%s: for %s, nFakeStars: %d, nStartStars: %d\n", __FUNCTION__,
	m_sName.c_str(), nFakeStars, nStartStars);
    bool bComma = false;
    for (int nStars = nStartStars; nStars >= nFakeStars; nStars--)
    {
        if (bComma)
	    *pFile << ", ";
        int i;
        // if not first and we have pointer type, write a stars for it
        if (bComma && bHasPointerType)
	    *pFile << "*";
        // write stars
        for (i = 0; i < nStars; i++)
	    *pFile << "*";
        // write underscores
        for (i = 0; i < (nStartStars - nStars); i++)
	    *pFile << "_";
        // write name
	*pFile << m_sName;
        // next please
        bComma = true;
    }

    // we make the last declarator the array declarator
    if (IsArray())
        WriteArrayIndirect(pFile);

    // write bitfields
    //     if (m_nBitfields > 0)
    //             pFile->Print(":%d", m_nBitfields);
}

/** \brief assigns pointered variables a reference to "unpointered" variables
 *  \param pFile the file to write to
 *  \param bUsePointer true if the variable is intended to be used as a pointer
 *
 * Does something like "t1 = \&_t1;" for a variable "CORBA_long *t1"
 *
 * \todo indirect var by underscore hard coded => replace with configurable
 */
void 
CBEDeclarator::WriteIndirectInitialization(CBEFile * pFile,
    bool bUsePointer)
{
    int nFakeStars = GetFakeStars();
    int i, nStartStars = m_nStars + nFakeStars;
    if (bUsePointer && (m_nStars > 0) && (nFakeStars == 0))
        nFakeStars++;
    // FIXME: when adding variable declaration use "temp" var for indirection
    for (int nStars = nStartStars; nStars > nFakeStars; nStars--)
    {
	*pFile << "\t";
        // write name (one _ less)
        for (i = 0; i < (nStartStars - nStars); i++)
	    *pFile << "_";
	*pFile << m_sName << " = ";
        // write name (one more _)
	*pFile << "&";
        for (i = 0; i < (nStartStars - nStars) + 1; i++)
	    *pFile << "_";
	*pFile << m_sName << ";\n";
    }
}

/** \brief assigns pointered variables a reference to "unpointered" variables
 *  \param pFile the file to write to
 *  \param bUsePointer true if the variable is intended to be used as a pointer
 *
 * Does something like "t1 = \&_t1;" for a variable "CORBA_long *t1"
 *
 * \todo indirect var by underscore hard coded => replace with configurable
 */
void
CBEDeclarator::WriteIndirectInitializationMemory(CBEFile * pFile,
    bool bUsePointer)
{
    assert(pFile);
    int nFakeStars = GetFakeStars();
    int nStartStars = m_nStars + nFakeStars;
    if (bUsePointer && (m_nStars > 0) && (nFakeStars == 0))
        nFakeStars++;
    // get function and parameter
    CBETypedDeclarator *pParameter = 0;
    CBEFunction *pFunction = GetSpecificParent<CBEFunction>();
    if (pFunction)
        pParameter = pFunction->FindParameter(m_sName);
    // skip the memory allocation if not preallocated
    if (pParameter && !pParameter->FindAttribute(ATTR_PREALLOC))
	return;
    /** now we have to initialize the fake stars (unbound array dimensions).
     * problem is to use an allocation routine appropriate for this. If there is
     * an max_is or upper bound it is used, but we have no upper bound.
     * CORBA defines to use CORBA_alloc, but this requires size and we don't
     * know how big it actually will be
     *
     * \todo maybe we can propagate the max size when connecting to the server...
     */
    for (int i = 0; i < nFakeStars; i++)
    {
        vector<CDeclaratorStackLocation*> vStack;
        CDeclaratorStackLocation *pLoc = new CDeclaratorStackLocation(this);
        vStack.push_back(pLoc);
	*pFile << "\t";
        for (int j = 0; j < nStartStars - nFakeStars; j++)
	    *pFile << "_";
	*pFile << m_sName << " = ";
        pParameter->GetType()->WriteCast(pFile, true);
        CBEContext::WriteMalloc(pFile, pFunction);
	*pFile << "(";
        /**
         * if the parameter is out and size parameter, then this
         * initialization is done before a component function.
         * We can use the size parameter only if it is set (it can't just
         * before the call, but it is before the component if the size
         * parameter is an IN.) So we check if parent func is CBESwitchCase
         * and size parameter is IN. Then we can use it to set the size
         * otherwise we have to use max.
         */
        bool bUseSize = false;
        if (pFunction && dynamic_cast<CBESwitchCase*>(pFunction))
        {
            // get size parameter
            CBETypedDeclarator *pSizeParam = 0;
            CBEAttribute *pAttr = pParameter->FindAttribute(ATTR_SIZE_IS);
            if (!pAttr)
                pAttr = pParameter->FindAttribute(ATTR_LENGTH_IS);
            if (!pAttr)
                pAttr = pParameter->FindAttribute(ATTR_MAX_IS);
            if (pAttr && pAttr->IsOfType(ATTR_CLASS_IS))
            {
                vector<CBEDeclarator*>::iterator iterD =
                    pAttr->GetFirstIsAttribute();
                CBEDeclarator *pD = pAttr->GetNextIsAttribute(iterD);
                if (pD)
                    pSizeParam = pFunction->FindParameter(pD->GetName());
            }
            if (pSizeParam && pSizeParam->FindAttribute(ATTR_IN))
                bUseSize = true;
        }
        if (bUseSize)
	{
            pParameter->WriteGetSize(pFile, &vStack, pFunction);
	    CBEType *pType = pParameter->GetType();
	    if (pType->GetSize() > 1)
	    {
		*pFile << "*sizeof";
		pType->WriteCast(pFile, false);
	    }
	}
        else
        {
            int nSize = 0;
            pParameter->GetMaxSize(true, nSize);
            *pFile << nSize;
        }
	*pFile << "); /* allocated for unbound array dimensions */\n";
        delete pLoc;
    }
}

/** \brief writes the cleanup routines for dynamically allocated variables
 *  \param pFile the file to write to
 *  \param bUsePointer true if the variable uses a pointer
 */
void CBEDeclarator::WriteCleanup(CBEFile * pFile, bool bUsePointer)
{
    int nFakeStars = GetFakeStars();
    int nStartStars = m_nStars + nFakeStars;
    if (bUsePointer && (m_nStars > 0) && (nFakeStars == 0))
        nFakeStars++;
    // get function and parameter
    CBETypedDeclarator *pParameter = 0;
    CBEFunction *pFunction = GetSpecificParent<CBEFunction>();
    if (pFunction)
        pParameter = pFunction->FindParameter(m_sName);
    // skip the memory allocation if not preallocated
    if (pParameter && !pParameter->FindAttribute(ATTR_PREALLOC))
	return;
    /** now we have to free the fake stars (unbound array dimensions).
     *
     * \todo maybe we can propagate the max size when connecting to the server...
     */
    for (int i = 0; i < nFakeStars; i++)
    {
	*pFile << "\t";
        CBEContext::WriteFree(pFile, GetSpecificParent<CBEFunction>());
	*pFile << "(";
        for (int j = 0; j < nStartStars - nFakeStars; j++)
	    *pFile << "_";
	*pFile << m_sName << ");\n";
    }
}

/** \brief writes the deferred cleanup, which means that this pointer is only rgistered to be cleaned later
 *  \param pFile the file to write to
 *  \param bUsePointer true if variable uses a pointer
 *
 * Same code as in \ref WriteCleanup but uses dice_set_ptr(Env) instead of Free.
 */
void CBEDeclarator::WriteDeferredCleanup(CBEFile* pFile,  bool bUsePointer)
{
    int nFakeStars = GetFakeStars();
    int nStartStars = m_nStars + nFakeStars;
    if (bUsePointer && (m_nStars > 0) && (nFakeStars == 0))
        nFakeStars++;
    // get environment
    CBEFunction *pFunction = GetSpecificParent<CBEFunction>();
    CBEDeclarator *pDecl = 0;
    if (pFunction)
    {
        CBETypedDeclarator *pEnv = pFunction->GetEnvironment();
        pDecl = pEnv->GetDeclarator();
    }
    if (!pDecl)
        return;

    /** now we have to free the fake stars (unbound array dimensions).
     *
     * \todo maybe we can propagate the max size when connecting to the server...
     */
    for (int i = 0; i < nFakeStars; i++)
    {
	*pFile << "\tdice_set_ptr(";
        if (pDecl->GetStars() == 0)
	    *pFile << "&";
        pDecl->WriteName(pFile);
	*pFile << ", ";
        for (i = 0; i < nStartStars - nFakeStars; i++)
	    *pFile << "_";
	*pFile << m_sName << ");\n";
    }
}

/** \brief tests if this declarator is an array declarator
 *  \return true if it is, false if not
 */
bool CBEDeclarator::IsArray()
{
    DTRACE("Test if %s is array (%s)\n", m_sName.c_str(), (m_nType == DECL_ARRAY)?"yes":"no");
    return (m_nType == DECL_ARRAY);
}

/** \brief calculates the number of array dimensions
 *  \return the number of array dimensions
 *
 * One array dimension is an expression in '['']' pairs.
 */
int CBEDeclarator::GetArrayDimensionCount()
{
    if (!IsArray())
        return 0;
    return m_vBounds.size();
}

/** \brief calculates the number of "fake" pointers
 *  \return the number of unbound array dimensions
 *
 * A fake pointer is an unbound array dimension. It is declared as array dimension,
 * but basicaly handled in C as pointer.
 */
int CBEDeclarator::GetFakeStars()
{
    if (!IsArray())
        return 0;
    int nFakeStars = 0;
    vector<CBEExpression*>::iterator iterB = GetFirstArrayBound();
    CBEExpression *pBound;
    while ((pBound = GetNextArrayBound(iterB)) != 0)
    {
        if (pBound->GetIntValue() == 0)
            nFakeStars++;
    }
    return nFakeStars;
}

/** \brief returns the number of bitfields used by this declarator
 *  \return the value of the member m_nBitfields
 */
int CBEDeclarator::GetBitfields()
{
    return m_nBitfields;
}

/** \brief modifies the number of stars
 *  \param nBy the number to add to m_nStars (if it is negative it decrements)
 *  \return the new number of stars
 */
int CBEDeclarator::IncStars(int nBy)
{
    m_nStars += nBy;
    return m_nStars;
}

/** \brief set the number of stars to a fixed value
 *  \param nStars the new number of stars
 *  \return the old number
 */
int CBEDeclarator::SetStars(int nStars)
{
    int nRet = m_nStars;
    m_nStars = nStars;
    return nRet;
}

/** \brief calculates the number of array bounds from the given iterator to the end
 *  \param iter the iterator pointing to the next array bounds
 *  \return the number of array bounds from the iterator to the end of the vector
 */
int
CBEDeclarator::GetRemainingNumberOfArrayBounds(
    vector<CBEExpression*>::iterator iter)
{
    int nCount = 0;
    while (GetNextArrayBound(iter)) nCount++;
    return nCount;
}
