/**
 *    \file    dice/src/be/BEComponent.cpp
 *  \brief   contains the implementation of the class CBEComponent
 *
 *    \date    01/11/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/BEComponent.h"
#include "be/BEFile.h"
#include "be/BEContext.h"
#include "be/BEWaitFunction.h"
#include "be/BESrvLoopFunction.h"
#include "be/BEUnmarshalFunction.h"
#include "be/BEComponentFunction.h"
#include "be/BEWaitAnyFunction.h"
#include "be/BEHeaderFile.h"
#include "be/BEImplementationFile.h"
#include "be/BERoot.h"
#include "be/BENameSpace.h"
#include "Compiler.h"
#include "fe/FEFile.h"
#include "fe/FELibrary.h"
#include "fe/FEInterface.h"
#include "fe/FEOperation.h"
#include <cassert>
using namespace std;

CBEComponent::CBEComponent()
{
}

CBEComponent::CBEComponent(CBEComponent & src)
: CBETarget(src)
{
}

/** \brief destructor
 */
CBEComponent::~CBEComponent()
{

}

/** \brief writes the output
 *
 * The component's write does initiate the write for each of it's files
 */
void CBEComponent::Write()
{
    CCompiler::Verbose(0, "%s called\n", __FUNCTION__);
    WriteHeaderFiles();
    WriteImplementationFiles();
    CCompiler::Verbose(0, "%s done.\n", __FUNCTION__);
}

/** \brief checks whether this interface needs a server loop
 *  \param pFEInterface the interface to check
 *  \return true if a server loop is needed
 *
 * A server loop is not needed if all functions of the interface and its base
 * interfaces are message passing interfaces.  So if at least one of the
 * functions (operations) is any RPC function we need a server loop.
 *
 * There is also no server loop needed if the interface is abstract.
 */
bool
CBEComponent::NeedServerLoop(CFEInterface * pFEInterface)
{
    // test astract attribute
    if (pFEInterface->FindAttribute(ATTR_ABSTRACT))
        return false;
    // test functions
    vector<CFEOperation*>::iterator iterO = pFEInterface->GetFirstOperation();
    CFEOperation *pOperation;
    while ((pOperation = pFEInterface->GetNextOperation(iterO)) != 0)
    {
        if (!(pOperation->FindAttribute(ATTR_IN)) &&
            !(pOperation->FindAttribute(ATTR_OUT)))
        return true;
    }

    vector<CFEInterface*>::iterator iterI = pFEInterface->GetFirstBaseInterface();
    CFEInterface *pBase;
    while ((pBase = pFEInterface->GetNextBaseInterface(iterI)) != 0)
    {
        if (NeedServerLoop(pBase))
            return true;
    }

    return false;
}

/** \brief creates the header files for the component
 *  \param pFEFile the front-end file to use as reference
 *  \return true if successful
 *
 * We could call the base class' implementation but we need a reference to the
 * header, we then would have to search for. Therefore we simply do what the
 * base class would do and use the local reference to the header file.
 */
void
CBEComponent::CreateBackEndHeader(CFEFile * pFEFile)
    throw (CBECreateException*)
{
    // the header files are created on a per IDL file basis, no matter
    // which option is set
    CBEHeaderFile *pHeader = CCompiler::GetClassFactory()->GetNewHeaderFile();
    AddFile(pHeader);
    try
    {
	pHeader->CreateBackEnd(pFEFile, FILETYPE_COMPONENTHEADER);
    }
    catch (CBECreateException *e)
    {
        RemoveFile(pHeader);
        delete pHeader;
	throw;
    }
    CBERoot *pRoot = GetSpecificParent<CBERoot>();
    pRoot->AddToFile(pHeader);
    // add include of opcode file to header file
    if (!CCompiler::IsOptionSet(PROGRAM_NO_OPCODES))
    {
        // get name
        // do not use include file name, since we assume, that opcode
        // file is in same directory
	CBENameFactory *pNF = CCompiler::GetNameFactory();
        string sOpcode = pNF->GetFileName(pFEFile, FILETYPE_OPCODE);
        pHeader->AddIncludedFileName(sOpcode, true, false, pFEFile);
    }
}

/** \brief create the back-end implementation file
 *  \param pFEFile the front-end file to use as reference
 *  \return true if successful
 */
void
CBEComponent::CreateBackEndImplementation(CFEFile * pFEFile)
    throw (CBECreateException*)
{
    string exc = string(__FUNCTION__);
    // find appropriate header file
    CBEHeaderFile *pHeader = FindHeaderFile(pFEFile, FILETYPE_COMPONENTHEADER);
    if (!pHeader)
    {
	exc += " failed, because could not find header file for " +
	    pFEFile->GetFileName();
	throw new CBECreateException(exc);
    }
    // the implementation files are created on a per IDL file basis, no matter
    // which option is set
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEImplementationFile *pImpl = pCF->GetNewImplementationFile();
    AddFile(pImpl);
    pImpl->SetHeaderFile(pHeader);
    try
    {
	pImpl->CreateBackEnd(pFEFile, FILETYPE_COMPONENTIMPLEMENTATION);
    }
    catch (CBECreateException *e)
    {
        RemoveFile(pImpl);
        delete pImpl;
	throw;
    }

    CBERoot *pRoot = GetSpecificParent<CBERoot>();
    assert(pRoot);
    pRoot->AddToFile(pImpl);
    // if create component function, we use seperate file for this
    if (CCompiler::IsOptionSet(PROGRAM_GENERATE_TEMPLATE))
    {
        pImpl = pCF->GetNewImplementationFile();
        AddFile(pImpl);
        pImpl->SetHeaderFile(pHeader);
	try
	{
	    pImpl->CreateBackEnd(pFEFile, FILETYPE_TEMPLATE);
	}
	catch (CBECreateException *e)
        {
            RemoveFile(pImpl);
            delete pImpl;
	    throw;
        }
        pRoot->AddToFile(pImpl);
    }
}

