/**
 *    \file    dice/src/be/BEClass.h
 *  \brief   contains the declaration of the class CBEClass
 *
 *    \date    Tue Jun 25 2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
 /*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

/** preprocessing symbol to check header file */

#ifndef BECLASS_H
#define BECLASS_H

#include <be/BEObject.h>
#include <vector>
using namespace std;

class CBEHeaderFile;
class CBEImplementationFile;
class CBEAttribute;
class CBEConstant;
class CBEType;
class CBETypedef;
class CBETypedDeclarator;
class CBEFunction;
class CBENameSpace;
class CBEFile;
class CBEMsgBuffer;
class CBESrvLoopFunction;

class CFEInterface;
class CFEAttribute;
class CFEConstDeclarator;
class CFETypedDeclarator;
class CFEOperation;
class CFEConstructedType;
class CFEAttributeDeclarator;

/** \struct CPredefinedFunctionID
 *  \ingroup backend
 *  \brief helper class to specify a user defined function ID (uuid attribute)
 */
typedef struct {
    /** \var string m_sName
     *  \brief the name of the function
     */
    string m_sName;
    /** \var int m_nNumber
     *  \brief its function id
     */
    int m_nNumber;
} CPredefinedFunctionID;

/** \class CFunctionGroup
 *  \ingroup backend
 *  \brief helper class to group multiple BE functions belonging to one FE function
 */
class CFunctionGroup : public CObject
{

public:
    /** \brief creates the function group
     *  \param pFEOperation the reference to the front-end function
     */
    CFunctionGroup(CFEOperation *pFEOperation);
    ~CFunctionGroup();

    string GetName();
    CFEOperation *GetOperation();
    void AddFunction(CBEFunction *pFunction);
    vector<CBEFunction*>::iterator GetFirstFunction();
    CBEFunction *GetNextFunction(vector<CBEFunction*>::iterator &iter);

protected:
    /** \var CFEOperation *m_pFEOperation
     *  \brief the reference to the "origin" of this group
     */
    CFEOperation *m_pFEOperation;

    /** \var vector<CBEFunction*> m_vFunctions
     *  \brief the back-end function belonging to the front-end function
     */
    vector<CBEFunction*> m_vFunctions;
};

/** \class CBEClass
 *  \ingroup backend
 *  \brief represents the front-end interface
 *
 * This class is the back-end equivalent to the front-end interface.
 */
class CBEClass : public CBEObject
{
public:
    /** creates an instance of this class */
    CBEClass();
    ~CBEClass();

public: // Public methods
    void CreateBackEnd(CFEInterface *pFEInterface)
	throw (CBECreateException*);
    bool AddToFile(CBEHeaderFile *pHeader);
    bool AddToFile(CBEImplementationFile *pImpl);
    int GetParameterCount(int nFEType, bool& bSameCount, int nDirection = 0);
    int GetStringParameterCount(int nDirection,
	int nMustAttrs = 0, int nMustNotAttrs = 0);
    int GetSize(int nDirection);
    string GetName();

    void Write(CBEHeaderFile *pFile);
    void Write(CBEImplementationFile *pFile);
    void WriteClassName(CBEFile *pFile);

    CBEAttribute* GetNextAttribute(vector<CBEAttribute*>::iterator &iter);
    vector<CBEAttribute*>::iterator GetFirstAttribute();
    void RemoveAttribute(CBEAttribute *pAttribute);
    CBEAttribute* FindAttribute(int nAttrType);
    void AddAttribute(CBEAttribute *pAttribute);

    CBEClass* GetNextBaseClass(vector<CBEClass*>::iterator &iter);
    vector<CBEClass*>::iterator GetFirstBaseClass();
    void RemoveBaseClass(CBEClass *pClass);
    void AddBaseClass(CBEClass *pClass);

    CBEClass* GetNextDerivedClass(vector<CBEClass*>::iterator &iter);
    vector<CBEClass*>::iterator GetFirstDerivedClass();
    void RemoveDerivedClass(CBEClass *pClass);
    void AddDerivedClass(CBEClass *pClass);

    CBEConstant* GetNextConstant(vector<CBEConstant*>::iterator &iter);
    vector<CBEConstant*>::iterator GetFirstConstant();
    void RemoveConstant(CBEConstant *pConstant);
    void AddConstant(CBEConstant *pConstant);
    CBEConstant* FindConstant(string sConstantName);

    CBETypedDeclarator* GetNextTypedef(
	vector<CBETypedDeclarator*>::iterator &iter);
    vector<CBETypedDeclarator*>::iterator GetFirstTypedef();
    void RemoveTypedef(CBETypedef *pTypedef);
    void AddTypedef(CBETypedef *pTypedef);
    CBETypedDeclarator* FindTypedef(string sTypeName);

    CBEFunction* GetNextFunction(vector<CBEFunction*>::iterator &iter);
    vector<CBEFunction*>::iterator GetFirstFunction();
    void RemoveFunction(CBEFunction *pFunction);
    void AddFunction(CBEFunction *pFunction);
    CBEFunction* FindFunction(string sFunctionName, int nFunctionType);

    bool AddOpcodesToFile(CBEHeaderFile *pFile);
    int GetClassNumber();
    CBENameSpace* GetNameSpace();

    CFunctionGroup* FindFunctionGroup(CBEFunction *pFunction);
    CBESrvLoopFunction* GetSrvLoopFunction();

    void RemoveTypeDeclaration(CBETypedDeclarator *pTypeDecl);
    void AddTypeDeclaration(CBETypedDeclarator *pTypeDecl);
    bool IsTargetFile(CBEHeaderFile * pFile);
    bool IsTargetFile(CBEImplementationFile * pFile);

    CBEType* FindTaggedType(int nType, string sTag);
    CBEType* GetNextTaggedType(vector<CBEType*>::iterator &iter);
    vector<CBEType*>::iterator GetFirstTaggedType();
    void RemoveTaggedType(CBEType *pType);
    void AddTaggedType(CBEType *pType);
    bool HasFunctionWithUserType(string sTypeName, CBEFile *pFile);
    int GetParameterCount(int nMustAttrs, int nMustNotAttrs,
	int nDirection = 0);

    bool HasParametersWithAttribute(int nAttribute1, int nAttribute2 = 0,
	int nAttribute3 = 0);

    /** \brief retrieve a handle to the message buffer
     *  \return a reference to the message buffer
     */
    CBEMsgBuffer* GetMessageBuffer()
    { return m_pMsgBuffer; }

protected:
    void CreateBackEnd(CFEConstDeclarator *pFEConstant)
	throw (CBECreateException*);
    void CreateBackEnd(CFETypedDeclarator *pFETypedef)
	throw (CBECreateException*);
    void CreateBackEnd(CFEAttributeDeclarator *pFEAttrDecl)
	throw (CBECreateException*);
    void CreateBackEnd(CFEAttribute *pFEAttribute)
	throw (CBECreateException*);
    void CreateBackEnd(CFEConstructedType *pFEType)
	throw (CBECreateException*);
    
    void CreateFunctionsNoClassDependency(CFEOperation *pFEOperation)
	throw (CBECreateException*);
    void CreateFunctionsClassDependency(CFEOperation *pFEOperation)
	throw (CBECreateException*);
    virtual void CreateObject(void)
	throw (CBECreateException*);
    virtual void CreateEnvironment(void)
	throw (CBECreateException*);
    
    void AddInterfaceFunctions(CFEInterface* pFEInterface)
	throw (CBECreateException*);
    void AddMessageBuffer(CFEInterface* pFEInterface)
	throw (CBECreateException*);

    void CreateAliasForClass(CFEInterface *pFEInterface)
	throw (CBECreateException*);
    bool AddOpcodesToFile(CFEOperation *pFEOperation, CBEHeaderFile *pFile);

    void WriteElements(CBEHeaderFile *pFile);
    void WriteElements(CBEImplementationFile *pFile);

    virtual void WriteMemberVariables(CBEHeaderFile *pFile);
    virtual void WriteConstructor(CBEHeaderFile *pFile);
    virtual void WriteDestructor(CBEHeaderFile *pFile);

    void WriteTypedef(CBETypedDeclarator *pTypedef, CBEHeaderFile *pFile);
    void WriteTaggedType(CBEType *pType, CBEHeaderFile *pFile);
    void WriteConstant(CBEConstant *pConstant, CBEHeaderFile *pFile);
    void WriteFunction(CBEFunction *pFunction, CBEHeaderFile *pFile);
    void WriteFunction(CBEFunction *pFunction, CBEImplementationFile *pFile);
    virtual void WriteHelperFunctions(CBEHeaderFile *pFile);
    virtual void WriteHelperFunctions(CBEImplementationFile *pFile);

    CFunctionGroup* GetNextFunctionGroup(
	vector<CFunctionGroup*>::iterator &iter);
    vector<CFunctionGroup*>::iterator GetFirstFunctionGroup();
    void RemoveFunctionGroup(CFunctionGroup *pGroup);
    void AddFunctionGroup(CFunctionGroup *pGroup);

    int GetOperationNumber(CFEOperation *pFEOperation);
    bool IsPredefinedID(vector<CPredefinedFunctionID> *pFunctionIDs,
	int nNumber);
    int GetMaxOpcodeNumber(CFEInterface *pFEInterface);
    int GetUuid(CFEOperation *pFEOperation);
    bool HasUuid(CFEOperation *pFEOperation);
    int GetInterfaceNumber(CFEInterface *pFEInterface);
    int FindInterfaceWithNumber(CFEInterface *pFEInterface,
        int nNumber,
        vector<CFEInterface*> *pCollection);
    int FindPredefinedNumbers(vector<CFEInterface*> *pCollection,
        vector<CPredefinedFunctionID> *pNumbers);
    int CheckOpcodeCollision(CFEInterface *pFEInterface,
        int nOpNumber,
        vector<CFEInterface*> *pCollection,
        CFEOperation *pFEOperation);
    void AddBaseName(string sName);
    int GetFunctionCount(void);
    int GetFunctionWriteCount(CBEFile *pFile);
    
    virtual bool MsgBufferInitialization(void);

    void CreateOrderedElementList(void);
    void InsertOrderedElement(CObject *pObj);

    virtual void WriteExternCStart(CBEFile *pFile);
    virtual void WriteExternCEnd(CBEFile *pFile);

protected: // Protected members
    /**    \var string m_sName
     *  \brief the name of the class
     */
    string m_sName;
    /** \var CBEMsgBuffer *m_pMsgBuffer
     *  \brief a reference to the server's message buffer
     */
    CBEMsgBuffer *m_pMsgBuffer;
    /** \var vector<CBEFunction*> m_vFunctions
     *  \brief the operation functions, which are used for calculations
     */
    vector<CBEFunction*> m_vFunctions;
    /** \var vector<CBEConstant*> m_vConstants
     *  \brief the constants of the Class
     */
    vector<CBEConstant*> m_vConstants;
    /** \var  vector<CBETypedDeclarator*> m_vTypedefs
     *  \brief the type definition of the Class
     */
    vector<CBETypedDeclarator*> m_vTypedefs;
    /** \var vector<CBEType*> m_vTypeDeclarations
     *  \brief contains the tagged type declarations
     */
    vector<CBEType*> m_vTypeDeclarations;
    /** \var vector<CBEAttribute*> m_vAttributes
     *  \brief contains the attributes of the Class
     */
    vector<CBEAttribute*> m_vAttributes;
    /** \var vector<CBEClass*> m_vBaseClasses
     *  \brief contains references to the base classes
     */
    vector<CBEClass*> m_vBaseClasses;
    /** \var vector<CBEClass*> m_vDerivedClasses
     *  \brief contains references to the derived classes
     */
    vector<CBEClass*> m_vDerivedClasses;
    /** \var vector<CFunctionGroup*> m_vFunctionGroups
     *  \brief contains function groups (BE-functions grouped by their FE-functions)
     */
    vector<CFunctionGroup*> m_vFunctionGroups;
    /** \var vector<string> m_vBaseNames
     *  \brief contains the names to the base interfaces
     */
    vector<string> m_vBaseNames;
    /** \var vector<CObject*> m_vOrderedElements
     *  \brief contains ordered list of elements
     */
    vector<CObject*> m_vOrderedElements;
    /** \var CBETypedDeclarator *m_pCorbaObject
     *  \brief contains a reference to the CORBA Object parameter
     */
    CBETypedDeclarator *m_pCorbaObject;
    /** \var CBETypedDeclarator *m_pCorbaEnv
     *  \brief contains a reference to the CORBA Environment parameter
     */
    CBETypedDeclarator *m_pCorbaEnv;

    friend class CBEMsgBufferType; // needs to access function groups
    friend class CBEMsgBuffer;
};

#endif
