/**
 *    \file    dice/src/Compiler.cpp
 *  \brief   contains the implementation of the class CCompiler
 *
 *    \date    03/06/2001
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "Compiler.h"

#include <unistd.h>
#include <stdarg.h>
#include <cctype>
#include <sys/wait.h>
#include <sys/timeb.h>
#include <errno.h>
#include <limits.h> // needed for realpath
#include <stdlib.h>

#include <string>
#include <algorithm>
#include <iostream>
using namespace std;

#if defined(HAVE_GETOPT_H)
#include <getopt.h>
#endif

#include "File.h"
#include "CParser.h"
#include "CPreProcess.h"
#include "ProgramOptions.h"
#include "be/BERoot.h"
#include "be/BESizes.h"
#include "be/BEContext.h"
// L4 specific
#include "be/l4/L4BENameFactory.h"
// L4V2
#include "be/l4/v2/L4V2BEClassFactory.h"
// L4V2 AMD64
#include "be/l4/v2/amd64/V2AMD64ClassFactory.h"
#include "be/l4/v2/amd64/V2AMD64NameFactory.h"
// L4V2 IA32
#include "be/l4/v2/ia32/V2IA32ClassFactory.h"
// L4X0
#include "be/l4/x0/L4X0BEClassFactory.h"
// L4V4
#include "be/l4/v4/L4V4BEClassFactory.h"
#include "be/l4/v4/ia32/L4V4IA32ClassFactory.h"
#include "be/l4/v4/L4V4BENameFactory.h"
// Sockets
#include "be/sock/SockBEClassFactory.h"
// CDR
#include "be/cdr/CCDRClassFactory.h"
// Language C
//#include "be/lang/c/LangCClassFactory.h"
// Language C++
//#include "be/lang/cxx/LangCPPClassFactory.h"

#include "fe/FEFile.h"
#include "fe/FELibrary.h"
#include "fe/FEInterface.h"
#include "fe/FEOperation.h"

// dynamic loadable modules
#include <dlfcn.h>

//@{
/** some config variables */
extern const char* dice_version;
extern const char* dice_build;
extern const char* dice_user;
//@}

/** defines the number of characters for dependency output per line */
#define MAX_SHELL_COLS 80

/////////////////////////////////////////////////////////////////////////////////
// error handling
//@{
/** globale variables used by the parsers to count errors and warnings */
int errcount = 0;
int erroccured = 0;
int warningcount = 0;
//@}

/**    debugging helper variable */
int nGlobalDebug = 0;

//@{
/** global variables for argument parsing */
extern char *optarg;
extern int optind, opterr, optopt;
//@}

//@{
/** static members */
unsigned int CCompiler::m_nOptions[PROGRAM_OPTION_GROUPS] = { 0, 0, 0, 0 };
unsigned int CCompiler::m_nBackEnd = 0;
int CCompiler::m_nVerboseInd = -1;
unsigned long CCompiler::m_nVerboseLevel = 0;
unsigned long CCompiler::m_nWarningLevel = 0;
int CCompiler::m_nDumpMsgBufDwords = 0;
CBENameFactory *CCompiler::m_pNameFactory = 0;
CBEClassFactory *CCompiler::m_pClassFactory = 0;
CBESizes *CCompiler::m_pSizes = 0;
/* they have empty constructors because we test for this */
string CCompiler::m_sTraceMsgBufFunc = string();
string CCompiler::m_sTraceServerFunc = string();
string CCompiler::m_sTraceClientFunc = string();
string CCompiler::m_sTraceLib = string();
string CCompiler::m_sInitRcvStringFunc;
string CCompiler::m_sFilePrefix;
string CCompiler::m_sOutputDir;
string CCompiler::m_sIncludePrefix;
//@}

CCompiler::CCompiler()
{
    m_bVerbose = false;
    m_nUseFrontEnd = USE_FE_NONE;
    m_nOpcodeSize = 0;
    m_pRootBE = 0;
    m_sDependsFile = string();
}

/** cleans up the compiler object */
CCompiler::~CCompiler()
{
    // we delete the preprocessor here, because ther should be only
    // one for the whole compiler run.
    CPreProcess *pPreProcess = CPreProcess::GetPreProcessor();
    delete pPreProcess;
}

/**
 *  \brief parses the arguments of the compiler call
 *  \param argc the number of arguments
 *  \param argv the arguments
 */
void CCompiler::ParseArguments(int argc, char *argv[])
{
    int c;
#if defined(HAVE_GETOPT_LONG)
    int index = 0;
#endif
    bool bHaveFileName = false;
    unsigned long nNoWarning = 0;
    bool bShowHelp = false;

#if defined(HAVE_GETOPT_LONG)
    static struct option long_options[] = {
        {"client", 0, 0, 'c'},
        {"server", 0, 0, 's'},
        {"create-skeleton", 0, 0, 't'},
        {"template", 0, 0, 't'},
        {"no-opcodes", 0, 0, 'n'},
        {"create-inline", 2, 0, 'i'},
        {"filename-prefix", 1, 0, 'F'},
        {"include-prefix", 1, 0, 'p'},
        {"verbose", 2, 0, 'v'},
        {"corba", 0, 0, 'C'},
	{"capidl", 0, 0, 'A'},
        {"preprocess", 1, 0, 'P'},
        {"f", 1, 0, 'f'},
        {"B", 1, 0, 'B'},
        {"stop-after-preprocess", 2, 0, 'E'},
        {"include", 1, 0, 'I'},
        {"nostdinc", 0, 0, 'N'},
        {"optimize", 2, 0, 'O'},
        {"testsuite", 2, 0, 'T'},
        {"version", 0, 0, 'V'},
        {"message-passing", 0, 0, 'm'},
        {"with-cpp", 1, 0, 'x'},
        {"o", 1, 0, 'o'},
        {"M", 2, 0, 'M'},
        {"W", 1, 0, 'W'},
        {"D", 1, 0, 'D'},
        {"help", 0, 0, 'h'},
        {0, 0, 0, 0}
    };
#endif

    // prevent getopt from writing error messages
    opterr = 0;
    // obtain a reference to the pre-processor
    // and create one if not existent
    CPreProcess *pPreProcess = CPreProcess::GetPreProcessor();


    while (1)
    {
#if defined(HAVE_GETOPT_LONG)
        c = getopt_long_only(argc, argv, "cstni::v::hF:p:CP:f:B:E::O::VI:NT::M::W:D:x:o:", long_options, &index);
#else
        // n has an optional parameter to recognize -nostdinc and -n (no-opcode)
        c = getopt(argc, argv, "cstn::i::v::hF:p:CP:f:B:E::O::VI:NT::M::W:D:x:o:");
#endif

        if (c == -1)
        {
            bHaveFileName = true;
            break;        // skip - evaluate later
        }

        Verbose("Read argument %c\n", c);

        switch (c)
        {
        case '?':
            // Unknown options might be used by plugins
#if defined(HAVE_GETOPT_LONG)
            Warning("unrecognized option: %s (%d)\n", argv[optind - 1],
		optind);
	    Warning("Use \'--help\' to show valid options.\n");
	    Warning("However, plugins might process this option.\n");
#else
            Warning("unrecognized option: %s (%d)\n", argv[optind], optind);
	    Warning("Use \'-h\' to show valid options.\n");
	    Warning("However, plugins might process this option.\n");
#endif
            break;
        case ':':
            // Error exits
            Error("missing argument for option: %s\n", argv[optind - 1]);
            break;
        case 'c':
            Verbose("Create client-side code.\n");
            SetOption(PROGRAM_GENERATE_CLIENT);
            break;
        case 's':
            Verbose("Create server/component-side code.\n");
            SetOption(PROGRAM_GENERATE_COMPONENT);
            break;
        case 't':
            Verbose("create skeletons enabled\n");
            SetOption(PROGRAM_GENERATE_TEMPLATE);
            break;
        case 'i':
            {
                // there may follow an optional argument stating whether this is "static" or "extern" inline
                SetOption(PROGRAM_GENERATE_INLINE);
                if (!optarg)
                {
                    Verbose("create client stub as inline\n");
                }
                else
                {
                    // make upper case
                    string sArg(optarg);
                    transform(sArg.begin(), sArg.end(), sArg.begin(), _toupper);
                    while (sArg[0] == '=')
                        sArg.erase(sArg.begin());
                    if (sArg == "EXTERN")
                    {
                        SetOption(PROGRAM_GENERATE_INLINE_EXTERN);
                        UnsetOption(PROGRAM_GENERATE_INLINE_STATIC);
                        Verbose("create client stub as extern inline\n");
                    }
                    else if (sArg == "STATIC")
                    {
                        SetOption(PROGRAM_GENERATE_INLINE_STATIC);
                        UnsetOption(PROGRAM_GENERATE_INLINE_EXTERN);
                        Verbose("create client stub as static inline\n");
                    }
                    else
                    {
                        Warning("dice: Inline argument \"%s\" not supported. (assume none)", optarg);
                        UnsetOption(PROGRAM_GENERATE_INLINE_EXTERN);
                        UnsetOption(PROGRAM_GENERATE_INLINE_STATIC);
                    }
                }
            }
            break;
        case 'n':
#if !defined(HAVE_GETOPT_LONG)
            {
                // check if -nostdinc is meant
                string sArg(optarg);
                if (sArg.empty())
                {
                    Verbose("create no opcodes\n");
                    SetOption(PROGRAM_NO_OPCODES);
                }
                else if (sArg == "ostdinc")
                {
                    Verbose("no standard include paths\n");
                    pPreProcess->AddCPPArgument(string("-nostdinc"));
                }
            }
#else
            Verbose("create no opcodes\n");
            SetOption(PROGRAM_NO_OPCODES);
#endif
            break;
        case 'v':
            {
		int nVerboseLevel = 0;
                SetOption(PROGRAM_VERBOSE);
                m_bVerbose = true;
                if (!optarg)
                    Verbose("Verbose level %d enabled\n", nVerboseLevel);
                else
                {
                    nVerboseLevel = atoi(optarg);
                    if ((nVerboseLevel < 0) ||
			(nVerboseLevel > PROGRAM_VERBOSE_MAXLEVEL))
                    {
                        Warning("dice: Verbose level %d not supported in this version.", nVerboseLevel);
                        nVerboseLevel = max(min(nVerboseLevel, 
				PROGRAM_VERBOSE_MAXLEVEL), 0);
                    }
                    Verbose("Verbose level %d enabled\n", nVerboseLevel);
                }
		CCompiler::SetVerboseLevel(nVerboseLevel);
            }
            break;
        case 'h':
            bShowHelp = true;
            break;
        case 'F':
            Verbose("file prefix %s used\n", optarg);
            SetFilePrefix(optarg);
            break;
        case 'p':
	    {
		Verbose("include prefix %s used\n", optarg);
		string sPrefix = optarg;
		// remove trailing slashes
		while (*(sPrefix.end()-1) == '/')
		    sPrefix.erase(sPrefix.end()-1);
		SetIncludePrefix(sPrefix);
	    }
            break;
        case 'C':
            Verbose("use the CORBA frontend\n");
            m_nUseFrontEnd = USE_FE_CORBA;
            break;
	case 'A':
	    Verbose("use the CapIDL frontend\n");
	    m_nUseFrontEnd = USE_FE_CAPIDL;
	    break;
        case 'P':
            Verbose("preprocessor option %s added\n", optarg);
            {
                // check for -I arguments, which we preprocess ourselves as well
                string sArg = optarg;
                pPreProcess->AddCPPArgument(sArg);
                if (sArg.substr(0, 2) == "-I")
                {
                    string sPath = sArg.substr(2);
                    pPreProcess->AddIncludePath(sPath);
                    Verbose("Added %s to include paths\n", sPath.c_str());
                }
            }
            break;
        case 'I':
            Verbose("add include path %s\n", optarg);
            {
                string sPath("-I");
                sPath += optarg;
                pPreProcess->AddCPPArgument(sPath);    // copies sPath
                // add to own include paths
                pPreProcess->AddIncludePath(optarg);
                Verbose("Added %s to include paths\n", optarg);
            }
            break;
        case 'N':
            Verbose("no standard include paths\n");
            pPreProcess->AddCPPArgument(string("-nostdinc"));
            break;
        case 'f':
            {
                // provide flags to compiler: this can be anything
                // make upper case
                string sArg = optarg;
                string sOrig = sArg;
                transform(sArg.begin(), sArg.end(), sArg.begin(), _toupper);
                // test first letter
                char cFirst = sArg[0];
                switch (cFirst)
                {
                case 'A':
                    if (sArg == "ALIGN-TO-TYPE")
                    {
                        SetOption(PROGRAM_ALIGN_TO_TYPE);
                        Verbose("align parameters in message buffer to type\n");
                    }
                    break;
                case 'C':
                    if (sArg == "CORBATYPES")
                    {
			SetOption(PROGRAM_USE_CORBA_TYPES);
			Verbose("Use CORBA types.");
                    }
		    else if (sArg == "CTYPES")
                    {
			Error("Option -fctypes is deprecated.\n");
                    }
                    else if (sArg == "CONST-AS-DEFINE")
                    {
                        SetOption(PROGRAM_CONST_AS_DEFINE);
                        Verbose("print const declarators as define statements\n");
                    }
                    break;
                case 'F':
                    if (sArg == "FORCE-CORBA-ALLOC")
                    {
                        SetOption(PROGRAM_FORCE_CORBA_ALLOC);
                        Verbose("Force use of CORBA_alloc (instead of Environment->malloc).\n");
                        if (IsOptionSet(PROGRAM_FORCE_ENV_MALLOC))
                        {
                            UnsetOption(PROGRAM_FORCE_ENV_MALLOC);
                            Verbose("Overrides previous -fforce-env-malloc.\n");
                        }
                    }
                    else if (sArg == "FORCE-ENV-MALLOC")
                    {
                        SetOption(PROGRAM_FORCE_ENV_MALLOC);
                        Verbose("Force use of Environment->malloc (instead of CORBA_alloc).\n");
                        if (IsOptionSet(PROGRAM_FORCE_CORBA_ALLOC))
                        {
                            UnsetOption(PROGRAM_FORCE_CORBA_ALLOC);
                            Verbose("Overrides previous -fforce-corba-alloc.\n");
                        }
                    }
                    else if (sArg == "FORCE-C-BINDINGS")
                    {
                        SetOption(PROGRAM_FORCE_C_BINDINGS);
                        Verbose("Force use of L4 C bindings (instead of inline assembler).\n");
                    }
                    else if (sArg == "FREE-MEM-AFTER-REPLY")
                    {
                        SetOption(PROGRAM_FREE_MEM_AFTER_REPLY);
                        Verbose("Free memory pointers stored in environment after reply.\n");
                    }
                    else
                    {
                        // XXX FIXME: test for 'FILETYPE=' too
                        UnsetOption(PROGRAM_FILE_MASK);    // remove previous setting
                        if ((sArg == "FIDLFILE") || (sArg == "F1"))
                        {
                            SetOption(PROGRAM_FILE_IDLFILE);
                            Verbose("filetype is set to IDLFILE\n");
                        }
                        else if ((sArg == "FMODULE") || (sArg == "F2"))
                        {
                            SetOption(PROGRAM_FILE_MODULE);
                            Verbose("filetype is set to MODULE\n");
                        }
                        else if ((sArg == "FINTERFACE") || (sArg == "F3"))
                        {
                            SetOption(PROGRAM_FILE_INTERFACE);
                            Verbose("filetype is set to INTERFACE\n");
                        }
                        else if ((sArg == "FFUNCTION") || (sArg == "F4"))
                        {
                            SetOption(PROGRAM_FILE_FUNCTION);
                            Verbose("filetype is set to FUNCTION\n");
                        }
                        else if ((sArg == "FALL") || (sArg == "F5"))
                        {
                            SetOption(PROGRAM_FILE_ALL);
                            Verbose("filetype is set to ALL\n");
                        }
                        else
                        {
                            Error("\"%s\" is an invalid argument for option -ff\n", &optarg[1]);
                        }
                    }
                    break;
                case 'G':
                    {
                        // PROGRAM_GENERATE_LINE_DIRECTIVE
                        if (sArg == "GENERATE-LINE-DIRECTIVE")
                        {
                            SetOption(PROGRAM_GENERATE_LINE_DIRECTIVE);
                            Verbose("generating IDL file line directives in target code");
                        }
                        else
                        {
                            Error("\"%s\" is an invalid argument for option -f\n", optarg);
                        }
                    }
                case 'I':
                    if (sArg.substr(0,14) == "INIT-RCVSTRING")
                    {
                        SetOption(PROGRAM_INIT_RCVSTRING);
                        if (sArg.length() > 14)
                        {
                            string sName = sOrig.substr(15);
                            Verbose("User provides function \"%s\" to init indirect receive strings\n", sName.c_str());
                            SetInitRcvStringFunc(sName);
                        }
                        else
                            Verbose("User provides function to init indirect receive strings\n");
                    }
                    break;
                case 'K':
                    if (sArg == "KEEP-TEMP-FILES")
                    {
                        SetOption(PROGRAM_KEEP_TMP_FILES);
                        Verbose("Keep temporary files generated during preprocessing\n");
                    }
                    break;
                case 'L':
                    if (sArg == "L4TYPES")
                    {
			Error("Option \"%s\" is deprecated.", sOrig.c_str());
                    }
                    break;
                case 'N':
                    if (sArg == "NO-SEND-CANCELED-CHECK")
                    {
                        SetOption(PROGRAM_NO_SEND_CANCELED_CHECK);
                        Verbose("Do not check if the send part of a call was canceled.\n");
                    }
                    else if ((sArg == "NO-SERVER-LOOP") ||
                             (sArg == "NO-SRVLOOP"))
                    {
                        SetOption(PROGRAM_NO_SERVER_LOOP);
                        Verbose("Do not generate server loop.\n");
                    }
                    else if ((sArg == "NO-DISPATCH") ||
                             (sArg == "NO-DISPATCHER"))
                    {
                        SetOption(PROGRAM_NO_DISPATCHER);
                        Verbose("Do not generate dispatch function.\n");
                    }
                    break;
                case 'O':
                    if (sArg.substr(0, 12) == "OPCODE-SIZE=")
                    {
                        string sType = sArg.substr(12);
                        if (isdigit(sArg[13]))
                        {
                            m_nOpcodeSize = atoi(sType.c_str());
                            if (m_nOpcodeSize > 8)
                                m_nOpcodeSize = 8;
                            else if (m_nOpcodeSize > 4)
                                m_nOpcodeSize = 4;
                            else if (m_nOpcodeSize > 2)
                                m_nOpcodeSize = 2;
                        }
                        else
                        {
                            if ((sType == "BYTE") || (sType == "CHAR"))
                                m_nOpcodeSize = 1;
                            else if (sType == "SHORT")
                                m_nOpcodeSize = 2;
                            else if ((sType == "INT") || (sType == "LONG"))
                                m_nOpcodeSize = 4;
                            else if (sType == "LONGLONG")
                                m_nOpcodeSize = 8;
                            else
                                m_nOpcodeSize = 0;
                        }
                        if (m_nOpcodeSize > 0)
                        {
                            Verbose("Set size of opcode type to %d bytes\n", m_nOpcodeSize);
                        }
                        else
                            Verbose("The opcode-size \"%s\" is not supported\n", sType.c_str());
                    }
                    break;
                case 'S':
                    if (sArg == "SERVER-PARAMETER")
                    {
			Error("Option \"%s\" is deprecated.", sOrig.c_str());
                    }
                    break;
                case 'T':
                    if (sArg.substr(0, 12) == "TRACE-SERVER")
                    {
                        SetOption(PROGRAM_TRACE_SERVER);
                        Verbose("Trace messages received by the server loop\n");
                        if (sArg.length() > 12)
                        {
                            string sName = sOrig.substr(13);
                            Verbose("User sets trace function to \"%s\".\n", sName.c_str());
                            SetTraceServerFunc(sName);
                        }
                    }
                    else if (sArg.substr(0, 12) == "TRACE-CLIENT")
                    {
                        SetOption(PROGRAM_TRACE_CLIENT);
                        Verbose("Trace messages send to the server and answers received from it\n");
                        if (sArg.length() > 12)
                        {
                            string sName = sOrig.substr(13);
                            Verbose("User sets trace function to \"%s\".\n", sName.c_str());
                            SetTraceClientFunc(sName);
                        }
                    }
                    else if (sArg.substr(0, 17) == "TRACE-DUMP-MSGBUF")
                    {
                        SetOption(PROGRAM_TRACE_MSGBUF);
                        Verbose("Trace the message buffer struct\n");
                        if (sArg.length() > 17)
                        {
                            // check if lines are meant
                            if (sArg.substr(0, 24) == "TRACE-DUMP-MSGBUF-DWORDS")
                            {
                                if (sArg.length() > 24)
                                {
                                    string sNumber = sArg.substr(25);
                                    int nDwords = atoi(sNumber.c_str());
                                    if (nDwords >= 0)
                                        SetOption(PROGRAM_TRACE_MSGBUF_DWORDS);
				    SetTraceMsgBufDwords(nDwords);
                                }
                                else
                                    Error("The option -ftrace-dump-msgbuf-dwords expects an argument (e.g. -ftrace-dump-msgbuf-dwords=10).\n");
                            }
                            else
                            {
                                string sName = sOrig.substr(18);
                                Verbose("User sets trace function to \"%s\".\n", sName.c_str());
                                SetTraceMsgBufFunc(sName);
                            }
                        }
                    }
                    else if (sArg.substr(0, 14) == "TRACE-FUNCTION")
                    {
                        if (sArg.length() > 14)
                        {
                            string sName = sOrig.substr(15);
                            Verbose("User sets trace function to \"%s\".\n", sName.c_str());
                            if (GetTraceServerFunc().empty())
                                SetTraceServerFunc(sName);
                            if (GetTraceClientFunc().empty())
                                SetTraceClientFunc(sName);
                            if (GetTraceMsgBufFunc().empty())
                                SetTraceMsgBufFunc(sName);
                        }
                        else
                            Error("The option -ftrace-function expects an argument (e.g. -ftrace-function=LOGl).\n");
                    }
		    else if (sArg.substr(0, 9) == "TRACE-LIB")
		    {
			if (sArg.length() > 9)
			{
			    string sName = sOrig.substr(10);
			    Verbose("User sets tracing lib to \"%s\".\n",
				sName.c_str());
			    SetTraceLib(sName);
			}
			else
			    Error("The option -ftrace-lib expects an argument.\n");
		    }
                    else if ((sArg == "TEST-NO-SUCCESS") || 
			     (sArg == "TEST-NO-SUCCESS-MESSAGE"))
                    {
			Error("Option \"%s\" is deprecated.", sOrig.c_str());
                    }
                    else if (sArg == "TESTSUITE-SHUTDOWN")
                    {
			Error("Option \"%s\" is deprecated.", sOrig.c_str());
                    }
                    break;
                case 'U':
                    if ((sArg == "USE-SYMBOLS") || (sArg == "USE-DEFINES"))
                    {
			Error("Option \"%s\" is deprecated.", sOrig.c_str());
                    }
                    break;
                case 'Z':
                    if (sArg == "ZERO-MSGBUF")
                    {
                        SetOption(PROGRAM_ZERO_MSGBUF);
                        Verbose("Zero message buffer before using.\n");
                    }
                    break;
                default:
                    Warning("unsupported argument \"%s\" for option -f\n", sArg.c_str());
                    break;
                }
            }
            break;
        case 'B':
            {
                // make upper case
                string sArg(optarg);
                transform(sArg.begin(), sArg.end(), sArg.begin(), _toupper);

                // test first letter
                // p - platform
                // i - kernel interface
                // m - language mapping
                char sFirst = sArg[0];
                switch (sFirst)
                {
                case 'P':
                    sArg = sArg.substr(1);
                    if (sArg == "IA32")
                    {
			ModifyBackEnd(PROGRAM_BE_IA32, PROGRAM_BE_PLATFORM);
                        Verbose("use back-end for IA32 platform\n");
                    }
                    else if (sArg == "IA64")
                    {
                        Warning("IA64 back-end not supported yet!");
			ModifyBackEnd(PROGRAM_BE_IA32, PROGRAM_BE_PLATFORM);
                        Verbose("use back-end for IA64 platform\n");
                    }
                    else if (sArg == "ARM")
                    {
			ModifyBackEnd(PROGRAM_BE_ARM, PROGRAM_BE_PLATFORM);
                        Verbose("use back-end for ARM platform\n");
                    }
                    else if (sArg == "AMD64")
                    {
			ModifyBackEnd(PROGRAM_BE_AMD64, PROGRAM_BE_PLATFORM);
                        Verbose("use back-end for AMD64 platform\n");
                    }
                    else
                    {
                        Error("\"%s\" is an invalid argument for option -B/--back-end\n", optarg);
                    }
                    break;
                case 'I':
                    sArg = sArg.substr(1);
                    if (sArg == "V2")
                    {
			ModifyBackEnd(PROGRAM_BE_V2, PROGRAM_BE_INTERFACE);
                        Verbose("use back-end L4 version 2\n");
                    }
                    else if (sArg == "X0")
                    {
			ModifyBackEnd(PROGRAM_BE_X0, PROGRAM_BE_INTERFACE);
                        Verbose("use back-end L4 version X.0\n");
                    }
                    else if (sArg == "X0ADAPT")
                    {
			Error("X0adapt Back-End deprecated.\n");
                    }
                    else if ((sArg == "X2") || (sArg == "V4"))
                    {
			ModifyBackEnd(PROGRAM_BE_V4, PROGRAM_BE_INTERFACE);
                        Verbose("use back-end L4 version 4 (X.2)\n");
                    }
                    else if (sArg == "FLICK")
                    {
                        Warning("Flick compatibility mode is no longer supported");
                    }
                    else if ((sArg == "SOCKETS") || (sArg == "SOCK"))
                    {
			ModifyBackEnd(PROGRAM_BE_SOCKETS, PROGRAM_BE_INTERFACE);
                        Verbose("use sockets back-end\n");
                    }
                    else if (sArg == "CDR")
                    {
			ModifyBackEnd(PROGRAM_BE_CDR, PROGRAM_BE_INTERFACE);
                        Verbose("use CDR back-end\n");
                    }
                    else
                    {
                        Error("\"%s\" is an invalid argument for option -B/--back-end\n", optarg);
                    }
                    break;
                case 'M':
                    sArg = sArg.substr(1);
                    if (sArg == "C")
                    {
			ModifyBackEnd(PROGRAM_BE_C, PROGRAM_BE_LANGUAGE);
                        Verbose("use back-end C language mapping\n");
                    }
                    else if (sArg == "CPP")
                    {
			ModifyBackEnd(PROGRAM_BE_CPP, PROGRAM_BE_LANGUAGE);
                        Verbose("use back-end C++ language mapping\n");
                    }
                    else
                    {
                        Error("\"%s\" is an invalid argument for option -B/--back-end\n", optarg);
                    }
                    break;
                default:
                    // unknown option
                    Error("\"%s\" is an invalid argument for option -B/--back-end\n", optarg);
                }
            }
            break;
        case 'E':
            if (!optarg)
            {
                Verbose("stop after preprocess option enabled.\n");
                SetOption(PROGRAM_STOP_AFTER_PRE);
            }
            else
            {
                string sArg(optarg);
                transform(sArg.begin(), sArg.end(), sArg.begin(), _toupper);
                if (sArg == "XML")
                {
		    Error("Option -EXML is deprecated.\n"); 
                }
                else
                {
                    Warning("unrecognized argument \"%s\" to option -E.\n", sArg.c_str());
                    SetOption(PROGRAM_STOP_AFTER_PRE);
                }
            }
            break;
        case 'o':
            if (!optarg)
            {
                Error("Option -o requires argument\n");
            }
            else
            {
                Verbose("output directory set to %s\n", optarg);
		string sDir = optarg;
		if (!sDir.empty())
		{
		    if (sDir[sDir.length()-1] != '/')
			sDir += "/";
		}
		SetOutputDir(sDir);
            }
            break;
        case 'T':
	    Error("Option \"-T%s\" is deprecated.", optarg ? optarg : "");
            break;
        case 'V':
            ShowVersion();
            break;
        case 'm':
            Verbose("generate message passing functions.\n");
            SetOption(PROGRAM_GENERATE_MESSAGE);
            break;
        case 'M':
            {
                // search for depedency options
                if (!optarg)
                {
		    UnsetOption(PROGRAM_DEPEND_MASK1);
                    SetOption(PROGRAM_DEPEND_M);
                    Verbose("Create dependencies without argument\n");
                }
                else
                {
                    // make upper case
                    string sArg = optarg;
                    transform(sArg.begin(), sArg.end(), sArg.begin(), _toupper);
                    if (sArg == "M")
                    {
			UnsetOption(PROGRAM_DEPEND_MASK1);
                        SetOption(PROGRAM_DEPEND_MM);
                    }
                    else if (sArg == "D")
                    {
			UnsetOption(PROGRAM_DEPEND_MASK1);
                        SetOption(PROGRAM_DEPEND_MD);
                    }
                    else if (sArg == "MD")
                    {
			UnsetOption(PROGRAM_DEPEND_MASK1);
                        SetOption(PROGRAM_DEPEND_MMD);
                    }
		    /* do not unset DEPEND_MASK1 generally, because then -MP
		     * might delete -M. Nor unset it for the following option
		     * for the same reason.
		     */
		    else if (sArg == "F")
		    {
			SetOption(PROGRAM_DEPEND_MF);
			// checking if parameter after -MF starts with a '-'
			// which would make it a likely argument
			if (argv[optind] && (argv[optind])[0] != '-')
			{
			    m_sDependsFile = argv[optind++];
			    Verbose("Set depends file name to \"%s\".\n", 
				m_sDependsFile.c_str());
			}
		    }
		    else if (sArg == "P")
		    {
			SetOption(PROGRAM_DEPEND_MP);
		    }
                    else
                    {
                        Warning("dice: Argument \"%s\" of option -M unrecognized: ignoring.", optarg);
                        SetOption(PROGRAM_DEPEND_M);
                    }
                    Verbose("Create dependencies with argument %s\n", optarg);
                }
            }
            break;
        case 'W':
            if (!optarg)
            {
                Error("The option '-W' has to be used with parameters (see --help for details)\n");
            }
            else
            {
                string sArg = optarg;
                transform(sArg.begin(), sArg.end(), sArg.begin(), _toupper);
                if (sArg == "ALL")
                {
		    SetWarningLevel(PROGRAM_WARNING_ALL);
                    Verbose("All warnings on.\n");
                }
                else if (sArg == "NO-ALL")
                {
                    nNoWarning |= PROGRAM_WARNING_ALL;
                    Verbose("All warnings off.\n");
                }
                else if (sArg == "IGNORE-DUPLICATE-FID")
                {
		    SetWarningLevel(PROGRAM_WARNING_IGNORE_DUPLICATE_FID);
                    Verbose("Warn if duplicate function IDs are ignored.\n");
                }
                else if (sArg == "NO-IGNORE-DUPLICATE-FID")
                {
                    nNoWarning |= PROGRAM_WARNING_IGNORE_DUPLICATE_FID;
                    Verbose("Do not warn if duplicate function IDs are ignored.\n");
                }
                else if (sArg == "PREALLOC")
                {
		    SetWarningLevel(PROGRAM_WARNING_PREALLOC);
                    Verbose("Warn if CORBA_alloc is used.\n");
                }
                else if (sArg == "NO-PREALLOC")
                {
                    nNoWarning |= PROGRAM_WARNING_PREALLOC;
                    Verbose("Do not warn if CORBA_alloc is used.\n");
                }
                else if (sArg == "MAXSIZE")
                {
		    SetWarningLevel(PROGRAM_WARNING_NO_MAXSIZE);
                    Verbose("Warn if max-size attribute is not set for unbound variable sized arguments.\n");
                }
                else if (sArg == "NO-MAXSIZE")
                {
                    nNoWarning |= PROGRAM_WARNING_NO_MAXSIZE;
                    Verbose("Do not warn if max-size attribute is not set for unbound variable sized arguments.\n");
                }
                else if (sArg.substr(0, 2) == "P,")
                {
                    string sTmp(optarg);
                    // remove "p,"
                    string sCppArg = sTmp.substr(2);
                    pPreProcess->AddCPPArgument(sCppArg);
                    Verbose("preprocessor argument \"%s\" added\n", sCppArg.c_str());
                    // check if CPP argument has special meaning
                    if (sCppArg.substr(0, 2) == "-I")
                    {
                        string sPath = sCppArg.substr(2);
                        pPreProcess->AddIncludePath(sPath);
                        Verbose("Added %s to include paths\n", sPath.c_str());
                    }
                }
                else
                {
                    Warning("dice: warning \"%s\" not supported.\n", optarg);
                }
            }
            break;
        case 'D':
            if (!optarg)
            {
                Error("There has to be an argument for the option '-D'\n");
            }
            else
            {
                string sArg("-D");
                sArg += optarg;
                pPreProcess->AddCPPArgument(sArg);
                Verbose("Found symbol \"%s\"\n", optarg);
            }
            break;
        case 'x':
            {
                string sArg = optarg;
                if (!pPreProcess->SetCPP(sArg.c_str()))
                    Error("Preprocessor \"%s\" does not exist.\n", optarg);
                Verbose("Use \"%s\" as preprocessor\n", optarg);
            }
            break;
        default:
            Error("You used an obsolete parameter (%c).\nPlease use dice --help to check your parameters.\n", c);
        }
    }

    if (optind < argc)
    {
	Verbose("Arguments not processed: ");
	while (optind < argc)
	    Verbose("%s ", argv[optind++]);
	Verbose("\n");
    }

    if (bHaveFileName && (argc > 1))
    {
        // infile left (should be last)
        // if argv is "-" the stdin should be used: set m_sInFileName to 0
        if (strcmp(argv[argc - 1], "-"))
            m_sInFileName = argv[argc - 1];
        else
            m_sInFileName = "";
        Verbose("Input file is: %s\n", m_sInFileName.c_str());
    }

    if (m_nUseFrontEnd == USE_FE_NONE)
        m_nUseFrontEnd = USE_FE_DCE;

    if (!IsOptionSet(PROGRAM_FILE_MASK))
        SetOption(PROGRAM_FILE_IDLFILE);

    if (!IsBackEndSet(PROGRAM_BE_INTERFACE))
    {
        if (IsBackEndSet(PROGRAM_BE_ARM))
            ModifyBackEnd(PROGRAM_BE_X0, PROGRAM_BE_INTERFACE);
        else
	    ModifyBackEnd(PROGRAM_BE_V2, PROGRAM_BE_INTERFACE);
    }
    if (!IsBackEndSet(PROGRAM_BE_PLATFORM))
        ModifyBackEnd(PROGRAM_BE_IA32, PROGRAM_BE_PLATFORM);
    if (!IsBackEndSet(PROGRAM_BE_LANGUAGE))
        ModifyBackEnd(PROGRAM_BE_C, PROGRAM_BE_LANGUAGE);
    if (IsBackEndSet(PROGRAM_BE_ARM) &&
        !IsBackEndSet(PROGRAM_BE_X0))
    {
        Warning("The Arm Backend currently works with X0 native only!");
        Warning("  -> Setting interface to X0 native.");
	ModifyBackEnd(PROGRAM_BE_X0, PROGRAM_BE_INTERFACE);
    }
    if (IsBackEndSet(PROGRAM_BE_AMD64) &&
        !IsBackEndSet(PROGRAM_BE_V2))
    {
        Warning("The AMD64 Backend currently works with V2 native only!");
        Warning("  -> Setting interface to V2 native.");
	ModifyBackEnd(PROGRAM_BE_V2, PROGRAM_BE_INTERFACE);
    }
    // with arm we *have to* marshal type aligned
    if (IsBackEndSet(PROGRAM_BE_ARM))
        SetOption(PROGRAM_ALIGN_TO_TYPE);
    // with AMD64 we *have to* use C bindings
    if (IsBackEndSet(PROGRAM_BE_AMD64))
        SetOption(PROGRAM_FORCE_C_BINDINGS);

    if (!IsOptionSet(PROGRAM_GENERATE_CLIENT) &&
        !IsOptionSet(PROGRAM_GENERATE_COMPONENT))
    {
        /* per default generate client AND server */
        SetOption(PROGRAM_GENERATE_CLIENT);
        SetOption(PROGRAM_GENERATE_COMPONENT);
    }

    // check if tracing functions are set
    if (IsOptionSet(PROGRAM_TRACE_SERVER) &&
	GetTraceServerFunc().empty())
	SetTraceServerFunc("printf");
    if (IsOptionSet(PROGRAM_TRACE_CLIENT) &&
	GetTraceClientFunc().empty())
	SetTraceClientFunc("printf");
    if (IsOptionSet(PROGRAM_TRACE_MSGBUF) &&
	GetTraceMsgBufFunc().empty())
	SetTraceMsgBufFunc("printf");

    if (nNoWarning != 0)
	UnsetWarningLevel(nNoWarning);

    // check if dependency options are set correctly
    if (IsOptionSet(PROGRAM_DEPEND_MF) ||
	IsOptionSet(PROGRAM_DEPEND_MP))
    {
	if (!IsOptionSet(PROGRAM_DEPEND_M) &&
	    !IsOptionSet(PROGRAM_DEPEND_MM) &&
	    !IsOptionSet(PROGRAM_DEPEND_MD) &&
	    !IsOptionSet(PROGRAM_DEPEND_MMD))
	{
	    Error("Option %s requires one of -M,-MM,-MD,-MMD.\n",
		IsOptionSet(PROGRAM_DEPEND_MP) ? "-MP" : "-MF");
	}
    }
    if (IsOptionSet(PROGRAM_DEPEND_MF) &&
	m_sDependsFile.empty())
    {
	Error("Option -MF requires argument.\n");
    }
	

    // init plugins
    InitTraceLib(argc, argv);
    
    if (bShowHelp)
	ShowHelp();
}

/** \brief initializes the trace lib
 *  \param argc the number of arguments
 *  \param argv the arguments of the program
 */
void CCompiler::InitTraceLib(int argc, char *argv[])
{
    // if trace lib was set, then open it and let it parse the arguments. This
    // way it might interpret some arguments that we ignored
    string sTraceLib = CCompiler::GetTraceLib();
    if (sTraceLib.empty())
	return;

    void* lib = dlopen(sTraceLib.c_str(), RTLD_NOW);
    if (lib == NULL)
    {
	fprintf(stderr, "%s\n", dlerror());
	// error exists
	Error("Could not load tracing library \"%s\".\n", sTraceLib.c_str());

	return;
    }

    // get symbol for init function
    void (*init)(int, char**);
    *(void **)(&init) = dlsym(lib, "dice_tracing_init");
    // use error message as error indicator
    const char* errmsg = dlerror();
    if (errmsg != NULL)
    {
	fprintf(stderr, "%s\n", errmsg);
	Error("Could not find symbol for init function.\n");
	return;
    }

    // call init function
    (*init) (argc, argv);
}

/** displays a copyright notice of this compiler */
void CCompiler::ShowCopyright()
{
    if (!m_bVerbose)
        return;
    cout << "DICE (c) 2001-2006 Dresden University of Technology" << endl;
    cout << "Author: Ronald Aigner <ra3@os.inf.tu-dresden.de>" << endl;
    cout << "e-Mail: dice@os.inf.tu-dresden.de" << endl << endl;
}

/** displays a help for this compiler */
void CCompiler::ShowHelp(bool bShort)
{
    ShowCopyright();
    cout << "Usage: dice [<options>] <idl-file>\n"
    //23456789+123456789+123456789+123456789+123456789+123456789+123456789+123456789+
    "\nPre-Processor/Front-End Options:\n"
    " -h"
#if defined(HAVE_GETOPT_LONG)
        ", --help"
#else
        "        "
#endif
                 "                 shows this help\n"
    " -v"
#if defined(HAVE_GETOPT_LONG)
        ", --verbose"
#endif
                    " [<level>]    print verbose output\n"
    "                              with optional verboseness level\n"
    " -C"
#if defined(HAVE_GETOPT_LONG)
    ", --corba"
#else
    "         "
#endif
    "                use the CORBA front-end (scan CORBA IDL)\n"
    " -P"
#if defined(HAVE_GETOPT_LONG)
    ", --preprocess"
#endif
    " <string>  hand <string> as argument to preprocessor\n"
    " -I<string>                 same as -P-I<string>\n"
    " -D<string>                 same as -P-D<string>\n"
    " -nostdinc                  same as -P-nostdinc\n"
    " -Wp,<string>               same as -P<string>\n"
    "    Arguments given to '-P' and '-Wp,' are checked for '-D' and '-I'\n"
    "    as well.\n"
    " -E                         stop processing after pre-processing\n"
    " -M                         print included file tree and stop after\n"
    "                            doing that\n"
    " -MM                        print included file tree for files included\n"
    "                            with '#include \"file\"' and stop\n"
    " -MD                        print included file tree into .d file and\n"
    "                            compile\n"
    " -MMD                       print included file tree for files included\n"
    "                            with '#include \"file\"' into .d file and\n"
    "                            compile\n"
    " -MF <string>               generates dependency output into the\n"
    "                            file specified by <string>\n"
    " --with-cpp=<string>        use <string> as cpp\n"
    "    Dice checks if it can run <string> before using it.\n"
    "\n"
    "\nBack-End Options:\n"
    " -i"
#if defined(HAVE_GETOPT_LONG)
    ", --create-inline"
#endif
    " <mode>  generate client stubs as inline\n";
    if (!bShort)
	cout <<
    "    set <mode> to \"static\" to generate static inline\n"
    "    set <mode> to \"extern\" to generate extern inline\n"
    "    <mode> is optional\n";
    cout <<
    " -n"
#if defined(HAVE_GETOPT_LONG)
    ", --no-opcodes"
#else
    "              "
#endif
    "            do not generate opcodes\n"
    " -c"
#if defined(HAVE_GETOPT_LONG)
    ", --client"
#else
    "          "
#endif
    "                generate client-side code\n"
    " -s"
#if defined(HAVE_GETOPT_LONG)
    ", --server"
#else
    "          "
#endif
    "                generate server/component-side code\n"
    "    if none of the two is specified both are set\n"
    " -t"
#if defined(HAVE_GETOPT_LONG)
    ", --template, --create-skeleton\n"
    "   "
#endif
    "              generate skeleton/templates for server side functions\n"
    " -F"
#if defined(HAVE_GETOPT_LONG)
    ", --filename-prefix"
#endif
    " <string>\n"
    "                 prefix each filename with the string\n"
    " -p"
#if defined(HAVE_GETOPT_LONG)
    ", --include-prefix"
#endif
    " <string>\n"
    "                 prefix each included file with string\n"
    " -o <string>\n"
    "                 specify an output directory (default is .)\n";
    if (bShort)
	cout <<
    " -f<string>      supply flags to compiler\n";
    else
	cout <<
    "\n"
    "Compiler Flags:\n"
    " -f<string>                   supply flags to compiler\n"
    "    if <string> starts with 'F'/'f' (filetype):\n"
    "      set <string> to \"Fidlfile\" for one client C file per IDL file\n"
    "      set <string> to \"Fmodule\" for one client C file per module\n"
    "      set <string> to \"Finterface\" for one client C file per interface\n"
    "      set <string> to \"Ffunction\" for one client C file per function\n"
    "      set <string> tp \"Fall\" for one client C file for everything\n"
    "      alternatively set -fF1, -fF2, -fF3, -fF4, -fF5 respectively\n"
    "      default is -fFidlfile\n"
    "    set <string> to 'corbatypes' to use CORBA-style type names\n"
    "    set <string> to 'opcodesize=<size>' to set the size of the opcode\n"
    "      set <size> to \"byte\" or 1 if opcode should use only 1 byte\n"
    "      set <size> to \"short\" or 2 if opcode should use 2 bytes\n"
    "      set <size> to \"long\" or 4 if opcode should use 4 bytes (default)\n"
    "      set <size> to \"longlong\" or 8 if opcode should use 8 bytes\n"
    "    set <string> to 'init-rcvstring[=<func-name>]' to make the\n"
    "       server-loop use a user-provided function to initialize the\n"
    "       receive buffers of indirect strings\n"
    "    set <string> to 'align-to-type' to align parameters in the message\n"
    "       buffer by the size of their type (or word) (default for ARM)\n"
    "\n"
    "    set <string> to 'force-corba-alloc' to force the usage of the\n"
    "       CORBA_alloc function instead of the CORBA_Environment's malloc\n"
    "       member\n"
    "    set <string> to 'force-env-malloc' to force the usage of\n"
    "       CORBA_Environment's malloc member instead of CORBA_alloc\n"
    "    depending on their order they may override each other.\n"
    "    set <string> to 'free-mem-after-reply' to force freeing of memory\n"
    "       pointers stored in the environment variable using the free\n"
    "       function of the environment or CORBA_free\n"
    "\n"
    "    set <string> to 'force-c-bindings' to force the usage of L4's\n"
    "       C-bindings instead of inline assembler\n"
    "    set <string> to 'no-server-loop' to not generate the server loop\n"
    "       function\n"
    "    set <string> to 'no-dispatcher' to not generate the dispatcher\n"
    "       function\n"
    "\n"
    "  Debug Options:\n"
    "    set <string> to 'trace-server' to trace all messages received by the\n"
    "       server-loop\n"
    "    set <string> to 'trace-client' to trace all messages send to the\n"
    "       server and answers received from it\n"
    "    set <string> to 'trace-dump-msgbuf' to dump the message buffer,\n"
    "       before a call, right after it, and after each wait\n"
    "    set <string> to 'trace-dump-msgbuf-dwords=<number>' to restrict the\n"
    "       number of dumped dwords. <number> can be any positive integer\n"
    "       including zero.\n"
    "    set <string> to 'trace-function=<function>' to use <function>\n"
    "       instead of 'printf' to print trace messages. This option sets\n"
    "       the function for client and server.\n"
    "       If you like to specify different functions for client, server,\n"
    "       or msgbuf, you can use -ftrace-client=<function1>,\n"
    "       -ftrace-server=<function2>, or -ftrace-dump-msgbuf=<function3>\n"
    "       respectively\n"
    "    set <string> to 'trace-lib=<libname>' to use a dynamic loadable \n"
    "       library that implements the tracing hooks. (see documentation for\n"
    "       details)\n"
    "    set <string> to 'zero-msgbuf' to zero out the message buffer before\n"
    "       each and wait/reply-and-wait\n"
    "    if <string> is set to 'use-symbols' or 'use-defines' the symbols\n"
    "       given with '-D' or '-P-D' are not just handed down to the\n"
    "       preprocessor, but also used to simplify the generated code.\n"
    "       USE FOR DEBUGGING ONLY!\n"
    "    if <string> is set to 'no-send-canceled-check' the client call will\n"
    "       not retry to send if it was canceled or aborted by another thread.\n"
    "    if <string> is set to 'const-as-define' all const declarations will\n"
    "       be printed as #define statements\n"
    "    if <string> is set to 'keep-temp-files' Dice will not delete the\n"
    "       temporarely during preprocessing created files.\n"
    "       USE FOR DEBUGGING ONLY!\n"
    "\n";
    cout <<
    " -B"
#if defined(HAVE_GETOPT_LONG)
    ", --back-end"
#endif
    " <string>     defines the back-end to use\n";
    if (!bShort)
	cout <<
    "    <string> starts with a letter specifying platform, kernel interface or\n"
    "    language mapping\n"
    "    p - specifies the platform (IA32, IA64, ARM, AMD64)\n"
    "    i - specifies the kernel interface (v2, x0, v4, sock)\n"
    "    m - specifies the language mapping (C, CPP)\n"
    "    example: -Bpia32 -Biv2 -BmC - which is default\n";
    cout <<
    " -m"
#if defined(HAVE_GETOPT_LONG)
    ", --message-passing"
#else
    "                   "
#endif
    "       generate MP functions for RPC functions as well\n"
    "\nGeneral Compiler Options:\n"
    " -Wall                       ignore all warnings\n"
    " -Wignore-duplicate-fids     duplicate function ID are no errors, but\n"
    "                             warnings\n"
    " -Wprealloc                  warn if memory has to be allocated using\n"
    "                             CORBA_alloc\n"
    " -Wno-maxsize                warn if a variable sized parameter has no\n"
    "                             maximum size to bound its required memory\n"
    "                             use\n"
    "\n\nexample: dice -v -i test.idl\n\n";
    exit(0);
}

/** display the compiler's version (requires a defines for the version number) */
void CCompiler::ShowVersion()
{
    ShowCopyright();
    cout << "DICE " << dice_version << " - built on " << dice_build;
    if (dice_user)
	cout << " by " << dice_user;
    cout << "." << endl;
    exit(0);
}

/** \brief creates a parser object and initializes it
 *
 * The parser object pre-processes and parses the input file.
 */
void CCompiler::Parse()
{
    // if sFilename contains a path, add it to include paths
    CPreProcess *pPreProcess = CPreProcess::GetPreProcessor();
    int nPos;
    if ((nPos = m_sInFileName.rfind('/')) >= 0)
    {
        string sPath("-I");
        sPath += m_sInFileName.substr(0, nPos);
        pPreProcess->AddCPPArgument(sPath);    // copies sPath
        pPreProcess->AddIncludePath(m_sInFileName.substr(0, nPos));
        Verbose("Added %s to include paths\n", m_sInFileName.substr(0, nPos).c_str());
    }
    // set namespace to uninitialized
    CParser *pParser = CParser::CreateParser(m_nUseFrontEnd);
    CParser::SetCurrentParser(pParser);
    if (!pParser->Parse(0 /* no exitsing scan buffer*/, m_sInFileName,
        m_nUseFrontEnd, IsOptionSet(PROGRAM_STOP_AFTER_PRE)))
    {
        if (!erroccured)
            Error("other parser error.");
    }
    // get file
    m_pRootFE = pParser->GetTopFileInScope();
    // now that parsing is over, get rid of it
    delete pParser;
    // if errors, print them and abort
    if (erroccured)
    {
        if (errcount > 0)
            Error("%d Error(s) and %d Warning(s) occured.", errcount, 
		warningcount);
        else
            Warning("%s: warning: %d Warning(s) occured while parsing.", 
		m_sInFileName.c_str(), warningcount);
    }
}

/**
 *  \brief prepares the write operation
 *
 * This method performs any tasks necessary before the write operation. E.g.
 * it creates the class and name factory depending on the compiler arguments,
 * it creates the context and finally creates the backend.
 *
 * First thing this function does is to force an consistency check for the
 * whole hierarchy of the IDL file. If this check fails the compile run is
 * aborted.
 *
 * Next the transmittable data is optimized.
 *
 * Finally the preparations for the write operation are made.
 */
void CCompiler::PrepareWrite()
{
    // if we should stop after preprocessing skip this function
    if (IsOptionSet(PROGRAM_STOP_AFTER_PRE))
        return;

    Verbose("Check consistency of parsed input ...\n");
    // consistency check
    if (!m_pRootFE)
        Error("Internal Error: Current file not set");
    if (!m_pRootFE->CheckConsistency())
        exit(1);
    Verbose("... finished consistency check.\n");

    // create context
    CBENameFactory *pNF;
    Verbose("Create Context...\n");

    // set platform class factory depending on arguments
    CBEClassFactory *pCF;
    if (IsBackEndSet(PROGRAM_BE_V2))
    {
	if (IsBackEndSet(PROGRAM_BE_AMD64))
	    pCF = new CL4V2AMD64BEClassFactory();
	else if (IsBackEndSet(PROGRAM_BE_IA32))
	    pCF = new CL4V2IA32BEClassFactory();
	else
	    pCF = new CL4V2BEClassFactory();
    }
    else if (IsBackEndSet(PROGRAM_BE_X0))
	pCF = new CL4X0BEClassFactory();
    else if (IsBackEndSet(PROGRAM_BE_V4))
    {
        if (IsBackEndSet(PROGRAM_BE_IA32))
            pCF = new CL4V4IA32ClassFactory();
        else
            pCF = new CL4V4BEClassFactory();
    }
    else if (IsBackEndSet(PROGRAM_BE_SOCKETS))
        pCF = new CSockBEClassFactory();
    else if (IsBackEndSet(PROGRAM_BE_CDR))
        pCF = new CCDRClassFactory();
    else
        pCF = new CBEClassFactory();
    SetClassFactory(pCF);

    // set name factory depending on arguments
    if ((IsBackEndSet(PROGRAM_BE_V2)) ||
        (IsBackEndSet(PROGRAM_BE_X0)))
    {
	if (IsBackEndSet(PROGRAM_BE_AMD64))
	    pNF = new CL4V2AMD64BENameFactory();
	else
	    pNF = new CL4BENameFactory();
    }
    else if (IsBackEndSet(PROGRAM_BE_V4))
        pNF = new CL4V4BENameFactory();
    else
        pNF = new CBENameFactory();
    SetNameFactory(pNF);

    // set sizes
    CBESizes *pSizes = pCF->GetNewSizes();
    if (m_nOpcodeSize > 0)
	pSizes->SetOpcodeSize(m_nOpcodeSize);
    SetSizes(pSizes);

    /** Prepare write for the back-end does some intialization which cannot be
     * performed during the write run.
     */
    Verbose("Create backend...\n");
    m_pRootBE = pCF->GetNewRoot();
    try
    {
	m_pRootBE->CreateBE(m_pRootFE);
    }
    catch (CBECreateException *e)
    {
        Verbose("Back-End creation failed\n");
        delete m_pRootBE;
        m_pRootBE = 0;
	e->Print();
	delete e;
        Error("Creation of the back-end failed. compilation aborted.\n");
    }
    Verbose("...done.\n");

    // print dependency tree
    Verbose("Print dependencies...\n");
    if (IsOptionSet(PROGRAM_DEPEND_MASK))
        PrintDependencies();
    Verbose("... dependencies done.\n");

    // if we should stop after printing the dependencies, stop here
    if (IsOptionSet(PROGRAM_DEPEND_M) || IsOptionSet(PROGRAM_DEPEND_MM))
    {
         delete m_pRootBE;
         delete pNF;
         delete pCF;
         return;
    }
}

/**
 *  \brief prints the target files
 *
 * This method calls the write operation of the backend
 */
void CCompiler::Write()
{
    // if we should stop after preprocessing skip this function
    if (IsOptionSet(PROGRAM_STOP_AFTER_PRE) ||
        IsOptionSet(PROGRAM_DEPEND_M) ||
        IsOptionSet(PROGRAM_DEPEND_MM))
        return;

    // We could load the tracing lib here. However, we want it to parse our
    // arguments. So we have to open it right at the beginning.

    // write backend
    Verbose("Write backend...\n");
    m_pRootBE->Write();
    Verbose("...done.\n");
}

/**
 *  \brief helper function
 *  \param sMsg the format string of the message
 *
 * This method prints a message if the ciompiler runs in verbose mode.
 */
void CCompiler::Verbose(const char *sMsg, ...)
{
    // if verbose turned off: return
    if (!m_bVerbose)
        return;
    va_list args;
    va_start(args, sMsg);
    vprintf(sMsg, args);
    va_end(args);
}

/**
 *  \brief helper function
 *  \param sMsg the error message to print before exiting
 *
 * This method prints an error message and exits the compiler. Any clean up
 * should be performed  BEFORE this method is called.
 */
void CCompiler::Error(const char *sMsg, ...)
{
    cerr << "dice: ";
    va_list args;
    va_start(args, sMsg);
    vfprintf(stderr, sMsg, args);
    va_end(args);
    cerr << "\n";

    fflush(stdout);
    fflush(stderr);
    sleep(1);
    exit(1);
}

/** \brief helper function
 *  \param pFEObject the front-end object where the error occurred
 *  \param nLinenb the linenumber of the error
 *  \param sMsg the error message
 *
 * The given front-end object can be used to determine the file this object
 * belonged to and the line, where this object has been declared. This is
 * useful if we do not have a current line number available (any time after
 * the parsing finished).
 */
void CCompiler::GccError(CFEBase * pFEObject, int nLinenb, const char *sMsg, ...)
{
    va_list args;
    va_start(args, sMsg);
    GccErrorVL(pFEObject, nLinenb, sMsg, args);
    va_end(args);
}

/** \brief helper function
 *  \param pFEObject the front-end object where the error occurred
 *  \param nLinenb the linenumber of the error
 *  \param sMsg the error message
 *  \param vl the argument list to be printed
 *
 * The given front-end object can be used to determine the file this object
 * belonged to and the line, where this object has been declared. This is
 * useful if we do not have a current line number available (any time after
 * the parsing finished).
 */
void CCompiler::GccErrorVL(CFEBase * pFEObject, int nLinenb, const char *sMsg, va_list vl)
{
    if (pFEObject)
    {
        // check line number
        if (nLinenb == 0)
            nLinenb = pFEObject->GetSourceLine();
        // iterate through include hierarchy
        CFEFile *pCur = (CFEFile*)pFEObject->GetSpecificParent<CFEFile>(0);
        vector<CFEFile*> *pStack = new vector<CFEFile*>();
        while (pCur)
        {
            pStack->insert(pStack->begin(), pCur);
            // get file of parent because GetFile start with "this"
            pCur = pCur->GetSpecificParent<CFEFile>(1);
        }
        // need at least one file
        if (!pStack->empty())
        {
            // walk down
            if (pStack->size() > 1)
                cerr << "In file included ";
            vector<CFEFile*>::iterator iter;
            for (iter = pStack->begin();
                (iter != pStack->end()) && (iter != pStack->end()-1); iter++)
            {
		// we do not use GetFullFileName, because the "normal"
		// filename already includes the whole path it is the filename
		// generated by Gcc
                CFEFile *pFEFile = *iter;
                int nIncludeLine = 1;
                if (iter != pStack->end()-1)
                    nIncludeLine = (*(iter+1))->GetIncludedOnLine();
                cerr << "from " << pFEFile->GetFileName() << ":" << nIncludeLine;
                if (iter + 2 != pStack->end())
                    cerr << ",\n                 ";
                else
                    cerr << ":\n";
            }
            if (*iter)
            {
		// we do not use GetFullFileName, because the "normal"
		// filename already includes the whole path it is the filename
		// generated by Gcc
                cerr << (*iter)->GetFileName() << ":" << nLinenb << ": ";
            }
        }
        // cleanup
        delete pStack;
    }
    vfprintf(stderr, sMsg, vl);
    cerr << endl;
}

/**
 *  \brief helper function
 *  \param sMsg the warning message to print
 *
 * This method prints an error message and returns.
 */
void CCompiler::Warning(const char *sMsg, ...)
{
    va_list args;
    va_start(args, sMsg);
    vfprintf(stderr, sMsg, args);
    va_end(args);
    cerr << endl;
}

/** \brief print verbose message
 *  \param level verboseness level of message
 *  \param format the format string
 */
void CCompiler::Verbose(unsigned long level, const char *format, ...)
{
    if (!CCompiler::IsVerbose())
	return;

    if (level > m_nVerboseLevel)
	return;
    
    if (m_nVerboseInd >= 0)
	fprintf(stdout, "[%02d] ", m_nVerboseInd);
    
    va_list args;
    va_start(args, format);
    vfprintf(stdout, format, args);
    va_end(args);
    fflush (stdout);
}

/** \brief print verbose message
 *  \param level verboseness level of message
 *  \param format the format string
 */
void CCompiler::VerboseI(unsigned long level, const char *format, ...)
{
    if (!CCompiler::IsVerbose())
	return;

    if (level > m_nVerboseLevel)
	return;
    
    m_nVerboseInd++;
    if (m_nVerboseInd >= 0)
	fprintf(stdout, "[%02d] ", m_nVerboseInd);
    
    va_list args;
    va_start(args, format);
    vfprintf(stdout, format, args);
    va_end(args);
    fflush (stdout);
}

/** \brief print verbose message
 *  \param level verboseness level of message
 *  \param format the format string
 */
void CCompiler::VerboseD(unsigned long level, const char *format, ...)
{
    if (!CCompiler::IsVerbose())
	return;
    
    if (level > m_nVerboseLevel)
	return;
    
    if (m_nVerboseInd >= 0)
	fprintf(stdout, "[%02d] ", m_nVerboseInd);
    if (m_nVerboseInd > 0)
	m_nVerboseInd--;
    
    va_list args;
    va_start(args, format);
    vfprintf(stdout, format, args);
    va_end(args);
    fflush (stdout);
}

/** \brief print warning messages
 *  \param pFEObject the object where the warning occured
 *  \param nLinenb the line in the file where the object originated
 *  \param sMsg the warning message
 */
void CCompiler::GccWarning(CFEBase * pFEObject, int nLinenb, const char *sMsg, ...)
{
    va_list args;
    va_start(args, sMsg);
    GccWarningVL(pFEObject, nLinenb, sMsg, args);
    va_end(args);
}

/** \brief print warning messages
 *  \param pFEObject the object where the warning occured
 *  \param nLinenb the line in the file where the object originated
 *  \param sMsg the warning message
 *  \param vl teh variable argument list
 */
void CCompiler::GccWarningVL(CFEBase * pFEObject, int nLinenb, const char *sMsg, va_list vl)
{
    erroccured++;
    warningcount++;
    if (pFEObject)
    {
        // check line number
        if (nLinenb == 0)
            nLinenb = pFEObject->GetSourceLine();
        // iterate through include hierarchy
        CFEFile *pCur = (CFEFile*)pFEObject->GetSpecificParent<CFEFile>();
        vector<CFEFile*> *pStack = new vector<CFEFile*>();
        while (pCur)
        {
            pStack->insert(pStack->begin(), pCur);
            // start with parent of current, because GetFile starts with "this"
            pCur = pCur->GetSpecificParent<CFEFile>(1);
        }
        // need at least one file
        if (!pStack->empty())
        {
            // walk down
            if (pStack->size() > 1)
                cerr << "In file included ";
            vector<CFEFile*>::iterator iter;
            for (iter = pStack->begin();
                (iter != pStack->end()) && (iter != pStack->end()-1); iter++)
            {
                cerr << "from " << (*iter)->GetFullFileName() << ":1";
                if (iter+2 != pStack->end())
                    cerr << ",\n                 ";
                else
                    cerr << ":\n";
            }
            if (*iter)
            {
                cerr << (*iter)->GetFullFileName() << ":" << nLinenb << ": warning: ";
            }
        }
        // cleanup
        delete pStack;
    }
    vfprintf(stderr, sMsg, vl);
    cerr << "\n";
}

/** \brief print the dependency tree
 *
 * The dependency tree contains the files the top IDL file depends on. These
 * are the included files.
 */
void CCompiler::PrintDependencies()
{
    if (!m_pRootBE)
        return;

    // if file, open file
    FILE *output = stdout;
    if (IsOptionSet(PROGRAM_DEPEND_MD) || 
	IsOptionSet(PROGRAM_DEPEND_MMD) ||
	IsOptionSet(PROGRAM_DEPEND_MF))
    {
	string sOutName = GetOutputDir();
	if (IsOptionSet(PROGRAM_DEPEND_MF))
	    sOutName += m_sDependsFile;
	else
	    sOutName += m_pRootFE->GetFileNameWithoutExtension() + ".d";
        output = fopen(sOutName.c_str(), "w");
        if (!output)
        {
            Warning("Could not open %s, use <stdout>\n", sOutName.c_str());
            output = stdout;
        }
	Verbose("%s: opened \"%s\" for output\n", __FUNCTION__,
	    sOutName.c_str());
    }

    Verbose("%s: start printing target files\n", __FUNCTION__);
    m_nCurCol = 0;
    m_pRootBE->PrintTargetFiles(output, m_nCurCol, MAX_SHELL_COLS);
    fprintf(output, ": ");
    m_nCurCol += 2;
    Verbose("%s: start with files\n", __FUNCTION__);
    PrintDependentFile(output, m_pRootFE->GetFullFileName());
    PrintDependencyTree(output, m_pRootFE);
    fprintf(output, "\n\n");

    // if phony dependencies are set then print them now
    if (IsOptionSet(PROGRAM_DEPEND_MP))
    {
	vector<string>::iterator iter = m_vPhonyDependencies.begin();
	for (; iter != m_vPhonyDependencies.end(); iter++)
	{
	    fprintf(output, "%s: \n\n", (*iter).c_str());
	}
    }

    // if file, close file
    if (IsOptionSet(PROGRAM_DEPEND_MD) || IsOptionSet(PROGRAM_DEPEND_MMD))
        fclose(output);
    Verbose("%s: finished\n", __FUNCTION__);
}

/** \brief prints the included files
 *  \param output the output stream
 *  \param pFEFile the current front-end file
 *
 * This implementation print the file names of the included files. It first
 * prints the names and then iterates over the included files.
 *
 * If the options PROGRAM_DEPEND_MM or PROGRAM_DEPEND_MMD are specified only
 * files included with '#include "file"' are printed.
 */
void CCompiler::PrintDependencyTree(FILE * output, CFEFile * pFEFile)
{
    if (!pFEFile)
        return;
    Verbose("%s: for \"%s\" called\n", __FUNCTION__, 
	pFEFile->GetFileName().c_str());
    // print names
    vector<CFEFile*>::iterator iterF = pFEFile->GetFirstChildFile();
    CFEFile *pIncFile;
    while ((pIncFile = pFEFile->GetNextChildFile(iterF)) != 0)
    {
	Verbose("%s: checking child file \"%s\" as directly included\n",
	    __FUNCTION__, pIncFile->GetFileName().c_str());
        if (pIncFile->IsStdIncludeFile() &&
            (IsOptionSet(PROGRAM_DEPEND_MM) || IsOptionSet(PROGRAM_DEPEND_MMD)))
            continue;
        PrintDependentFile(output, pIncFile->GetFullFileName());
    }
    // ierate over included files
    iterF = pFEFile->GetFirstChildFile();
    while ((pIncFile = pFEFile->GetNextChildFile(iterF)) != 0)
    {
	Verbose("%s: checking child file \"%s\" for trees\n", __FUNCTION__,
	    pIncFile->GetFileName().c_str());
        if (pIncFile->IsStdIncludeFile() &&
            (IsOptionSet(PROGRAM_DEPEND_MM) || IsOptionSet(PROGRAM_DEPEND_MMD)))
            continue;
        PrintDependencyTree(output, pIncFile);
    }
}

/** \brief prints a list of generated files for the given options
 *  \param output the output stream
 *  \param pFEFile the file to scan for generated files
 */
void CCompiler::PrintGeneratedFiles(FILE * output, CFEFile * pFEFile)
{
    if (!pFEFile->IsIDLFile())
        return;

    Verbose("%s: for \"%s\" called\n", __FUNCTION__, 
	pFEFile->GetFileName().c_str());

    if (IsOptionSet(PROGRAM_FILE_IDLFILE) || IsOptionSet(PROGRAM_FILE_ALL))
    {
        PrintGeneratedFiles4File(output, pFEFile);
    }
    else if (IsOptionSet(PROGRAM_FILE_MODULE))
    {
        PrintGeneratedFiles4Library(output, pFEFile);
    }
    else if (IsOptionSet(PROGRAM_FILE_INTERFACE))
    {
        PrintGeneratedFiles4Interface(output, pFEFile);
    }
    else if (IsOptionSet(PROGRAM_FILE_FUNCTION))
    {
        PrintGeneratedFiles4Operation(output, pFEFile);
    }

    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sName;
    // create client header file
    if (IsOptionSet(PROGRAM_GENERATE_CLIENT))
    {
        sName = pNF->GetFileName(pFEFile, FILETYPE_CLIENTHEADER);
        PrintDependentFile(output, sName);
    }
    // create server header file
    if (IsOptionSet(PROGRAM_GENERATE_COMPONENT))
    {
        // component file
        sName = pNF->GetFileName(pFEFile, FILETYPE_COMPONENTHEADER);
        PrintDependentFile(output, sName);
    }
    // opcodes
    if (!IsOptionSet(PROGRAM_NO_OPCODES))
    {
        sName = pNF->GetFileName(pFEFile, FILETYPE_OPCODE);
        PrintDependentFile(output, sName);
    }

    if (IsOptionSet(PROGRAM_FILE_ALL))
    {
        vector<CFEFile*>::iterator iter = pFEFile->GetFirstChildFile();
        CFEFile *pFile;
        while ((pFile = pFEFile->GetNextChildFile(iter)) != 0)
            PrintGeneratedFiles(output, pFile);
    }

    Verbose("%s: done\n", __FUNCTION__);
}

/** \brief prints the file-name generated for the front-end file
 *  \param output the target stream
 *  \param pFEFile the current front-end file
 */
void CCompiler::PrintGeneratedFiles4File(FILE * output, CFEFile * pFEFile)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sName;

    Verbose("%s: for %s called\n", __FUNCTION__, 
	pFEFile->GetFileName().c_str());

    // client
    if (IsOptionSet(PROGRAM_GENERATE_CLIENT))
    {
        // client implementation file
        sName = pNF->GetFileName(pFEFile, FILETYPE_CLIENTIMPLEMENTATION);
        PrintDependentFile(output, sName);
    }
    // server
    if (IsOptionSet(PROGRAM_GENERATE_COMPONENT))
    {
        // component file
        sName = pNF->GetFileName(pFEFile, FILETYPE_COMPONENTIMPLEMENTATION);
        PrintDependentFile(output, sName);
    }

    Verbose("%s: done.\n", __FUNCTION__);
}

/** \brief prints a list of generated files for the library granularity
 *  \param output the output stream
 *  \param pFEFile the front-end file to scan for generated files
 */
void CCompiler::PrintGeneratedFiles4Library(FILE * output, CFEFile * pFEFile)
{
    Verbose("%s: for file \"%s\" called\n", __FUNCTION__, 
	pFEFile->GetFileName().c_str());
    
    // iterate over libraries
    vector<CFELibrary*>::iterator iterL = pFEFile->GetFirstLibrary();
    CFELibrary *pLibrary;
    while ((pLibrary = pFEFile->GetNextLibrary(iterL)) != 0)
    {
        PrintGeneratedFiles4Library(output, pLibrary);
    }
    // iterate over interfaces
    vector<CFEInterface*>::iterator iterI = pFEFile->GetFirstInterface();
    CFEInterface *pInterface;
    while ((pInterface = pFEFile->GetNextInterface(iterI)) != 0)
    {
        PrintGeneratedFiles4Library(output, pInterface);
    }

    Verbose("%s: done\n", __FUNCTION__);
}

/** \brief print the list of generated files for the library granularity
 *  \param output the output stream to print to
 *  \param pFELibrary the library to scan
 */
void CCompiler::PrintGeneratedFiles4Library(FILE * output,
    CFELibrary * pFELibrary)
{
    Verbose("%s: for lib \"%s\" called\n", __FUNCTION__,
	pFELibrary->GetName().c_str());
    
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sName;
    // client file
    if (IsOptionSet(PROGRAM_GENERATE_CLIENT))
    {
        // implementation
        sName = pNF->GetFileName(pFELibrary, FILETYPE_CLIENTIMPLEMENTATION);
        PrintDependentFile(output, sName);
    }
    // component file
    if (IsOptionSet(PROGRAM_GENERATE_COMPONENT))
    {
        // implementation
        sName = pNF->GetFileName(pFELibrary, FILETYPE_COMPONENTIMPLEMENTATION);
        PrintDependentFile(output, sName);
    }
    // nested libraries
    vector<CFELibrary*>::iterator iterL = pFELibrary->GetFirstLibrary();
    CFELibrary *pLibrary;
    while ((pLibrary = pFELibrary->GetNextLibrary(iterL)) != 0)
    {
        PrintGeneratedFiles4Library(output, pLibrary);
    }

    Verbose("%s: done.\n", __FUNCTION__);
}

/** \brief print the list of generated files for the library granularity
 *  \param output the output stream to print to
 *  \param pFEInterface the interface to scan
 */
void CCompiler::PrintGeneratedFiles4Library(FILE * output,
    CFEInterface * pFEInterface)
{
    Verbose("%s: for interface \"%s\" called\n", __FUNCTION__,
	pFEInterface->GetName().c_str());
    
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sName;
    // client file
    if (IsOptionSet(PROGRAM_GENERATE_CLIENT))
    {
        // implementation
        sName = pNF->GetFileName(pFEInterface, FILETYPE_CLIENTIMPLEMENTATION);
        PrintDependentFile(output, sName);
    }
    // component file
    if (IsOptionSet(PROGRAM_GENERATE_COMPONENT))
    {
        // implementation
        sName = pNF->GetFileName(pFEInterface,
	    FILETYPE_COMPONENTIMPLEMENTATION);
        PrintDependentFile(output, sName);
    }

    Verbose("%s: done.\n", __FUNCTION__);
}

/** \brief print the list of generated files for interface granularity
 *  \param output the output stream to print to
 *  \param pFEFile the front-end file to be scanned
 */
void CCompiler::PrintGeneratedFiles4Interface(FILE * output, CFEFile * pFEFile)
{
    Verbose("%s: for file \"%s\" called\n", __FUNCTION__,
	pFEFile->GetFileName().c_str());

    // iterate over interfaces
    vector<CFEInterface*>::iterator iterI = pFEFile->GetFirstInterface();
    CFEInterface *pInterface;
    while ((pInterface = pFEFile->GetNextInterface(iterI)) != 0)
    {
	PrintGeneratedFiles4Interface(output, pInterface);
    }
    // iterate over libraries
    vector<CFELibrary*>::iterator iterL = pFEFile->GetFirstLibrary();
    CFELibrary *pLibrary;
    while ((pLibrary = pFEFile->GetNextLibrary(iterL)) != 0)
    {
	PrintGeneratedFiles4Interface(output, pLibrary);
    }

    Verbose("%s: done.\n", __FUNCTION__);
}

/** \brief print the list of generated files for interface granularity
 *  \param output the output stream to print to
 *  \param pFELibrary the front-end library to be scanned
 */
void CCompiler::PrintGeneratedFiles4Interface(FILE * output,
    CFELibrary * pFELibrary)
{
    Verbose("%s: for lib \"%s\" called\n", __FUNCTION__,
	pFELibrary->GetName().c_str());
    
    // iterate over interfaces
    vector<CFEInterface*>::iterator iterI = pFELibrary->GetFirstInterface();
    CFEInterface *pInterface;
    while ((pInterface = pFELibrary->GetNextInterface(iterI)) != 0)
    {
        PrintGeneratedFiles4Interface(output, pInterface);
    }
    // iterate over nested libraries
    vector<CFELibrary*>::iterator iterL = pFELibrary->GetFirstLibrary();
    CFELibrary *pLibrary;
    while ((pLibrary = pFELibrary->GetNextLibrary(iterL)) != 0)
    {
        PrintGeneratedFiles4Interface(output, pLibrary);
    }

    Verbose("%s: done.\n", __FUNCTION__);
}

/** \brief print the list of generated files for interface granularity
 *  \param output the output stream to print to
 *  \param pFEInterface the front-end interface to be scanned
 */
void CCompiler::PrintGeneratedFiles4Interface(FILE * output,
    CFEInterface * pFEInterface)
{
    Verbose("%s: for interface \"%s\" called\n", __FUNCTION__,
	pFEInterface->GetName().c_str());
    
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sName;
    // client file
    if (IsOptionSet(PROGRAM_GENERATE_CLIENT))
    {
        // implementation
        sName = pNF->GetFileName(pFEInterface, FILETYPE_CLIENTIMPLEMENTATION);
        PrintDependentFile(output, sName);
    }
    // component file
    if (IsOptionSet(PROGRAM_GENERATE_COMPONENT))
    {
        // implementation
        sName = pNF->GetFileName(pFEInterface,
	    FILETYPE_COMPONENTIMPLEMENTATION);
        PrintDependentFile(output, sName);
    }

    Verbose("%s: done.\n", __FUNCTION__);
}

/** \brief print the list of generated files for operation granularity
 *  \param output the output stream to print to
 *  \param pFEFile the front-end file to be scanned
 */
void CCompiler::PrintGeneratedFiles4Operation(FILE * output, CFEFile * pFEFile)
{
    Verbose("%s: for file \"%s\" called\n", __FUNCTION__, 
	pFEFile->GetFileName().c_str());
    
    // iterate over interfaces
    vector<CFEInterface*>::iterator iterI = pFEFile->GetFirstInterface();
    CFEInterface *pInterface;
    while ((pInterface = pFEFile->GetNextInterface(iterI)) != 0)
    {
        PrintGeneratedFiles4Operation(output, pInterface);
    }
    // iterate over libraries
    vector<CFELibrary*>::iterator iterL = pFEFile->GetFirstLibrary();
    CFELibrary *pLibrary;
    while ((pLibrary = pFEFile->GetNextLibrary(iterL)) != 0)
    {
        PrintGeneratedFiles4Operation(output, pLibrary);
    }

    Verbose("%s: done.\n", __FUNCTION__);
}

/** \brief print the list of generated files for operation granularity
 *  \param output the output stream to print to
 *  \param pFELibrary the front-end library to be scanned
 */
void CCompiler::PrintGeneratedFiles4Operation(FILE * output,
    CFELibrary * pFELibrary)
{
    Verbose("%s: for lib \"%s\" called\n", __FUNCTION__, 
	pFELibrary->GetName().c_str());
    
    // iterate over interfaces
    vector<CFEInterface*>::iterator iterI = pFELibrary->GetFirstInterface();
    CFEInterface *pInterface;
    while ((pInterface = pFELibrary->GetNextInterface(iterI)) != 0)
    {
        PrintGeneratedFiles4Operation(output, pInterface);
    }
    // iterate over nested libraries
    vector<CFELibrary*>::iterator iterL = pFELibrary->GetFirstLibrary();
    CFELibrary *pLibrary;
    while ((pLibrary = pFELibrary->GetNextLibrary(iterL)) != 0)
    {
        PrintGeneratedFiles4Operation(output, pLibrary);
    }

    Verbose("%s: done\n", __FUNCTION__);
}

/** \brief print the list of generated files for operation granularity
 *  \param output the output stream to print to
 *  \param pFEInterface the front-end interface to be scanned
 */
void CCompiler::PrintGeneratedFiles4Operation(FILE * output,
    CFEInterface * pFEInterface)
{
    Verbose("%s: for interface \"%s\" caled\n", __FUNCTION__,
	pFEInterface->GetName().c_str());
    
    // iterate over operations
    vector<CFEOperation*>::iterator iter = pFEInterface->GetFirstOperation();
    CFEOperation *pOperation;
    while ((pOperation = pFEInterface->GetNextOperation(iter)) != 0)
    {
        PrintGeneratedFiles4Operation(output, pOperation);
    }

    Verbose("%s: done.\n", __FUNCTION__);
}

/** \brief print the list of generated files for operation granularity
 *  \param output the output stream to print to
 *  \param pFEOperation the front-end operation to be scanned
 */
void CCompiler::PrintGeneratedFiles4Operation(FILE * output,
    CFEOperation * pFEOperation)
{
    Verbose("%s: for op \"%s\" called\n", __FUNCTION__, 
	pFEOperation->GetName().c_str());
    
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sName;
    // client file
    if (IsOptionSet(PROGRAM_GENERATE_CLIENT))
    {
	// implementation
	sName = pNF->GetFileName(pFEOperation, FILETYPE_CLIENTIMPLEMENTATION);
	PrintDependentFile(output, sName);
    }
    // component file
    if (IsOptionSet(PROGRAM_GENERATE_COMPONENT))
    {
	// implementation
	sName = pNF->GetFileName(pFEOperation,
	    FILETYPE_COMPONENTIMPLEMENTATION);
	PrintDependentFile(output, sName);
    }

    Verbose("%s: done.\n", __FUNCTION__);
}

/** \brief prints a filename to the dependency tree
 *  \param output the stream to write to
 *  \param sFileName the name to print
 *
 * This implementation adds a spacer after each file name and also checks
 * before writing if the maximum column number would be exceeded by this file.
 * If it would a new line is started The length of the filename is added to
 * the columns.
 */
void CCompiler::PrintDependentFile(FILE * output, string sFileName)
{
    Verbose("%s(%s) called\n", __FUNCTION__, sFileName.c_str());
    
    char *real_path;
    /* Cite: Avoid using this function. It is broken by design... */
    real_path = realpath(sFileName.c_str(), NULL /* glibc extension! */);
    if (!real_path)
    {
	Error("Calling realpath(%s) returned an error: %s\n",
	    sFileName.c_str(), strerror(errno));
    }
    
    int nLength = strlen(real_path);
    if ((m_nCurCol + nLength + 1) >= MAX_SHELL_COLS)
    {
	fprintf(output, "\\\n ");
	m_nCurCol = 1;
    }
    fprintf(output, "%s ", real_path);
    m_nCurCol += nLength + 1;

    // if phony dependency option is set, add this file to phony dependency
    // list
    if (IsOptionSet(PROGRAM_DEPEND_MP))
	m_vPhonyDependencies.push_back(string(real_path));

    Verbose("%s done\n", __FUNCTION__);
}

