INTERFACE [integrator]:

#include "types.h"

EXTENSION class Uart
{
public:

  bool startup(Address address, unsigned irq);

  enum {
    MODE_8N1 = 1,
  };

private:
  Address address;
  unsigned _irq;
  enum {
    UART011_RXIM = 1 << 4,
    UART011_TXIM = 1 << 5,
    UART011_RTIM = 1 << 6,
    UART011_FEIM = 1 << 7,
    UART011_PEIM = 1 << 8,
    UART011_BEIM = 1 << 9,
    UART011_OEIM = 1 << 10,

    UART011_RXIS = 1 << 4,
    UART011_RTIS = 1 << 6,

    UART01x_CR_UARTEN = 1, // UART enable
    UART011_CR_LBE    = 0x080, // loopback enable
    UART011_CR_TXE    = 0x100, // transmit enable
    UART011_CR_RXE    = 0x200, // receive enable

    UART01x_FR_BUSY   = 0x008,
    UART01x_FR_RXFE   = 0x010,
    UART01x_FR_TXFF   = 0x020,

    UART01x_LCRH_PEN    = 0x02, // parity enable
    UART01x_LCRH_WLEN_8 = 0x60,
  };

};

//---------------------------------------------------------------------------
IMPLEMENTATION [integrator]:

#include <cassert>
#include "processor.h"

PRIVATE INLINE static Address Uart::_UART01x_DR(Address a)
{ return a + 0x00; }

PRIVATE INLINE static Address Uart::_UART01x_FR(Address a)
{ return a + 0x18; }

PRIVATE INLINE static Address Uart::_UART011_IBRD(Address a)
{ return a + 0x24; }

PRIVATE INLINE static Address Uart::_UART011_FBRD(Address a)
{ return a + 0x28; }

PRIVATE INLINE static Address Uart::_UART011_LCRH(Address a)
{ return a + 0x2c; }

PRIVATE INLINE static Address Uart::_UART011_CR(Address a)
{ return a + 0x30; }

PRIVATE INLINE static Address Uart::_UART011_IMSC(Address a)
{ return a + 0x38; }

PRIVATE INLINE static Address Uart::_UART011_MIS(Address a)
{ return a + 0x40; }

PRIVATE INLINE static Address Uart::_UART011_ICR(Address a)
{ return a + 0x44; }


// getters
PRIVATE INLINE NEEDS[Uart::_UART01x_DR]
unsigned Uart::UART01x_DR()  const
{ return *((volatile unsigned*)(_UART01x_DR(address))); }

PRIVATE INLINE NEEDS[Uart::_UART01x_FR]
unsigned Uart::UART01x_FR()  const
{ return *((volatile unsigned*)(_UART01x_FR(address))); }

PRIVATE INLINE NEEDS[Uart::_UART011_CR]
unsigned Uart::UART011_CR()  const
{ return *((volatile unsigned*)(_UART011_CR(address))); }

PRIVATE INLINE NEEDS[Uart::_UART011_MIS]
unsigned Uart::UART011_MIS()  const
{ return *((volatile unsigned*)(_UART011_MIS(address))); }

PRIVATE INLINE NEEDS[Uart::_UART011_IMSC]
unsigned Uart::UART011_IMSC()  const
{ return *((volatile unsigned*)(_UART011_IMSC(address))); }


// setters
PRIVATE INLINE NEEDS[Uart::_UART01x_DR]
void Uart::UART01x_DR(unsigned v)
{ *((volatile unsigned*)(_UART01x_DR(address)))= v; }

PRIVATE INLINE NEEDS[Uart::_UART011_IBRD]
void Uart::UART011_IBRD(unsigned v)
{ *((volatile unsigned*)(_UART011_IBRD(address)))= v; }

PRIVATE INLINE NEEDS[Uart::_UART011_FBRD]
void Uart::UART011_FBRD(unsigned v)
{ *((volatile unsigned*)(_UART011_FBRD(address)))= v; }

PRIVATE INLINE NEEDS[Uart::_UART011_LCRH]
void Uart::UART011_LCRH(unsigned v)
{ *((volatile unsigned*)(_UART011_LCRH(address)))= v; }

PRIVATE INLINE NEEDS[Uart::_UART011_CR]
void Uart::UART011_CR(unsigned v)
{ *((volatile unsigned*)(_UART011_CR(address)))= v; }

PRIVATE INLINE NEEDS[Uart::_UART011_IMSC]
void Uart::UART011_IMSC(unsigned v)
{ *((volatile unsigned*)(_UART011_IMSC(address)))= v; }

PRIVATE INLINE NEEDS[Uart::_UART011_ICR]
void Uart::UART011_ICR(unsigned v)
{ *((volatile unsigned*)(_UART011_ICR(address)))= v; }


IMPLEMENT Uart::Uart()
{
  address = (unsigned)-1;
  _irq = (unsigned)-1;
}

IMPLEMENT Uart::~Uart()
{
  shutdown();
}


IMPLEMENT bool Uart::startup( Address _address, unsigned irq ) 
{
  unsigned cr = UART01x_CR_UARTEN | UART011_CR_TXE | UART011_CR_LBE;

  address =_address;
  _irq = irq;

  UART011_CR(cr);
  UART011_FBRD(0);
  UART011_IBRD(1);
  UART011_LCRH(0);
  UART01x_DR(1);

  while (UART01x_FR() & UART01x_FR_BUSY)
    asm volatile("" : : : "memory");

  return true;
}

IMPLEMENT void Uart::shutdown()
{
  UART011_IMSC(0); // all interrupts off
  UART011_ICR(0xffff);

  UART011_CR(UART01x_CR_UARTEN | UART011_CR_TXE);
}


IMPLEMENT bool Uart::change_mode(TransferMode /*m*/, BaudRate baud)
{
  /* Be lazy, only support one mode */
  if (baud != 115200)
    return false;

  unsigned old_cr = UART011_CR();
  UART011_CR(0);

  UART011_FBRD(0x0);
  UART011_IBRD(0x4);

  UART011_LCRH(UART01x_LCRH_WLEN_8 | UART01x_LCRH_PEN);
  UART011_CR(old_cr);

  return true;
}

PUBLIC bool Uart::tx_empty()
{
  return 1; //XXX
}


IMPLEMENT
int Uart::write( const char *s, unsigned count )
{
  unsigned old_im;
  Proc::Status st;

  st = Proc::cli_save();

  old_im = UART011_IMSC();
  UART011_IMSC(old_im | UART011_TXIM);

  /* transmission */
  for (unsigned i =0; i<count; i++) 
    {
      while (UART01x_FR() & UART01x_FR_TXFF)
	;
      UART01x_DR(s[i]);
      if (s[i] == '\n')
        {
	  while (UART01x_FR() & UART01x_FR_TXFF)
	    ;
	  UART01x_DR('\r');
        }
    }

  /* wait till everything is transmitted */
  while (UART01x_FR() & UART01x_FR_BUSY)
    ;

  UART011_IMSC(old_im & ~UART011_TXIM);

  Proc::sti_restore(st);
  return 1;
}

IMPLEMENT
int Uart::getchar( bool blocking )
{

  while (!char_avail())
    if (!blocking)
      return -1;

  return UART01x_DR();
}


IMPLEMENT
int Uart::char_avail() const
{
  return (UART011_MIS() & (UART011_RTIS | UART011_RXIS))
         && !(UART01x_FR() & UART01x_FR_RXFE);
}


IMPLEMENT inline
int const Uart::irq() const
{
  return _irq;
}

IMPLEMENT
void Uart::enable_rcv_irq()
{
  UART011_IMSC(UART011_IMSC() | (UART011_RXIM | UART011_RTIM
	                         | UART011_FEIM | UART011_PEIM | UART011_BEIM
                                 | UART011_OEIM));
}

IMPLEMENT
void Uart::disable_rcv_irq()
{
  UART011_IMSC(UART011_IMSC() & (UART011_RXIM | UART011_RTIM
	                         | UART011_FEIM | UART011_PEIM | UART011_BEIM
                                 | UART011_OEIM));
}

