/*
 * Ore network driver stub.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>

#include <asm/l4lxapi/irq.h>
#include <asm/l4lxapi/misc.h>
#include <asm/l4lxapi/thread.h>

#include <asm/generic/sched.h>
#include <asm/generic/setup.h>
#include <asm/generic/do_irq.h>

#include <l4/ore/ore.h>

MODULE_AUTHOR("Adam Lackorzynski <adam@os.inf.tu-dresden.de>");
MODULE_DESCRIPTION("Ore stub driver");
MODULE_LICENSE("GPL");

static char l4x_ore_devname[8] = "eth0";
static int  l4x_ore_irqnum = -1;

#define MAC_FMT    "%02X:%02X:%02X:%02X:%02X:%02X"
#define MAC_ARG(x) x->dev_addr[0], x->dev_addr[1], x->dev_addr[2], \
                   x->dev_addr[3], x->dev_addr[4], x->dev_addr[5]

struct l4x_ore_priv {
	struct net_device_stats    net_stats;

	int                        handle;
	l4ore_config               config;
	unsigned char              *pkt_buffer;
	unsigned long              pkt_size;

	l4_threadid_t              irq_thread;
	struct hw_interrupt_type   *previous_interrupt_type;
};

static struct net_device *l4x_ore_dev;

static int l4x_ore_xmit_frame(struct sk_buff *skb, struct net_device *netdev)
{
	struct l4x_ore_priv *priv = netdev_priv(netdev);

	if (skb->len < ETH_ZLEN) {
		skb = skb_padto(skb, ETH_ZLEN);
		if (skb == NULL)
			return 0;
		skb->len = ETH_ZLEN;
	}

	if (l4ore_send(priv->handle, (char *)skb->data, skb->len) < 0) {
		LOG_printf("%s: send failed\n", netdev->name);
		return 1; /* Hmm, which return type to take? */
	}

	dev_kfree_skb(skb);

	netdev->trans_start = jiffies;
	priv->net_stats.tx_packets++;
	priv->net_stats.tx_bytes += skb->len;

	return 0;
}

static struct net_device_stats *l4x_ore_get_stats(struct net_device *netdev)
{
	struct l4x_ore_priv *priv = netdev_priv(netdev);
	return &priv->net_stats;
}

static void l4x_ore_tx_timeout(struct net_device *netdev)
{
	LOG_printf("%s\n", __func__);
}

/*
 * Interrupt.
 */
static irqreturn_t l4x_ore_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *netdev = dev_id;
	struct l4x_ore_priv *priv = netdev_priv(netdev);
	struct sk_buff *skb;

	skb = dev_alloc_skb(priv->pkt_size);
	if (likely(skb)) {
		skb->dev = netdev;
		memcpy(skb_put(skb, priv->pkt_size),
		       priv->pkt_buffer, priv->pkt_size);

		skb->protocol = eth_type_trans(skb, netdev);
		netif_rx(skb);

		netdev->last_rx = jiffies;
		priv->net_stats.rx_bytes += skb->len;
		priv->net_stats.rx_packets++;

	} else {
		printk(KERN_WARNING "%s: dropping packet.\n", netdev->name);
		priv->net_stats.rx_dropped++;
	}

	return IRQ_HANDLED;
}

/*
 * Receive thread to get packets
 */
static void l4x_ore_irq_thread(void *data)
{
	struct net_device *netdev = *(struct net_device **)data;
	struct l4x_ore_priv *priv = netdev_priv(netdev);
	int ret;
	struct thread_info *ctx = current_thread_info();

	l4x_setup_ti_thread_stack_irq(ctx);

	while (1) {
		unsigned int size = ETH_FRAME_LEN;
		ret = l4ore_recv_blocking(priv->handle,
		                          (char **)&priv->pkt_buffer, &size,
		                          L4_IPC_NEVER);
		if (unlikely(ret < 0)) {
			LOG_printf("%s: l4ore_recv_blocking failed: %d\n",
			           netdev->name, ret);
			l4_sleep(100);
			continue;
		} else if (unlikely(ret > 0)) {
			LOG_printf("%s: buffer too small (%d)\n", netdev->name, ret);
		}

		priv->pkt_size = size;

		l4x_do_IRQ(netdev->irq, ctx);
	}
}

/* ----- */
static unsigned int l4x_ore_irq_startup(unsigned int irq)
{
	LOG_printf("%s\n", __func__);
	return 1;
}

static void l4x_ore_irq_dummy_void(unsigned int irq)
{
}

struct hw_interrupt_type l4x_ore_irq_type = {
	.typename	= "L4Ore IRQ",
	.startup	= l4x_ore_irq_startup,
	.shutdown	= l4x_ore_irq_dummy_void,
	.enable		= l4x_ore_irq_dummy_void,
	.disable	= l4x_ore_irq_dummy_void,
	.ack		= l4x_ore_irq_dummy_void,
	.end		= l4x_ore_irq_dummy_void,
};
/* ----- */

static int l4x_ore_open(struct net_device *netdev)
{
	struct l4x_ore_priv *priv = netdev_priv(netdev);
	int err = -ENODEV;

	netif_carrier_off(netdev);

	if (!priv->config.rw_active) {
		priv->config.rw_active = 1;
		l4ore_set_config(priv->handle, &priv->config);
	}

	printk("%s: Overwriting IRQ type for IRQ %d with l4ore type!\n",
	       netdev->name, netdev->irq);

	priv->previous_interrupt_type = irq_desc[netdev->irq].handler;

	if (netdev->irq < NR_IRQS)
		irq_desc[netdev->irq].handler = &l4x_ore_irq_type;
	else {
		printk("%s: irq(%d) > NR_IRQS(%d), failing\n",
		       netdev->name, netdev->irq, NR_IRQS);
		goto err_out_close;
	}

	priv->pkt_buffer = kmalloc(ETH_FRAME_LEN, GFP_KERNEL);
	if (!priv->pkt_buffer) {
		printk("%s: kmalloc error\n", netdev->name);
		goto err_out_close;
	}

	if ((err = request_irq(netdev->irq, l4x_ore_interrupt,
	                       SA_SAMPLE_RANDOM, netdev->name,
	                       netdev))) {
		printk("%s: request_irq(%d, ...) failed.\n",
		       netdev->name, netdev->irq);
		goto err_out_kfree;
	}

	priv->irq_thread = l4lx_thread_create(l4x_ore_irq_thread,
	                                      NULL, &netdev, sizeof(netdev),
	                                      -1, "L4OreRcv");
	if (l4_is_invalid_id(priv->irq_thread)) {
		printk("%s: Cannot create thread\n", netdev->name);
		err = -EBUSY;
		goto err_out_free_irq;
	}

	netif_carrier_on(netdev);
	netif_wake_queue(netdev);

	printk("%s: interface up.\n", netdev->name);

	return 0;

err_out_free_irq:
	free_irq(netdev->irq, netdev);

err_out_kfree:
	kfree(priv->pkt_buffer);

err_out_close:
	irq_desc[netdev->irq].handler = priv->previous_interrupt_type;
	l4ore_close(priv->handle);
	priv->config.rw_active = 0;
	priv->handle           = 0;
	return err;
}

static int l4x_ore_close(struct net_device *netdev)
{
	struct l4x_ore_priv *priv = netdev_priv(netdev);

	l4lx_thread_shutdown(priv->irq_thread);
	priv->irq_thread = L4_INVALID_ID;

	free_irq(netdev->irq, netdev);
	irq_desc[netdev->irq].handler = priv->previous_interrupt_type;
	netif_stop_queue(netdev);
	netif_carrier_off(netdev);

	kfree(priv->pkt_buffer);

	priv->config.rw_active = 0;

	return 0;
}

static int __init l4x_ore_init(void)
{
	struct l4x_ore_priv *priv;
	struct net_device *dev;
	int err = -ENODEV;

	if (!(dev = alloc_etherdev(sizeof(struct l4x_ore_priv))))
		return -ENOMEM;

	l4x_ore_dev          = dev;
	dev->open            = l4x_ore_open;
	dev->stop            = l4x_ore_close;
	dev->hard_start_xmit = l4x_ore_xmit_frame;
	dev->get_stats       = l4x_ore_get_stats;
	dev->tx_timeout      = l4x_ore_tx_timeout;

	priv = netdev_priv(dev);

	priv->config = L4ORE_DEFAULT_CONFIG;
	priv->config.rw_debug           = 0;
	priv->config.rw_broadcast       = 1;
	priv->config.ro_keep_device_mac = 1;
	priv->config.rw_active          = 0;

	/* Hmm, we need to open the connection here to get the MAC :/ */
	if ((priv->handle = l4ore_open(l4x_ore_devname, dev->dev_addr,
	                               &priv->config)) < 0) {
		printk("%s: l4ore_open failed: %d\n",
		       dev->name, priv->handle);
		goto err_out_free_dev;
	}

	if (l4x_ore_irqnum != -1)
		dev->irq = l4x_ore_irqnum;
	else
		dev->irq = priv->config.ro_irq;
	dev->mtu = priv->config.ro_mtu;

	if ((err = register_netdev(dev))) {
		printk("l4ore: Cannot register net device, aborting.\n");
		goto err_out_free_dev;
	}

	printk(KERN_INFO "%s: L4Ore card found with " MAC_FMT ", IRQ %d\n",
	                 dev->name, MAC_ARG(dev), dev->irq);

	return 0;

err_out_free_dev:
	free_netdev(dev);

	return err;
}

static void __exit l4x_ore_exit(void)
{
	l4ore_close(((struct l4x_ore_priv *)l4x_ore_dev->priv)->handle);
	unregister_netdev(l4x_ore_dev);
	free_netdev(l4x_ore_dev);
}

module_init(l4x_ore_init);
module_exit(l4x_ore_exit);

module_param_string(oredev, l4x_ore_devname, sizeof(l4x_ore_devname), 0);
MODULE_PARM_DESC(oredev, "Device name to request at ORe server");
module_param_named(irqnum, l4x_ore_irqnum, int, 0);
MODULE_PARM_DESC(irqnum, "Virtual IRQ number override");
