/**
 *    \file    dice/src/be/l4/x0adapt/L4X0aBECallFunction.cpp
 *    \brief   contains the implementation of the class CL4X0aBECallFunction
 *
 *    \date    01/07/2004
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/* Copyright (C) 2001-2004 by
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/l4/x0adapt/L4X0aBECallFunction.h"
#include "be/l4/L4BENameFactory.h"
#include "be/l4/L4BEMsgBufferType.h"
#include "be/l4/L4BEIPC.h"
#include "be/BEDeclarator.h"
#include "be/BEFile.h"
#include "be/BEContext.h"
#include "be/BEMarshaller.h"
#include "be/BEType.h"
#include "be/BETrace.h"
#include "TypeSpec-Type.h"
#include "Compiler.h"

CL4X0aBECallFunction::CL4X0aBECallFunction()
 : CL4BECallFunction()
{
}

/** destroys the object */
CL4X0aBECallFunction::~CL4X0aBECallFunction()
{
}

/** \brief Writes the variable declaration
 *  \param pFile the file to write to

 *
 * If we have a short IPC in both direction, we only need the result dope,
 * and two dummy dwords.
 */
void CL4X0aBECallFunction::WriteVariableDeclaration(CBEFile * pFile)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    bool bShortIPC = pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0);
    
    assert(m_pTrace);
    m_pTrace->VariableDeclaration(pFile, this);

    // check if we use assembler
    bool bAssembler = m_pComm->CheckProperty(this, COMM_PROP_USE_ASM);
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sDummy = pNF->GetDummyVariable();
    string sMWord = pNF->GetTypeName(TYPE_MWORD, false);
    if (bAssembler && bShortIPC)
    {
        // write dummys
        string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
        // to increase the confusing code:
        // if we have PROGRAM_USE_SYMBOLS set, we test for __PIC__ and PROFILE
        // then we write the three parts bPIC, bPROF, bNPROF if they are set
        bool bPIC = true;
        bool bPROF = true;
        bool bNPROF = true;
        bool bSymbols = CCompiler::IsOptionSet(PROGRAM_USE_SYMBOLS);
        if (bSymbols)
        {
            bPIC = CCompiler::HasSymbol("__PIC__");
            bPROF = CCompiler::HasSymbol("PROFILE") && !bPIC;
            bNPROF = !bPROF && !bPIC;
        }

        // test if we need dummies
        if (!bSymbols)
            pFile->Print("#if defined(__PIC__)\n");
        if (bPIC)
        {
            WriteReturnVariableDeclaration(pFile);
            // write result variable
            pFile->PrintIndent("l4_msgdope_t %s = { msgdope: 0 };\n", sResult.c_str());
            pFile->PrintIndent("%s %s __attribute__((unused));\n",
                                sMWord.c_str(), sDummy.c_str());
        }
        if (!bSymbols)
        {
            pFile->Print("#else // !PIC\n");
            pFile->Print("#if !defined(PROFILE)\n");
        }
        if (bNPROF)
        {
            WriteReturnVariableDeclaration(pFile);
            // write result variable
            pFile->PrintIndent("l4_msgdope_t %s = { msgdope: 0 };\n", sResult.c_str());
            pFile->PrintIndent("%s %s = 0;\n", sMWord.c_str(), sDummy.c_str());
        }
        if (!bSymbols)
        {
            pFile->Print("#endif // PROFILE\n");
            pFile->Print("#endif // !PIC\n");
        }
        // if we have in either direction some bit-stuffing, we need more dummies
        // declare local exception variable
        WriteExceptionWordDeclaration(pFile, false /* do not init variable*/);
        // finished with declaration
    }
    else
    {
        CL4BECallFunction::WriteVariableDeclaration(pFile);
        // we need the dummy for the assembler statements
        if (bAssembler)
            pFile->PrintIndent("%s %s;\n", sMWord.c_str(), sDummy.c_str());
    }
}

/** \brief writes the variable initialization
 *  \param pFile the file to write to

 *
 * If we use the asm short IPC code we do not init any variables (Its done during declaration).
 */
void CL4X0aBECallFunction::WriteVariableInitialization(CBEFile * pFile)
{
    // check if we use assembler
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (!(m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
        pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0)))
        CL4BECallFunction::WriteVariableInitialization(pFile);
}

/* \brief write the marshalling code for the short IPC
 * \param pFile the file to write to
 * \param nStartOffset the offset where to start marshalling
 * \param bUseConstOffset true if nStartOffset can be used
 *
 * If we have a short IPC into both direction, we skip the marshalling.
 */
void CL4X0aBECallFunction::WriteMarshalling(CBEFile * pFile,  int nStartOffset,
    bool & bUseConstOffset)
{
    assert(m_pTrace);
    bool bLocalTrace = false;
    if (!m_bTraceOn)
    {
	m_pTrace->AfterMarshalling(pFile, this);
	m_bTraceOn = bLocalTrace = true;
    }
    
    // check if we use assembler
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (!(m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
        pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0)))
    {
        CL4BECallFunction::WriteMarshalling(pFile, nStartOffset,
	    bUseConstOffset);
    }

    if (bLocalTrace)
    {
	m_pTrace->AfterMarshalling(pFile, this);
	m_bTraceOn = false;
    }
}

/* \brief write the unmarshalling code for the short IPC
 * \param pFile the file to write to
 * \param nStartOffset the offset where to start marshalling
 * \param bUseConstOffset true if nStartOffset can be used
 *
 * If we have a short IPC in both direction, we can skip the unmarshalling.
 * In case we use the assembler code and its a short IPC, all the returned
 * parameters land in the variables directly, but we still have to extract
 * the exception from its variable.
 */
void CL4X0aBECallFunction::WriteUnmarshalling(CBEFile* pFile, int nStartOffset,
    bool & bUseConstOffset)
{
    assert(m_pTrace);
    bool bLocalTrace = false;
    if (!m_bTraceOn)
    {
	m_pTrace->BeforeUnmarshalling(pFile, this);
	m_bTraceOn = bLocalTrace = true;
    }
    
    // check if we use assembler
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
        pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0))
    {
        WriteEnvExceptionFromWord(pFile);
        return;
    }

    CL4BECallFunction::WriteUnmarshalling(pFile, nStartOffset, bUseConstOffset);

    if (bLocalTrace)
    {
	m_pTrace->AfterMarshalling(pFile, this);
	m_bTraceOn = false;
    }
}

/** \brief writes the invocation code for the short IPC
 *  \param pFile the file to write to

 *
 * If we have a short IPC in both direction, we write assembler code
 * directly.
 */
void CL4X0aBECallFunction::WriteInvocation(CBEFile * pFile)
{
    // check if we use assembler
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
        pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0))
    {
        if (!CCompiler::IsOptionSet(PROGRAM_NO_SEND_CANCELED_CHECK))
        {
            // sometimes it's possible to abort a call of a client.
            // but the client wants his call made, so we try until
            // the call completes
            pFile->PrintIndent("do\n");
            pFile->PrintIndent("{\n");
            pFile->IncIndent();
        }
        // skip send dope init
        WriteIPC(pFile);
        if (!CCompiler::IsOptionSet(PROGRAM_NO_SEND_CANCELED_CHECK))
        {
            // now check if call has been canceled
            string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
            pFile->DecIndent();
            pFile->PrintIndent("} while ((L4_IPC_ERROR(%s) == L4_IPC_SEABORTED) || (L4_IPC_ERROR(%s) == L4_IPC_SECANCELED));\n",
                                sResult.c_str(), sResult.c_str());
        }
        WriteIPCErrorCheck(pFile);
    }
    else
        CL4BECallFunction::WriteInvocation(pFile);
}

