/**
 *    \file    dice/src/be/l4/x0/L4X0BEIPC.cpp
 *    \brief   contains the declaration of the class CL4X0BEIPC
 *
 *    \date    08/13/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */
#include "be/l4/x0/L4X0BEIPC.h"
#include "be/l4/L4BENameFactory.h"
#include "be/l4/L4BEMsgBufferType.h"
#include "be/BEContext.h"
#include "be/BEFile.h"
#include "be/BEFunction.h"
#include "be/BEDeclarator.h"
#include "be/BESizes.h"
#include "Compiler.h"
#include "TypeSpec-Type.h"

CL4X0BEIPC::CL4X0BEIPC()
 : CL4BEIPC()
{
}

/** destroys the IPC object */
CL4X0BEIPC::~CL4X0BEIPC()
{
}

/** \brief writes the IPC call
 *  \param pFile the file to write to
 *  \param pFunction the function to write for

 */
void CL4X0BEIPC::WriteCall(CBEFile* pFile,
    CBEFunction* pFunction,
    bool bSendFlexpage, 
    bool bSendShortIPC)
{
    string sServerID = CCompiler::GetNameFactory()->GetComponentIDVariable();
    string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
    string sTimeout = CCompiler::GetNameFactory()->GetTimeoutClientVariable();
    string sMWord = CCompiler::GetNameFactory()->GetTypeName(TYPE_MWORD, true);
    string sMsgBuffer = CCompiler::GetNameFactory()->GetMessageBufferVariable();
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    int nSendDir = pFunction->GetSendDirection();
    int nRecvDir = pFunction->GetReceiveDirection();
    bool bShortRecv = pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, nRecvDir);

    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    pFile->PrintIndent("l4_ipc_call_w3(*%s,\n", sServerID.c_str());
    pFile->IncIndent();
    pFile->PrintIndent("");
    if (bSendShortIPC)
        pFile->Print("L4_IPC_SHORT_MSG");
    else
    {
        if (bSendFlexpage)
            pFile->Print("(%s*)((%s)", sMWord.c_str(), sMWord.c_str());

        if (pMsgBuffer->HasReference())
            pFile->Print("%s", sMsgBuffer.c_str());
        else
            pFile->Print("&%s", sMsgBuffer.c_str());

        if (bSendFlexpage)
            pFile->Print("|2)");
    }
    pFile->Print(",\n");

    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    #warning for short IPC print parameters
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nSendDir);
    pFile->Print("[0]))),\n");
    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nSendDir);
    pFile->Print("[%d]))),\n", nWordSize);
    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nSendDir);
    pFile->Print("[%d]))),\n", nWordSize*2);

    if (bShortRecv)
        pFile->PrintIndent("L4_IPC_SHORT_MSG,\n");
    else
    {
        if (pMsgBuffer->HasReference())
            pFile->PrintIndent("%s,\n", sMsgBuffer.c_str());
        else
            pFile->PrintIndent("&%s,\n", sMsgBuffer.c_str());
    }

    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nRecvDir);
    pFile->Print("[0])),\n");
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nRecvDir);
    pFile->Print("[%d])),\n", nWordSize);
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nRecvDir);
    pFile->Print("[%d])),\n", nWordSize*2);

    pFile->PrintIndent("%s, &%s);\n", sTimeout.c_str(), sResult.c_str());

    pFile->DecIndent();
}

/** \brief writes the IPC receive
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 *  \param bAllowShortIPC true if short IPC is allowed

 */
void CL4X0BEIPC::WriteReceive(CBEFile* pFile,  CBEFunction* pFunction)
{
    string sServerID = CCompiler::GetNameFactory()->GetComponentIDVariable();
    string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
    string sTimeout;
    if (pFunction->IsComponentSide())
        sTimeout = CCompiler::GetNameFactory()->GetTimeoutServerVariable();
    else
        sTimeout = CCompiler::GetNameFactory()->GetTimeoutClientVariable();
    string sMsgBuffer = CCompiler::GetNameFactory()->GetMessageBufferVariable();
    string sMWord = CCompiler::GetNameFactory()->GetTypeName(TYPE_MWORD, true);
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);

    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    pFile->PrintIndent("l4_ipc_receive_w3(*%s,\n", sServerID.c_str());
    pFile->IncIndent();

    if (pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, DIRECTION_OUT))
        pFile->PrintIndent("L4_IPC_SHORT_MSG,\n ");
    else
    {
        if (pMsgBuffer->HasReference())
            pFile->PrintIndent("%s,\n", sMsgBuffer.c_str());
        else
            pFile->PrintIndent("&%s,\n", sMsgBuffer.c_str());
    }

    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    #warning for short IPC print parameters
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_OUT);
    pFile->Print("[0])),\n");
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_OUT);
    pFile->Print("[%d])),\n", nWordSize);
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_OUT);
    pFile->Print("[%d])),\n", nWordSize);

    pFile->PrintIndent("%s, &%s);\n", sTimeout.c_str(), sResult.c_str());

    pFile->DecIndent();
}

/** \brief writes the IPC reply-and-wait
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 *  \param bSendFlexpage true if a flexpage is sent
 *  \param bSendShortIPC true if a short IPC is sent

 */
void CL4X0BEIPC::WriteReplyAndWait(CBEFile* pFile,
    CBEFunction* pFunction,
    bool bSendFlexpage,
    bool bSendShortIPC)
{
    string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
    string sTimeout;
    if (pFunction->IsComponentSide())
        sTimeout = CCompiler::GetNameFactory()->GetTimeoutServerVariable();
    else
        sTimeout = CCompiler::GetNameFactory()->GetTimeoutClientVariable();
    string sServerID = CCompiler::GetNameFactory()->GetComponentIDVariable();
    string sMsgBuffer = CCompiler::GetNameFactory()->GetMessageBufferVariable();
    string sMWord = CCompiler::GetNameFactory()->GetTypeName(TYPE_MWORD, true);
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);

    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    pFile->PrintIndent("l4_ipc_reply_and_wait_w3(*%s,\n", sServerID.c_str());
    pFile->IncIndent();
    pFile->PrintIndent("");
    if (bSendFlexpage)
        pFile->Print("(%s*)((%s)", sMWord.c_str(), sMWord.c_str());
    if (bSendShortIPC)
        pFile->Print("L4_IPC_SHORT_MSG");
    else
        pFile->Print("%s", sMsgBuffer.c_str());
    if (bSendFlexpage)
        pFile->Print("|2)");
    pFile->Print(",\n");

    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    pFile->Print("[0]))),\n");
    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    pFile->Print("[%d]))),\n", nWordSize);
    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    pFile->Print("[%d]))),\n", nWordSize*2);

    pFile->PrintIndent("%s,\n", sServerID.c_str());

    pFile->PrintIndent("%s,\n", sMsgBuffer.c_str());
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_OUT);
    pFile->Print("[0])),\n");
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_OUT);
    pFile->Print("[%d])),\n", nWordSize);
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_OUT);
    pFile->Print("[%d])),\n", nWordSize*2);

    pFile->PrintIndent("%s, &%s);\n", sTimeout.c_str(), sResult.c_str());

    pFile->DecIndent();
}

/** \brief writes the IPC send
 *  \param pFile the file to write to
 *  \param pFunction the function to write for

 */
void CL4X0BEIPC::WriteSend(CBEFile* pFile,  CBEFunction* pFunction)
{
    int nDirection = pFunction->GetSendDirection();
    string sServerID = CCompiler::GetNameFactory()->GetComponentIDVariable();
    string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
    string sTimeout;
    if (pFunction->IsComponentSide())
        sTimeout = CCompiler::GetNameFactory()->GetTimeoutServerVariable();
    else
        sTimeout = CCompiler::GetNameFactory()->GetTimeoutClientVariable();
    string sMsgBuffer = CCompiler::GetNameFactory()->GetMessageBufferVariable();
    string sMWord = CCompiler::GetNameFactory()->GetTypeName(TYPE_MWORD, true);
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);

    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    pFile->PrintIndent("l4_ipc_send_w3(*%s,\n", sServerID.c_str());
    pFile->IncIndent();
    pFile->PrintIndent("");
    bool bVarBuffer = pMsgBuffer->IsVariableSized(nDirection) ||
                     (pMsgBuffer->GetAlias()->GetStars() > 0);
    bool bShortIPC = pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, nDirection);
    if (!bShortIPC && (pMsgBuffer->GetCount(TYPE_FLEXPAGE, nDirection) > 0))
        pFile->Print("(%s*)((%s)", sMWord.c_str(), sMWord.c_str());
    if (bShortIPC)
        pFile->Print("L4_IPC_SHORT_MSG");
    else
    {
        if (bVarBuffer)
            pFile->Print("%s", sMsgBuffer.c_str());
        else
            pFile->Print("&%s", sMsgBuffer.c_str());
    }
    if (!bShortIPC && (pMsgBuffer->GetCount(TYPE_FLEXPAGE, nDirection) > 0))
        pFile->Print(")|2)");
    pFile->Print(",\n");

    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    #warning for short IPC print parameter
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[0]))),\n");
    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[%d]))),\n", nWordSize);
    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[%d]))),\n", nWordSize*2);

    pFile->PrintIndent("%s, &%s);\n", sTimeout.c_str(), sResult.c_str());

    pFile->DecIndent();
}

/** \brief writes the IPC wait
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 *  \param bAllowShortIPC true if a short IPC is received

 */
void CL4X0BEIPC::WriteWait(CBEFile* pFile,  CBEFunction* pFunction)
{
    string sServerID = CCompiler::GetNameFactory()->GetComponentIDVariable();
    string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
    string sTimeout;
    if (pFunction->IsComponentSide())
        sTimeout = CCompiler::GetNameFactory()->GetTimeoutServerVariable();
    else
        sTimeout = CCompiler::GetNameFactory()->GetTimeoutClientVariable();
    string sMsgBuffer = CCompiler::GetNameFactory()->GetMessageBufferVariable();
    string sMWord = CCompiler::GetNameFactory()->GetTypeName(TYPE_MWORD, true);
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    int nDirection = pFunction->GetReceiveDirection();
    bool bVarBuffer = pMsgBuffer->IsVariableSized(nDirection) ||
                     (pMsgBuffer->GetAlias()->GetStars() > 0);
    bool bShortIPC = pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, nDirection);

    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    pFile->PrintIndent("l4_ipc_wait_w3(%s,\n", sServerID.c_str());
    pFile->IncIndent();
    if (bShortIPC)
        pFile->PrintIndent("L4_IPC_SHORT_MSG,\n");
    else
    {
        if (bVarBuffer)
            pFile->PrintIndent("%s,\n", sMsgBuffer.c_str());
        else
            pFile->PrintIndent("&%s,\n", sMsgBuffer.c_str());
    }
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    #warning for short IPC print parameter
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[0])),\n");
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[%d])),\n", nWordSize);
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[%d])),\n", nWordSize*2);
    pFile->PrintIndent("%s, &%s);\n", sTimeout.c_str(), sResult.c_str());
    pFile->DecIndent();
}
