/**
 *    \file    dice/src/be/l4/x0/L4X0BECallFunction.cpp
 *    \brief   contains the implementation of the class CL4X0BECallFunction
 *
 *    \date    01/07/2004
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/* Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/l4/x0/L4X0BECallFunction.h"
#include "be/l4/L4BENameFactory.h"
#include "be/l4/L4BEIPC.h"
#include "be/l4/L4BEMsgBufferType.h"
#include "be/BEFile.h"
#include "be/BEContext.h"
#include "be/BEDeclarator.h"
#include "be/BEMarshaller.h"
#include "be/BETrace.h"

#include "TypeSpec-Type.h"
#include "Attribute-Type.h"
#include "Compiler.h"

CL4X0BECallFunction::CL4X0BECallFunction()
 : CL4BECallFunction()
{
}

/** destroys the object */
CL4X0BECallFunction::~CL4X0BECallFunction()
{
}

/** \brief Writes the variable declaration
 *  \param pFile the file to write to

 *
 * If we have a short IPC in both direction, we only need the result dope,
 * and two dummy dwords.
 */
void CL4X0BECallFunction::WriteVariableDeclaration(CBEFile * pFile)
{
    CL4BECallFunction::WriteVariableDeclaration(pFile); // msgdope, return var, msgbuffer
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
        pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0))
    {
        // write dummys
        CBENameFactory *pNF = CCompiler::GetNameFactory();
        string sDummy = pNF->GetDummyVariable();
        string sMWord = pNF->GetTypeName(TYPE_MWORD, false);
        // write dummy variable
        *pFile << "\t" << sMWord << " " << sDummy <<
            " __attribute__((unused));\n";
    }
}

/* \brief write the marshalling code for the short IPC
 * \param pFile the file to write to
 * \param nStartOffset the offset where to start marshalling
 * \param bUseConstOffset true if nStartOffset can be used
 *
 * If we have a short IPC into both direction, we skip the marshalling.
 */
void CL4X0BECallFunction::WriteMarshalling(CBEFile * pFile,  int nStartOffset,
    bool & bUseConstOffset)
{
    assert(m_pTrace);
    bool bLocalTrace = false;
    if (!m_bTraceOn)
    {
	m_pTrace->BeforeMarshalling(pFile, this);
	m_bTraceOn = bLocalTrace = true;
    }
    
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (!(m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
        pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0)))
        CL4BECallFunction::WriteMarshalling(pFile, nStartOffset,
	    bUseConstOffset);

    if (bLocalTrace)
    {
	m_pTrace->AfterMarshalling(pFile, this);
	m_bTraceOn = false;
    }
}

/* \brief write the unmarshalling code for the short IPC
 * \param pFile the file to write to
 * \param nStartOffset the offset where to start marshalling
 * \param bUseConstOffset true if nStartOffset can be used
 *
 * If we have a short IPC in both direction, we can skip the unmarshalling.
 */
void CL4X0BECallFunction::WriteUnmarshalling(CBEFile* pFile, int nStartOffset,
    bool & bUseConstOffset)
{
    assert(m_pTrace);
    bool bLocalTrace = false;
    if (!m_bTraceOn)
    {
	m_pTrace->BeforeUnmarshalling(pFile, this);
	m_bTraceOn = bLocalTrace = true;
    }
    
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
        pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0) &&
        !FindAttribute(ATTR_NOEXCEPTIONS))
        // unmarshal the exception from its word representation
        WriteEnvExceptionFromWord(pFile);
    else
        CL4BECallFunction::WriteUnmarshalling(pFile, nStartOffset,
	    bUseConstOffset);

    if (bLocalTrace)
    {
	m_pTrace->AfterUnmarshalling(pFile, this);
	m_bTraceOn = false;
    }
}

/** \brief writes the invocation code for the short IPC
 *  \param pFile the file to write to

 *
 * If we have a short IPC in both direction, we write assembler code
 * directly.
 */
void CL4X0BECallFunction::WriteInvocation(CBEFile * pFile)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
        pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0))
    {
        // skip send dope init
        WriteIPC(pFile);
        WriteIPCErrorCheck(pFile);
    }
    else
        CL4BECallFunction::WriteInvocation(pFile);
}

