/**
 *    \file    dice/src/be/l4/v2/L4V2BECallFunction.cpp
 *    \brief   contains the implementation of the class CL4V2BECallFunction
 *
 *    \date    06/01/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/* Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/l4/v2/L4V2BECallFunction.h"
#include "be/l4/L4BENameFactory.h"
#include "be/l4/L4BEClassFactory.h"
#include "be/l4/L4BEMsgBufferType.h"
#include "be/l4/L4BEIPC.h"
#include "be/l4/L4BETrace.h"
#include "be/BEFile.h"
#include "be/BEContext.h"
#include "be/BEDeclarator.h"
#include "be/BEType.h"
#include "be/BEMarshaller.h"
#include "be/BECommunication.h"
#include "Compiler.h"

#include "TypeSpec-Type.h"
#include "Attribute-Type.h"

CL4V2BECallFunction::CL4V2BECallFunction()
 : CL4BECallFunction()
{

}

/** destroys the object */
CL4V2BECallFunction::~CL4V2BECallFunction()
{
}

/** \brief writes the invocation code for the short IPC
 *  \param pFile the file to write to
 *
 * If we have a short IPC in both direction, we write assembler code
 * directly.
 *
 * We have three branches: PIC, !PIC && PROFILE, and else. We always assume
 * gcc versions above 2.95 -> no 2.7 support. And we ignore the BIGASM support.
 *
 * If no message buffer is given this is a short IPC.
 */
void CL4V2BECallFunction::WriteInvocation(CBEFile * pFile)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);

    VERBOSE("CL4V2BECallFunction::WriteInvocation(%s) called\n",
        GetName().c_str());

    if (m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
        pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0))
    {
        int nSendDirection = GetSendDirection();
        if (CCompiler::IsOptionSet(PROGRAM_USE_SYMBOLS))
        {
            if (!CCompiler::HasSymbol("__PIC__") &&
                 CCompiler::HasSymbol("PROFILE"))
            {
                pMsgBuffer->WriteInitialization(pFile, TYPE_MSGDOPE_SEND, nSendDirection);
            }
        }
        else
        {
            pFile->Print("#if !defined(__PIC__) && defined(PROFILE)\n");
            pMsgBuffer->WriteInitialization(pFile, TYPE_MSGDOPE_SEND, nSendDirection);
            pFile->Print("#endif\n");
        }
        // skip send dope init
        if (!CCompiler::IsOptionSet(PROGRAM_NO_SEND_CANCELED_CHECK))
        {
            // sometimes it's possible to abort a call of a client.
            // but the client wants his call made, so we try until
            // the call completes
            pFile->PrintIndent("do\n");
            pFile->PrintIndent("{\n");
            pFile->IncIndent();
        }
        WriteIPC(pFile);
        if (!CCompiler::IsOptionSet(PROGRAM_NO_SEND_CANCELED_CHECK))
        {
            // now check if call has been canceled
            string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
            pFile->DecIndent();
            pFile->PrintIndent("} while ((L4_IPC_ERROR(%s) == L4_IPC_SEABORTED) || (L4_IPC_ERROR(%s) == L4_IPC_SECANCELED));\n",
                                sResult.c_str(), sResult.c_str());
        }
        WriteIPCErrorCheck(pFile);
    }
    else
        CL4BECallFunction::WriteInvocation(pFile);

    VERBOSE("CL4V2BECallFunction::WriteInvocation(%s) finished\n",
        GetName().c_str());
}

/* \brief write the marshalling code for the short IPC
 * \param pFile the file to write to
 * \param nStartOffset the offset where to start marshalling
 * \param bUseConstOffset true if nStartOffset can be used

 *
 * If we have a short IPC into both direction, we skip the marshalling.
 *
 * The special about V2 is, that we have to marshal flexpage before the
 * opcode. They always have to come first.
 */
void CL4V2BECallFunction::WriteMarshalling(CBEFile * pFile,  int nStartOffset,
                           bool & bUseConstOffset)
{
    VERBOSE("CL4V2BECallFunction::WriteMarshalling(%s) called\n",
        GetName().c_str());

    assert(m_pTrace);
    bool bLocalTrace = false;
    if (!m_bTraceOn)
    {
	m_pTrace->BeforeMarshalling(pFile, this);
	m_bTraceOn = bLocalTrace = true;
    }

    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    bool bUseAsmShort = m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
      pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0);
    bool bUseSymbols = CCompiler::IsOptionSet(PROGRAM_USE_SYMBOLS);
    // for asm short IPC we only need it if !PIC && PROFILE
    if (bUseAsmShort && !bUseSymbols)
        pFile->Print("#if !defined(__PIC__) && defined(PROFILE)\n");
    if ((bUseAsmShort && bUseSymbols &&
        !CCompiler::HasSymbol("__PIC__") && CCompiler::HasSymbol("PROFILE")) ||
        !bUseAsmShort)
    {
        CL4BECallFunction::WriteMarshalling(pFile, nStartOffset,
	    bUseConstOffset);
    }
    if (bUseAsmShort && !bUseSymbols)
        pFile->Print("#endif\n");

    if (bLocalTrace)
    {
	m_pTrace->AfterMarshalling(pFile, this);
	m_bTraceOn = false;
    }

    VERBOSE("CL4V2BECallFunction::WriteMarshalling(%s) finished\n",
        GetName().c_str());
}

/* \brief write the unmarshalling code for the short IPC
 * \param pFile the file to write to
 * \param nStartOffset the offset where to start marshalling
 * \param bUseConstOffset true if nStartOffset can be used
 *
 * If we have a short IPC in both direction, we can skip the unmarshalling.
 */
void CL4V2BECallFunction::WriteUnmarshalling(CBEFile * pFile,  int nStartOffset,
    bool & bUseConstOffset)
{
    VERBOSE("CL4V2BEFunction::WriteUnmarshalling(%s) called\n",
        GetName().c_str());

    assert(m_pTrace);
    bool bLocalTrace = false;
    if (!m_bTraceOn)
    {
	m_pTrace->BeforeUnmarshalling(pFile, this);
	m_bTraceOn = bLocalTrace = true;
    }

    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
	pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0) )
    {
	// if short IPC consists of more than two parameters (bit-stuffing),
	// then we have to "unstuff" now
        // if !PIC && PROFILE we use "normal" IPC
        if (CCompiler::IsOptionSet(PROGRAM_USE_SYMBOLS))
        {
            if (!CCompiler::HasSymbol("__PIC__") &&
                 CCompiler::HasSymbol("PROFILE"))
            {
		CL4BECallFunction::WriteUnmarshalling(pFile, nStartOffset,
		    bUseConstOffset);
	    }
        }
        else
        {
            pFile->Print("#if !defined(__PIC__) && defined(PROFILE)\n");
	    CL4BECallFunction::WriteUnmarshalling(pFile, nStartOffset,
		bUseConstOffset);
	    pFile->Print("#endif\n");
        }
    }
    else
	CL4BECallFunction::WriteUnmarshalling(pFile, nStartOffset,
	    bUseConstOffset);

    if (bLocalTrace)
    {
	m_pTrace->AfterUnmarshalling(pFile, this);
	m_bTraceOn = false;
    }
    
    VERBOSE("CL4V2BEFunction::WriteUnmarshalling(%s) finished\n",
        GetName().c_str());
}

/** \brief Writes the variable declaration
 *  \param pFile the file to write to

 *
 * If we have a short IPC in both direction, we only need the result dope,
 * and two dummy dwords.
 */
void
CL4V2BECallFunction::WriteVariableDeclaration(CBEFile * pFile)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    bool bUseAssembler = m_pComm->CheckProperty(this, COMM_PROP_USE_ASM);
    bool bShortIPC =
      pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0);

    assert(m_pTrace);
    m_pTrace->VariableDeclaration(pFile, this);

    if (bUseAssembler && bShortIPC)
    {
        // write dummys
        CBENameFactory *pNF = CCompiler::GetNameFactory();
        string sDummy = pNF->GetDummyVariable();
        string sMWord = pNF->GetTypeName(TYPE_MWORD, false);
        string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);

        if (CCompiler::IsOptionSet(PROGRAM_USE_SYMBOLS))
        {
            if (CCompiler::HasSymbol("__PIC__"))
            {
                WriteReturnVariableDeclaration(pFile);
                // write result variable
                *pFile << "\tl4_msgdope_t " << sResult << " = { msgdope: 0 };\n";
                *pFile << "\t" << sMWord << " " << sDummy << " __attribute__((unused));\n";
                if (!FindAttribute(ATTR_NOEXCEPTIONS))
                    // declare local exception variable
                    WriteExceptionWordDeclaration(pFile, false);
            }
            else
            {
                if (CCompiler::HasSymbol("PROFILE"))
                    CL4BECallFunction::WriteVariableDeclaration(pFile);
                else
                {
                    WriteReturnVariableDeclaration(pFile);
                    // write result variable
                    pFile->PrintIndent("l4_msgdope_t %s = { msgdope: 0 };\n", sResult.c_str());
                    pFile->PrintIndent("%s %s = 0;\n", sMWord.c_str(), sDummy.c_str());
                    if (!FindAttribute(ATTR_NOEXCEPTIONS))
                        // declare local exception variable
                        WriteExceptionWordDeclaration(pFile, false);
                }
            }
        }
        else
        {
            // test if we need dummies
            pFile->Print("#if defined(__PIC__)\n");
            WriteReturnVariableDeclaration(pFile);
            // write result variable
            pFile->PrintIndent("l4_msgdope_t %s = { msgdope: 0 };\n", sResult.c_str());
            pFile->PrintIndent("%s %s __attribute__((unused));\n",
                                sMWord.c_str(), sDummy.c_str());

            if (!FindAttribute(ATTR_NOEXCEPTIONS))
                // declare local exception variable
                WriteExceptionWordDeclaration(pFile, false);
            pFile->Print("#else // !PIC\n");
            pFile->Print("#if defined(PROFILE)\n");
            CL4BECallFunction::WriteVariableDeclaration(pFile);

            pFile->Print("#else // !PROFILE\n");
            WriteReturnVariableDeclaration(pFile);
            // write result variable
            pFile->PrintIndent("l4_msgdope_t %s = { msgdope: 0 };\n", sResult.c_str());
            pFile->PrintIndent("%s %s = 0;\n", sMWord.c_str(), sDummy.c_str());
            if (!FindAttribute(ATTR_NOEXCEPTIONS))
                // declare local exception variable
                WriteExceptionWordDeclaration(pFile, false);
            pFile->Print("#endif // PROFILE\n");
            pFile->Print("#endif // !PIC\n");
            // if we have in either direction some bit-stuffing, we need more dummies
            // finished with declaration
        }
    }
    else
    {
        CL4BECallFunction::WriteVariableDeclaration(pFile);
        if (bUseAssembler)
        {
            // need dummies
            CBENameFactory *pNF = CCompiler::GetNameFactory();
            string sDummy = pNF->GetDummyVariable();
            string sMWord = pNF->GetTypeName(TYPE_MWORD, false);
            if (CCompiler::IsOptionSet(PROGRAM_USE_SYMBOLS))
            {
                if (CCompiler::HasSymbol("__PIC__"))
                    pFile->PrintIndent("%s %s __attribute__((unused));\n",
                        sMWord.c_str(), sDummy.c_str());
                else if (!CCompiler::HasSymbol("PROFILE"))
                    pFile->PrintIndent("%s %s __attribute__((unused));\n",
                        sMWord.c_str(), sDummy.c_str());
            }
            else
            {
                pFile->Print("#if defined(__PIC__)\n");
                pFile->PrintIndent("%s %s __attribute__((unused));\n",
                    sMWord.c_str(), sDummy.c_str());
                pFile->Print("#else // !PIC\n");
                pFile->Print("#if !defined(PROFILE)\n");
                pFile->PrintIndent("%s %s __attribute__((unused));\n",
                    sMWord.c_str(), sDummy.c_str());
                pFile->Print("#endif // !PROFILE\n");
                pFile->Print("#endif // !PIC\n");
            }
        }
    }
}

/** \brief writes the variable initialization
 *  \param pFile the file to write to

 *
 * If we use the asm short IPC code we do not init any variables (Its done during declaration).
 */
void CL4V2BECallFunction::WriteVariableInitialization(CBEFile * pFile)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (m_pComm->CheckProperty(this, COMM_PROP_USE_ASM) &&
	pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0))
    {
        if (CCompiler::IsOptionSet(PROGRAM_USE_SYMBOLS))
        {
            if (!CCompiler::HasSymbol("__PIC__") &&
                 CCompiler::HasSymbol("PROFILE"))
            {
                CL4BECallFunction::WriteVariableInitialization(pFile);
            }
        }
        else
        {
            pFile->Print("#if !defined(__PIC__) && defined(PROFILE)\n");
            CL4BECallFunction::WriteVariableInitialization(pFile);
            pFile->Print("#endif // !PIC && PROFILE\n");
        }
    }
    else
        CL4BECallFunction::WriteVariableInitialization(pFile);
}

