/**
 *    \file    dice/src/be/l4/L4BEWaitAnyFunction.cpp
 *    \brief   contains the implementation of the class CL4BEWaitAnyFunction
 *
 *    \date    03/07/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "L4BEWaitAnyFunction.h"
#include "L4BENameFactory.h"
#include "L4BEClassFactory.h"
#include "L4BESizes.h"
#include "be/BEContext.h"
#include "be/BEFile.h"
#include "be/BEUserDefinedType.h"
#include "be/BEDeclarator.h"
#include "be/BETypedDeclarator.h"
#include "be/BEOperationFunction.h"
#include "L4BEMsgBufferType.h"
#include "L4BEIPC.h"
#include "be/BEMarshaller.h"
#include "be/BETrace.h"

#include "TypeSpec-Type.h"
#include "Attribute-Type.h"
#include "Compiler.h"

CL4BEWaitAnyFunction::CL4BEWaitAnyFunction(bool bOpenWait, bool bReply)
: CBEWaitAnyFunction(bOpenWait, bReply)
{
}

CL4BEWaitAnyFunction::CL4BEWaitAnyFunction(CL4BEWaitAnyFunction & src)
: CBEWaitAnyFunction(src)
{
}

/** \brief destructor of target class */
CL4BEWaitAnyFunction::~CL4BEWaitAnyFunction()
{

}

/** \brief writes the variable declarations of this function
 *  \param pFile the file to write to

 *
 * The variable declarations of the wait-any function only contains so-called
 * helper variables. This is the result variable.
 */
void
CL4BEWaitAnyFunction::WriteVariableDeclaration(CBEFile * pFile)
{
    // first call base class
    CBEWaitAnyFunction::WriteVariableDeclaration(pFile);

    // write result variable
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sResult = pNF->GetString(STR_RESULT_VAR);
    *pFile << "\tl4_msgdope_t " << sResult << " = { msgdope: 0 };\n";

    CBEClass *pClass = GetSpecificParent<CBEClass>();
    assert(pClass);
    if (pClass->FindAttribute(ATTR_DEDICATED_PARTNER))
    {
	string sDummy = pNF->GetDummyVariable();
	string sMWord = pNF->GetTypeName(TYPE_MWORD, false);
	*pFile << "\t" << sMWord << " " << sDummy << 
	    " __attribute__((unused));\n";
    }

    assert(m_pTrace);
    m_pTrace->VariableDeclaration(pFile, this);
}

/** \brief initializes the variables
 *  \param pFile the file to write to

 *
 * For reply only:
 * We do not initialize the receive indirect strings, because we assume that
 * they have been initialized by the server loop. After that the buffer is
 * handed to the server function. If they intend to use it after the component
 * function is left, they have to copy it.
 *
 * The receive flexpage is reinitialized, because it might have changed.
 */
void
CL4BEWaitAnyFunction::WriteVariableInitialization(CBEFile * pFile)
{
    // call base class
    CBEWaitAnyFunction::WriteVariableInitialization(pFile);
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (pMsgBuffer && m_bReply)
    {
        // init receive flexpage
        pMsgBuffer->WriteInitialization(pFile, TYPE_FLEXPAGE,
            GetReceiveDirection());
    }
}

/** \brief writes the invocation call to thetarget file
 *  \param pFile the file to write to
 *
 * The wait any function simply waits for any message and unmarshals the opcode.
 * Since the message buffer is a referenced parameter, we know for sure, that
 * the "buffer" is a pointer.
 */
void CL4BEWaitAnyFunction::WriteInvocation(CBEFile * pFile)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    // init size and send dopes
    if (!m_bReply)
        // do not overwrite message buffer if reply is sent
        pMsgBuffer->WriteInitialization(pFile, TYPE_MSGDOPE_SEND, 0);

    // invocate
    WriteIPC(pFile);
    WriteExceptionCheck(pFile); // reset exception

    // print trace code before IPC error check to have unmodified values in
    // message buffer
    assert(m_pTrace);
    m_pTrace->AfterReplyWait(pFile, this);
    
    WriteIPCErrorCheck(pFile); // set IPC exception

    if (m_bReply)
        WriteReleaseMemory(pFile);
}

/** \brief writes the ipc code
 *  \param pFile the file to write to

 */
void CL4BEWaitAnyFunction::WriteIPC(CBEFile *pFile)
{
    assert(m_pComm);
    if (m_bOpenWait)
    {
        if (m_bReply)
            WriteIPCReplyWait(pFile);
        else
	{
	    CBEClass *pClass = GetSpecificParent<CBEClass>();
	    assert(pClass);
	    if (pClass->FindAttribute(ATTR_DEDICATED_PARTNER))
		WriteDedicatedWait(pFile);
	    else
		m_pComm->WriteWait(pFile, this);
	}
    }
    else
        m_pComm->WriteReceive(pFile, this);
}

/** \brief writes the switch between receive and wait
 *  \param pFile the file to write to
 */
void
CL4BEWaitAnyFunction::WriteDedicatedWait(CBEFile *pFile)
{
    CL4BENameFactory *pNF = static_cast<CL4BENameFactory*>(
	CCompiler::GetNameFactory());
    string sPartner = pNF->GetPartnerVariable();
    *pFile << "\tif (l4_is_invalid_id(" << sPartner << "))" <<
	" /* no dedicated partner */\n";
    pFile->IncIndent();
    m_pComm->WriteWait(pFile, this);
    pFile->DecIndent();
    *pFile << "\telse /* dedicated partner */\n";
    pFile->IncIndent();
    m_pComm->WriteReceive(pFile, this);
    pFile->DecIndent();
}

/** \brief writes the ipc code
 *  \param pFile the file to write to
 */
void
CL4BEWaitAnyFunction::WriteIPCReplyWait(CBEFile *pFile)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    
    assert(m_pTrace);
    m_pTrace->BeforeReplyWait(pFile, this);

    CL4BESizes *pSizes = (CL4BESizes*)CCompiler::GetSizes();
    int nShortWords = pSizes->GetMaxShortIPCSize(DIRECTION_IN) /
	pSizes->GetSizeOfType(TYPE_MWORD);
    // to determine if we can send a short IPC we have to test the size dope
    // of the message
    pFile->PrintIndent("if ((");
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_OUT);
    pFile->Print(".md.dwords <= %d) && (", nShortWords);
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_OUT);
    pFile->Print(".md.strings == 0))\n");
    pFile->IncIndent();
    // if fpage
    pFile->PrintIndent("if (");
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_OUT);
    pFile->Print(".md.fpage_received == 1)\n");
    pFile->IncIndent();
    // short IPC
    WriteShortFlexpageIPC(pFile);
    // else (fpage)
    pFile->DecIndent();
    pFile->PrintIndent("else\n");
    pFile->IncIndent();
    // !fpage
    WriteShortIPC(pFile);
    pFile->DecIndent();
    pFile->DecIndent();
    pFile->PrintIndent("else\n");
    pFile->IncIndent();
    // if fpage
    pFile->PrintIndent("if (");
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_OUT);
    pFile->Print(".md.fpage_received == 1)\n");
    pFile->IncIndent();
    // long IPC
    WriteLongFlexpageIPC(pFile);
    // else (fpage)
    pFile->DecIndent();
    pFile->PrintIndent("else\n");
    pFile->IncIndent();
    // ! fpage
    WriteLongIPC(pFile);
    pFile->DecIndent();
    pFile->DecIndent();
}

/** \brief write the ip code with a short msg reply containing a flexpage
 *  \param pFile the file to write to
 */
void
CL4BEWaitAnyFunction::WriteShortFlexpageIPC(CBEFile *pFile)
{
    assert(m_pComm);

    CBEClass *pClass = GetSpecificParent<CBEClass>();
    assert(pClass);
    if (pClass->FindAttribute(ATTR_DEDICATED_PARTNER))
    {
	CL4BENameFactory *pNF = static_cast<CL4BENameFactory*>(
	    CCompiler::GetNameFactory());
	string sPartner = pNF->GetPartnerVariable();
	*pFile << "\tif (l4_is_invalid_id(" << sPartner << "))" <<
	    " /* no dedicated partner */\n";
	pFile->IncIndent();
	((CL4BEIPC*)m_pComm)->WriteReplyAndWait(pFile, this, true, true);
	pFile->DecIndent();
	*pFile << "\telse /* dedicated partner */\n";
	pFile->IncIndent();
	((CL4BEIPC*)m_pComm)->WriteCall(pFile, this, true, true);
	pFile->DecIndent();
    }
    else
	((CL4BEIPC*)m_pComm)->WriteReplyAndWait(pFile, this, true, true);
}

/** \brief write the ipc code with a short msg reply
 *  \param pFile the file to write to
 */
void CL4BEWaitAnyFunction::WriteShortIPC(CBEFile *pFile)
{
    assert(m_pComm);

    CBEClass *pClass = GetSpecificParent<CBEClass>();
    assert(pClass);
    if (pClass->FindAttribute(ATTR_DEDICATED_PARTNER))
    {
	CL4BENameFactory *pNF = static_cast<CL4BENameFactory*>(
	    CCompiler::GetNameFactory());
	string sPartner = pNF->GetPartnerVariable();
	*pFile << "\tif (l4_is_invalid_id(" << sPartner << "))" <<
	    " /* no dedicated partner */\n";
	pFile->IncIndent();
	((CL4BEIPC*)m_pComm)->WriteReplyAndWait(pFile, this, false, true);
	pFile->DecIndent();
	*pFile << "\telse /* dedicated partner */\n";
	pFile->IncIndent();
	((CL4BEIPC*)m_pComm)->WriteCall(pFile, this, false, true);
	pFile->DecIndent();
    }
    else
	((CL4BEIPC*)m_pComm)->WriteReplyAndWait(pFile, this, false, true);
}

/** \brief write the ipc code with a long msg containing flexpages
 *  \param pFile the file to write to
 */
void
CL4BEWaitAnyFunction::WriteLongFlexpageIPC(CBEFile *pFile)
{
    assert(m_pComm);

    CBEClass *pClass = GetSpecificParent<CBEClass>();
    assert(pClass);
    if (pClass->FindAttribute(ATTR_DEDICATED_PARTNER))
    {
	CL4BENameFactory *pNF = static_cast<CL4BENameFactory*>(
	    CCompiler::GetNameFactory());
	string sPartner = pNF->GetPartnerVariable();
	*pFile << "\tif (l4_is_invalid_id(" << sPartner << "))" <<
	    " /* no dedicated partner */\n";
	pFile->IncIndent();
	((CL4BEIPC*)m_pComm)->WriteReplyAndWait(pFile, this, true, false);
	pFile->DecIndent();
	*pFile << "\telse /* dedicated partner */\n";
	pFile->IncIndent();
	((CL4BEIPC*)m_pComm)->WriteCall(pFile, this, true, false);
	pFile->DecIndent();
    }
    else
	((CL4BEIPC*)m_pComm)->WriteReplyAndWait(pFile, this, true, false);
}

/** \brief write ipc code with a long msg
 *  \param pFile the file to write to
 */
void CL4BEWaitAnyFunction::WriteLongIPC(CBEFile *pFile)
{
    assert(m_pComm);

    CBEClass *pClass = GetSpecificParent<CBEClass>();
    assert(pClass);
    if (pClass->FindAttribute(ATTR_DEDICATED_PARTNER))
    {
	CL4BENameFactory *pNF = static_cast<CL4BENameFactory*>(
	    CCompiler::GetNameFactory());
	string sPartner = pNF->GetPartnerVariable();
	*pFile << "\tif (l4_is_invalid_id(" << sPartner << "))" <<
	    " /* no dedicated partner */\n";
	pFile->IncIndent();
	((CL4BEIPC*)m_pComm)->WriteReplyAndWait(pFile, this, false, false);
	pFile->DecIndent();
	*pFile << "\telse /* dedicated partner */\n";
	pFile->IncIndent();
	((CL4BEIPC*)m_pComm)->WriteCall(pFile, this, false, false);
	pFile->DecIndent();
    }
    else
	((CL4BEIPC*)m_pComm)->WriteReplyAndWait(pFile, this, false, false);
}

/** \brief write the checking code for opcode exceptions
 *  \param pFile the file to write to
 *
 * reset any previous exceptions. Must be called before IPC Error check
 */
void
CL4BEWaitAnyFunction::WriteExceptionCheck(CBEFile * pFile)
{
    // set exception if not set already
    *pFile << "\t// clear exception if set\n";
    CBEDeclarator *pDecl = m_pCorbaEnv->GetDeclarator();
    pFile->PrintIndent("if (");
    pDecl->WriteName(pFile);
    if (pDecl->GetStars() > 0)
        pFile->Print("->");
    else
        pFile->Print(".");
    pFile->Print("major != CORBA_NO_EXCEPTION)\n");
    pFile->IncIndent();
    // set exception
    pFile->PrintIndent("CORBA_server_exception_set(");
    if (pDecl->GetStars() == 0)
        pFile->Print("&");
    pDecl->WriteName(pFile);
    pFile->Print(",\n");
    pFile->IncIndent();
    pFile->PrintIndent("CORBA_NO_EXCEPTION,\n");
    pFile->PrintIndent("CORBA_DICE_EXCEPTION_NONE,\n");
    pFile->PrintIndent("0);\n");
    pFile->DecIndent();
    pFile->DecIndent();
}

/** \brief write the error checking code for the IPC
 *  \param pFile the file to write to

 */
void CL4BEWaitAnyFunction::WriteIPCErrorCheck(CBEFile * pFile)
{
    string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
    pFile->PrintIndent("/* test for IPC errors */\n");
    pFile->PrintIndent("if (L4_IPC_IS_ERROR(%s))\n", sResult.c_str());
    pFile->PrintIndent("{\n");
    pFile->IncIndent();
    // set opcode to zero value
    if (m_pReturnVar)
        m_pReturnVar->WriteSetZero(pFile);
    if (!m_sErrorFunction.empty())
    {
        *pFile << "\t" << m_sErrorFunction << "(" << sResult << ", ";
        WriteCallParameter(pFile, m_pCorbaEnv);
        *pFile << ");\n";
    }
    // set zero value in msgbuffer
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEMarshaller *pMarshaller = pCF->GetNewMarshaller();
    bool bUseConstOffset = true;
    pMarshaller->MarshalValue(pFile, this,
	CCompiler::GetSizes()->GetOpcodeSize(), 0, 0, bUseConstOffset);
    delete pMarshaller;
    // set exception if not set already
    CBEDeclarator *pDecl = m_pCorbaEnv->GetDeclarator();;
    pFile->PrintIndent("if (");
    pDecl->WriteName(pFile);
    if (pDecl->GetStars() > 0)
        pFile->Print("->");
    else
        pFile->Print(".");
    pFile->Print("major == CORBA_NO_EXCEPTION)\n");
    pFile->IncIndent();
    // set exception
    string sSetFunc;
    if (((CBEUserDefinedType*)m_pCorbaEnv->GetType())->GetName() ==
        "CORBA_Server_Environment")
        sSetFunc = "CORBA_server_exception_set";
    else
        sSetFunc = "CORBA_exception_set";
    *pFile << "\t" << sSetFunc << "(";
    if (pDecl->GetStars() == 0)
        pFile->Print("&");
    pDecl->WriteName(pFile);
    pFile->Print(",\n");
    pFile->IncIndent();
    pFile->PrintIndent("CORBA_SYSTEM_EXCEPTION,\n");
    pFile->PrintIndent("CORBA_DICE_INTERNAL_IPC_ERROR,\n");
    pFile->PrintIndent("0);\n");
    pFile->DecIndent();
    pFile->DecIndent();
    // returns 0 -> falls into default branch of server loop
    WriteReturn(pFile);
    pFile->DecIndent();
    pFile->PrintIndent("}\n");
}

/** \brief free memory allocated at the server which should be freed after the reply
 *  \param pFile the file to write to

 *
 * This function uses the dice_get_last_ptr method to free memory
 *
 * Only print this if the class has any [out, ref] parameters and if the option
 * -ffree_mem_after_reply is set
 */
void
CL4BEWaitAnyFunction::WriteReleaseMemory(CBEFile *pFile)
{
    // check [out, ref] parameters
    assert(m_pClass);
    if (!CCompiler::IsOptionSet(PROGRAM_FREE_MEM_AFTER_REPLY) &&
        !m_pClass->HasParametersWithAttribute(ATTR_REF, ATTR_OUT))
        return;

    pFile->PrintIndent("{\n");
    pFile->IncIndent();
    pFile->PrintIndent("void* ptr;\n");
    pFile->PrintIndent("while ((ptr = dice_get_last_ptr(");
    // env
    CBEDeclarator *pDecl = m_pCorbaEnv->GetDeclarator();
    if (pDecl->GetStars() == 0)
        pFile->Print("&");
    pDecl->WriteName(pFile);
    pFile->Print(")) != 0)\n");
    pFile->IncIndent();
    pFile->PrintIndent("");
    CBEContext::WriteFree(pFile, this);
    pFile->Print("(ptr);\n");
    pFile->DecIndent();
    pFile->DecIndent();
    pFile->PrintIndent("}\n");
}

/** \brief writes the unmarshalling code for this function
 *  \param pFile the file to write to
 *  \param nStartOffset the offset where to start unmarshalling
 *  \param bUseConstOffset true if a constant offset should be used, set it to \
 *           false if not possible
 *
 * The wait-any function does only unmarshal the opcode. We can print this code
 * by hand. We should use a marshaller anyways.
 */
void CL4BEWaitAnyFunction::WriteUnmarshalling(CBEFile * pFile,
    int nStartOffset,
    bool& bUseConstOffset)
{
    assert (m_pTrace);
    bool bLocalTrace = false;
    if (!m_bTraceOn)
    {
	m_pTrace->BeforeUnmarshalling(pFile, this);
	m_bTraceOn = bLocalTrace = true;
    }
    
    /* If the option noopcode is set, we do not unmarshal anything at all. */
    if (FindAttribute(ATTR_NOOPCODE))
        return;
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (pMsgBuffer->GetCount(TYPE_FLEXPAGE, GetReceiveDirection()) > 0)
    {
        // we have to always check if this was a flexpage IPC
	//
	// Because the wait-any function always has a return type (the opcode)
	// we do not have to check for it separately
        string sResult =
           CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
        *pFile << "\tif (l4_ipc_fpage_received(" << sResult << "))\n";
        WriteFlexpageOpcodePatch(pFile);  // does indent itself
        *pFile << "\telse\n";
        pFile->IncIndent();
        WriteUnmarshalReturn(pFile, nStartOffset, bUseConstOffset);
        pFile->DecIndent();
    }
    else
    {
        WriteUnmarshalReturn(pFile, nStartOffset, bUseConstOffset);
    }

    if (bLocalTrace)
    {
	m_pTrace->AfterUnmarshalling(pFile, this);
	m_bTraceOn = false;
    }
}

/** \brief writes a patch to find the opcode if flexpage were received
 *  \param pFile the file to write to

 *
 * This function may receive messages from different function. Because we
 * don't know at compile time, which function sends, we don't know if the
 * message contains a flexpage.  If it does the unmarshalled opcode is wrong.
 * Because flexpages have to come first in the message buffer, the opcode
 * cannot be the first parameter. We have to check this condition and get the
 * opcode from behind the flexpages.
 *
 * First we get the number of flexpages of the interface. If it has none, we
 * don't need this extra code. If it has a fixed number of flexpages (either
 * none is sent or one, but if flexpages are sent it is always the same number
 * of flexpages) we can hard code the offset were to find the opcode. If we
 * have different numbers of flexpages (one function may send one, another
 * sends two) we have to use code which can deal with a variable number of
 * flexpages.
 */
void CL4BEWaitAnyFunction::WriteFlexpageOpcodePatch(CBEFile *pFile)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (pMsgBuffer->GetCount(TYPE_FLEXPAGE, GetReceiveDirection()) == 0)
        return;
    bool bFixedNumberOfFlexpages = true;
    int nNumberOfFlexpages = m_pClass->GetParameterCount(TYPE_FLEXPAGE, bFixedNumberOfFlexpages);
    // if fixed number  (should be true for only one flexpage as well)
    if (bFixedNumberOfFlexpages)
    {
        // the fixed offset (where to find the opcode) is:
        // offset = 8*nMaxNumberOfFlexpages + 8
        bool bUseConstOffset = true;
        pFile->IncIndent();
        WriteUnmarshalReturn(pFile, 8*nNumberOfFlexpages+8, bUseConstOffset);
        pFile->DecIndent();
    }
    else
    {
	CBESizes *pSizes = CCompiler::GetSizes();
	int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);
	// the variable offset can be determined by searching for the
	// delimiter flexpage which is two zero dwords
        pFile->PrintIndent("{\n");
        pFile->IncIndent();
        // search for delimiter flexpage
        string sTempVar = CCompiler::GetNameFactory()->GetTempOffsetVariable();
        // init temp var
        pFile->PrintIndent("%s = 0;\n", sTempVar.c_str());
        pFile->PrintIndent("while ((");
        pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_OUT);
        pFile->Print("[%s] != 0) && (", sTempVar.c_str());
        pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_OUT);
        pFile->Print("[%s+%d] != 0)) %s += %d;\n", sTempVar.c_str(), 
	    nWordSize, sTempVar.c_str(), nWordSize*2);
        // now sTempVar points to the delimiter flexpage
	// we have to add another 8 bytes to find the opcode, because
	// UnmarshalReturn does only use temp-var
        pFile->PrintIndent("%s += %d;\n", sTempVar.c_str(), nWordSize*2);
        // now unmarshal opcode
        bool bUseConstOffset = false;
        WriteUnmarshalReturn(pFile, 0, bUseConstOffset);
        pFile->DecIndent();
        pFile->PrintIndent("}\n");
    }
}
