/**
 *    \file    dice/src/be/l4/L4BETrace.cpp
 *    \brief   contains the implementation of the class CL4BETrace
 *
 *    \date    12/05/2005
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2005
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "L4BETrace.h"
#include "L4BEMsgBufferType.h"
#include "L4BEIPC.h"
#include "L4BENameFactory.h"
#include "be/BEContext.h"
#include "be/BEFunction.h"
#include "be/BECommunication.h"
#include "be/BETypedDeclarator.h"
#include "be/BEDeclarator.h"
#include "be/BEMarshaller.h"
#include "be/BESizes.h"
#include "TypeSpec-Type.h"
#include "Compiler.h"

CL4BETrace::CL4BETrace()
{
}

CL4BETrace::~CL4BETrace()
{
}

/** \brief writes necessary variable declarations
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CL4BETrace::VariableDeclaration(CBEFile *pFile,
    CBEFunction *pFunction)
{
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    bool bShortIPC =
      pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, 0);
    
    // write loop variable for msg buffer dump
    if (CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF) &&
        !bShortIPC)
        *pFile << "\tint _i;\n";
}

/** \brief prints the tracing message before a call
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CL4BETrace::BeforeCall(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!(CCompiler::IsOptionSet(PROGRAM_TRACE_CLIENT) ||
	  CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF)) &&
	!pFunction->IsComponentSide())
	return;
    if (!(CCompiler::IsOptionSet(PROGRAM_TRACE_SERVER) ||
	  CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF)) &&
	pFunction->IsComponentSide())
	return;

    int nSndDir = pFunction->GetSendDirection();
    int nRcvDir = pFunction->GetReceiveDirection();
    // check if we send a short IPC
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    bool bIsShortIPC =
      pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, nSndDir) &&
      pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, nRcvDir);
    // check if we use assembler for call
    CBECommunication *pComm = pFunction->GetCommunication();
    assert(pComm);
    bool bUseAssembler = pComm->CheckProperty(pFunction, COMM_PROP_USE_ASM);
    // get the CORBA Object name
    CBETypedDeclarator *pObj = pFunction->GetObject();
    assert(pObj);
    CBEDeclarator *pObjName = pObj->GetDeclarator();
    
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    // get tracing function
    string sFunc = CCompiler::GetTraceClientFunc();

    if (CCompiler::IsOptionSet(PROGRAM_TRACE_CLIENT) ||
	CCompiler::IsOptionSet(PROGRAM_TRACE_SERVER))
    {
	string sMWord = pNF->GetTypeName(TYPE_MWORD, false, 0);
	*pFile << "\t" << sFunc << " (\"" << pFunction->GetName() << 
	    ": server %%2X.%%X\\n\", " << pObjName->GetName() << 
	    "->id.task, " << pObjName->GetName() << "->id.lthread);\n";
	*pFile << "\t" << sFunc << " (\"" << pFunction->GetName() <<
	    ": with dw0=0x%%x, dw1=0x%%x\\n\", ";
	if (bUseAssembler && bIsShortIPC)
	{
	    string sOpcode = pFunction->GetOpcodeConstName();
	    *pFile << sOpcode << ", (unsigned)";
	    CBEMarshaller *pMarshaller = pCF->GetNewMarshaller();
	    if (!pMarshaller->MarshalToPosition(pFile, pFunction, 1, nWordSize, 
		    nSndDir, false))
		*pFile << "0";
	}
	else
	{
	    *pFile << "(unsigned)(*((" << sMWord << "*)(&(";
	    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
	    *pFile << "[0])))), ";
	    *pFile << "(unsigned)(*((" << sMWord << "*)(&(";
	    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
	    *pFile << "[" << nWordSize << "]))))";
	}
	*pFile << ");\n";
    }
    
    if (CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF) &&
	!bIsShortIPC)
    {
	*pFile << "\t" << sFunc << " (\"" << pFunction->GetName() <<
	    ": before call\\n\");\n";
        pMsgBuffer->WriteDump(pFile, string());
    }
}

/** \brief write the tracing code after the call
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CL4BETrace::AfterCall(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!(CCompiler::IsOptionSet(PROGRAM_TRACE_CLIENT) ||
	  CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF)) &&
	!pFunction->IsComponentSide())
	return;
    if (!(CCompiler::IsOptionSet(PROGRAM_TRACE_SERVER) ||
	  CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF)) &&
	pFunction->IsComponentSide())
	return;

    int nSndDir = pFunction->GetSendDirection();
    int nRcvDir = pFunction->GetReceiveDirection();
    // check if we send a short IPC
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    bool bIsShortIPC =
      pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, nSndDir) &&
      pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, nRcvDir);

    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sResult = pNF->GetString(STR_RESULT_VAR);

    // get tracing function
    string sFunc = CCompiler::GetTraceClientFunc();

    if (CCompiler::IsOptionSet(PROGRAM_TRACE_CLIENT) ||
	CCompiler::IsOptionSet(PROGRAM_TRACE_SERVER))
    {
	*pFile << "\t" << sFunc << " (\"" << pFunction->GetName() << 
	    ": return dope %%x (ipc error %%x)\\n\", (unsigned)" << 
	    sResult << ".msgdope, (unsigned)L4_IPC_ERROR(" << 
	    sResult << "));\n";
    }

    if (CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF) &&
	!bIsShortIPC)
    {
	*pFile << "\t" << sFunc << " (\"" << pFunction->GetName() <<
	    ": after call\\n\");\n";
        pMsgBuffer->WriteDump(pFile, sResult);
    }
}

/** \brief write the tracing code shortly before calling dispatch func
 *  \param pFile the file to write to
 *  \param pFunction the dispatch function to write for
 */
void
CL4BETrace::BeforeDispatch(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!pFunction->IsComponentSide())
	return;
    if (!CCompiler::IsOptionSet(PROGRAM_TRACE_SERVER))
	return;
    
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sOpcodeVar = pNF->GetOpcodeVariable();
    string sObjectVar = pNF->GetCorbaObjectVariable();
    string sMWord = pNF->GetTypeName(TYPE_MWORD, false);
    string sFunc = CCompiler::GetTraceServerFunc();
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    *pFile << "\t" << sFunc << " (\"opcode %%x received from %%2X.%%X\\n\", "
	<< sOpcodeVar << ", " << sObjectVar << "->id.task, " << sObjectVar <<
	"->id.lthread);\n";
    *pFile << "\t" << sFunc << " (\"received dw0=%%x, dw1=%%x\\n\", "
	<< "(unsigned)(*((" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    *pFile << "[0])))), ";
    *pFile << "(unsigned)(*((" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    *pFile << "[" << nWordSize << "])))) );\n";
}

/** \brief write the tracing code shortly after the call to dispatch func
 *  \param pFile the file to write to
 *  \param pFunction the dispatch function to write for
 */
void
CL4BETrace::AfterDispatch(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!pFunction->IsComponentSide())
	return;
    if (!CCompiler::IsOptionSet(PROGRAM_TRACE_SERVER))
	return;
    
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sMWord = pNF->GetTypeName(TYPE_MWORD, false);
    string sFunc = CCompiler::GetTraceServerFunc();
    string sReply = pNF->GetReplyCodeVariable();
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    *pFile << "\t" << sFunc << " (\"reply %%s (dw0=%%x, dw1=%%x)\\n\", (" <<
	sReply << "==DICE_REPLY)?\"DICE_REPLY\":" <<
	"((" << sReply << "==DICE_DEFERRED_REPLY)?\"DICE_DEFERRED_REPLY\":" <<
	"\"DICE_NEVER_REPLY\"), " <<
	"(unsigned)(*((" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    *pFile << "[0])))), ";
    *pFile << "(unsigned)(*((" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    *pFile << "[" << nWordSize << "])))) );\n";
    // print if we got an fpage
    *pFile << "\t" << sFunc << " (\"  fpage: %%s\\n\", (";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    *pFile << ".md.fpage_received==1)?\"yes\":\"no\");\n";
}

/** \brief write the tracing the code before reply IPC
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CL4BETrace::BeforeReplyWait(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!pFunction->IsComponentSide())
	return;
    if (!CCompiler::IsOptionSet(PROGRAM_TRACE_SERVER))
	return;

    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sMWord = pNF->GetTypeName(TYPE_MWORD, false);
    string sFunc = CCompiler::GetTraceServerFunc();
    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    *pFile << "\t" << sFunc << " (\"reply (dw0=%%x, dw1=%%x)\\n\", " <<
        "(unsigned)(*((" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_OUT);
    *pFile << "[0])))), ";
    *pFile << "(unsigned)(*((" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_OUT);
    *pFile << "[" << nWordSize << "])))) );\n";
    // dwords
    *pFile << "\t" << sFunc << " (\"  words: %%d\\n\", ";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_OUT);
    *pFile << ".md.dwords);\n";
    // strings
    *pFile << "\t" << sFunc << " (\"  strings; %%d\\n\", ";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_OUT);
    *pFile << ".md.strings);\n";
    // print if we got an fpage
    *pFile << "\t" << sFunc << " (\"  fpage: %%s\\n\", (";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_OUT);
    *pFile << ".md.fpage_received==1)?\"yes\":\"no\");\n";
}

/** \brief writes the tracing code after the server waited for a message
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CL4BETrace::AfterReplyWait(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF))
	return;
    
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sResult = pNF->GetString(STR_RESULT_VAR);
    // get tracing function
    string sFunc = CCompiler::GetTraceClientFunc();

    *pFile << "\t" << sFunc << " (\"" << pFunction->GetName() <<
	": after wait\\n\");\n";
    pMsgBuffer->WriteDump(pFile, sResult);
}
