/**
 *    \file    dice/src/be/l4/L4BESrvLoopFunction.cpp
 *    \brief   contains the implementation of the class CL4BESrvLoopFunction
 *
 *    \date    02/10/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/l4/L4BENameFactory.h"
#include "be/l4/L4BESrvLoopFunction.h"
#include "be/l4/L4BEMsgBufferType.h"
#include "be/BEContext.h"
#include "be/BEFunction.h"
#include "be/BEOperationFunction.h"
#include "be/BEClass.h"
#include "be/BETypedDeclarator.h"
#include "be/BEType.h"
#include "be/BEContext.h"
#include "be/BEDeclarator.h"
#include "be/BEWaitAnyFunction.h"

#include "TypeSpec-Type.h"
#include "Attribute-Type.h"
#include "Compiler.h"

CL4BESrvLoopFunction::CL4BESrvLoopFunction()
{
}

CL4BESrvLoopFunction::CL4BESrvLoopFunction(CL4BESrvLoopFunction & src)
:CBESrvLoopFunction(src)
{
}

/** \brief destructor of target class */
CL4BESrvLoopFunction::~CL4BESrvLoopFunction()
{

}

/** \brief write the declaration of the CORBA_Object variable
 *  \param pFile the file to write to
 */
void CL4BESrvLoopFunction::WriteCorbaObjectDeclaration(CBEFile *pFile)
{
    if (m_pCorbaObject)
    {
        CBEDeclarator *pName = GetObject()->GetDeclarator();
	*pFile << "\tCORBA_Object_base _" << pName->GetName() << ";\n";
	*pFile << "\t";
        GetObject()->WriteDeclaration(pFile);
	*pFile << " = &_" << pName->GetName() << "; // is client id\n";
    }
}

/** \brief writes the varaible initialization
 *  \param pFile the file to write to
 */
void CL4BESrvLoopFunction::WriteVariableInitialization(CBEFile * pFile)
{
    // call base class - initializes opcode
    CBESrvLoopFunction::WriteVariableInitialization(pFile);
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    // zero msg buffer
    if (CCompiler::IsOptionSet(PROGRAM_ZERO_MSGBUF))
        pMsgBuffer->WriteSetZero(pFile);
    // set the size dope here, so we do not need to set it anywhere else
    pMsgBuffer->WriteInitialization(pFile, TYPE_MSGDOPE_SIZE, 0);
    // init receive flexpage
    pMsgBuffer->WriteInitialization(pFile, TYPE_FLEXPAGE,
        GetReceiveDirection());
    // init indirect strings
    pMsgBuffer->WriteInitialization(pFile, TYPE_REFSTRING,
        GetReceiveDirection());

    // set CORBA_Object depending on [dedicated_partner]. Here, the
    // environment is initialized, so it is save to use the environment's
    // partner member (independent of server parameter)
    CBEClass *pClass = GetSpecificParent<CBEClass>();
    assert(pClass);
    if (pClass->FindAttribute(ATTR_DEDICATED_PARTNER))
    {
	string sObj = GetObject()->GetDeclarator()->GetName();
	CL4BENameFactory *pNF =
	    static_cast<CL4BENameFactory*>(CCompiler::GetNameFactory());
	string sPartner = pNF->GetPartnerVariable();
	*pFile << "\t_" << sObj << " = " << sPartner << ";\n";
    }
}

/** \brief create this instance of a server loop function
 *  \param pFEInterface the interface to use as reference
 *  \return true if create was successful
 *
 * We have to check whether the server loop might receive flexpages. If it
 * does, we have to use a server function parameter (the Corba Environment) to
 * set the receive flexpage.  To find out if we have receive flexpages, we use
 * the message buffer, but we then have to set the context option.
 */
bool CL4BESrvLoopFunction::CreateBackEnd(CFEInterface * pFEInterface)
{
    if (!CBESrvLoopFunction::CreateBackEnd(pFEInterface))
        return false;
    // test for flexpages
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if ((pMsgBuffer->GetCount(TYPE_FLEXPAGE, GetReceiveDirection()) > 0) &&
        m_pCorbaEnv)
    {
        CBEDeclarator *pDecl = m_pCorbaEnv->GetDeclarator();
        if (pDecl->GetStars() == 0)
            pDecl->IncStars(1);
        // set the call variables
        if (m_pWaitAnyFunction)
            m_pWaitAnyFunction->SetCallVariable(pDecl->GetName(),
                pDecl->GetStars(), pDecl->GetName());
        if (m_pReplyAnyWaitAnyFunction)
            m_pReplyAnyWaitAnyFunction->SetCallVariable(pDecl->GetName(),
                pDecl->GetStars(), pDecl->GetName());
    }
    return true;
}

/** \brief writes the parameters of the server loop
 *  \param pFile the file to write to
 *  \param bComma true if a comma has to be written before the parameters
 *
 * This server loop will only have one parameter: a void pointer.
 */
void CL4BESrvLoopFunction::WriteAfterParameters(CBEFile * pFile, bool bComma)
{
    if (bComma)
        pFile->Print(", ");
    string sServerParam = CCompiler::GetNameFactory()->GetServerParameterName();
    pFile->Print("void* %s", sServerParam.c_str());
}

/** \brief writes the dispatcher invocation
 *  \param pFile the file to write to
 */
void CL4BESrvLoopFunction::WriteDispatchInvocation(CBEFile *pFile)
{
    CBESrvLoopFunction::WriteDispatchInvocation(pFile);

    // set CORBA_Object depending on [dedicated_partner]. Here, the
    // environment is initialized, so it is save to use the environment's
    // partner member (independent of server parameter)
    CBEClass *pClass = GetSpecificParent<CBEClass>();
    assert(pClass);
    if (pClass->FindAttribute(ATTR_DEDICATED_PARTNER))
    {
	string sObj = GetObject()->GetDeclarator()->GetName();
	CL4BENameFactory *pNF =
	    static_cast<CL4BENameFactory*>(CCompiler::GetNameFactory());
	string sPartner = pNF->GetPartnerVariable();
	*pFile << "\tif (!l4_is_invalid_id(" << sPartner << "))\n";
	pFile->IncIndent();
	*pFile << "\t_" << sObj << " = " << sPartner << ";\n";
	pFile->DecIndent();
    }
}

