/*
 *    \file    dice/src/be/l4/L4BESndFunction.cpp
 *    \brief   contains the implementation of the class CL4BESndFunction
 *
 *    \date    06/01/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/l4/L4BESndFunction.h"
#include "be/l4/L4BENameFactory.h"
#include "be/l4/L4BEClassFactory.h"
#include "be/BEContext.h"
#include "be/BEOpcodeType.h"
#include "be/BETypedDeclarator.h"
#include "be/BEMarshaller.h"
#include "be/BEClient.h"
#include "be/BEUserDefinedType.h"
#include "be/BETrace.h"
#include "be/l4/L4BEMsgBufferType.h"
#include "be/l4/L4BESizes.h"
#include "be/l4/L4BEIPC.h"

#include "TypeSpec-Type.h"
#include "Attribute-Type.h"
#include "Compiler.h"

CL4BESndFunction::CL4BESndFunction()
{
}

/** destructs the send function class */
CL4BESndFunction::~CL4BESndFunction()
{
}

/** \brief writes the variable declaration
 *  \param pFile the file to write to

 */
void CL4BESndFunction::WriteVariableDeclaration(CBEFile * pFile)
{
    // first call base class
    CBESndFunction::WriteVariableDeclaration(pFile);

    // write result variable
    string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
    pFile->PrintIndent("l4_msgdope_t %s = { msgdope: 0 };\n", sResult.c_str());

    // tracing vars
    assert(m_pTrace);
    m_pTrace->VariableDeclaration(pFile, this);
}

/** \brief writes the invocation code
 *  \param pFile the file tow rite to
 */
void CL4BESndFunction::WriteInvocation(CBEFile * pFile)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    // after marshalling set the message dope
    pMsgBuffer->WriteInitialization(pFile, TYPE_MSGDOPE_SEND,
	GetSendDirection());
    // invocate
    if (!CCompiler::IsOptionSet(PROGRAM_NO_SEND_CANCELED_CHECK))
    {
        // sometimes it's possible to abort a call of a client.
        // but the client wants his call made, so we try until
        // the call completes
        *pFile << "\tdo\n";
        *pFile << "\t{\n";
        pFile->IncIndent();
    }
    WriteIPC(pFile);
    if (!CCompiler::IsOptionSet(PROGRAM_NO_SEND_CANCELED_CHECK))
    {
        // now check if call has been canceled
        string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
        pFile->DecIndent();
        *pFile << "\t} while ((L4_IPC_ERROR(" << sResult <<
            ") == L4_IPC_SEABORTED) ||\n";
        *pFile << "\t         (L4_IPC_ERROR(" << sResult <<
            ") == L4_IPC_SECANCELED));\n";
    }
    WriteIPCErrorCheck(pFile);
}

/** \brief writes the IPC error check
 *  \param pFile the file to write to
 *
 * \todo write IPC error checking
 */
void CL4BESndFunction::WriteIPCErrorCheck(CBEFile * pFile)
{
    string sResult = CCompiler::GetNameFactory()->GetString(STR_RESULT_VAR);
    CBEDeclarator *pDecl = m_pCorbaEnv->GetDeclarator();
    string sSetFunc;
    if (((CBEUserDefinedType*)m_pCorbaEnv->GetType())->GetName() ==
        "CORBA_Server_Environment")
        sSetFunc = "CORBA_server_exception_set";
    else
        sSetFunc = "CORBA_exception_set";

    *pFile << "\tif (L4_IPC_IS_ERROR(" << sResult << "))\n" <<
              "\t{\n";
    pFile->IncIndent();
    // env.major = CORBA_SYSTEM_EXCEPTION;
    // env.repos_id = DICE_IPC_ERROR;
    *pFile << "\t" << sSetFunc << " (";
    if (pDecl->GetStars() == 0)
        *pFile << "&";
    pDecl->WriteName(pFile);
    *pFile << ",\n";
    pFile->IncIndent();
    *pFile << "\tCORBA_SYSTEM_EXCEPTION,\n" <<
              "\tCORBA_DICE_EXCEPTION_IPC_ERROR,\n" <<
              "\t0);\n";
    pFile->DecIndent();
    // env.ipc_error = L4_IPC_ERROR(result);
    *pFile << "\t";
    pDecl->WriteName(pFile);
    if (pDecl->GetStars())
        *pFile << "->";
    else
        *pFile << ".";
    *pFile << "_p.ipc_error = L4_IPC_ERROR(" << sResult << ");\n";
    // return
    WriteReturn(pFile);
    // close }
    pFile->DecIndent();
    *pFile << "\t}\n";
}

/** \brief init message buffer size dope
 *  \param pFile the file to write to
 */
void CL4BESndFunction::WriteVariableInitialization(CBEFile * pFile)
{
    CBESndFunction::WriteVariableInitialization(pFile);
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    pMsgBuffer->WriteInitialization(pFile, TYPE_MSGDOPE_SIZE, 0);
}

/** \brief decides whether two parameters should be exchanged during sort
 *  \param pPrecessor the 1st parameter
 *  \param pSuccessor the 2nd parameter
 *  \return true if parameters pPrecessor is smaller than pSuccessor
 */
bool
CL4BESndFunction::DoExchangeParameters(CBETypedDeclarator * pPrecessor,
    CBETypedDeclarator * pSuccessor)
{
    if (!(pPrecessor->GetType()->IsOfType(TYPE_FLEXPAGE)) &&
        pSuccessor->GetType()->IsOfType(TYPE_FLEXPAGE))
        return true;
    // if first is flexpage and second is not, we prohibit an exchange
    // explicetly
    if ( pPrecessor->GetType()->IsOfType(TYPE_FLEXPAGE) &&
        !pSuccessor->GetType()->IsOfType(TYPE_FLEXPAGE))
        return false;
    if (m_pReturnVar)
    {
        CBEDeclarator *pDecl = m_pReturnVar->GetDeclarator();
	// if the 1st parameter is the return variable, we cannot exchange it,
	// because we make assumptions about its position in the message
	// buffer
        if (pPrecessor->FindDeclarator(pDecl->GetName()))
            return false;
        // if successor is return variable (should not occur) move it forward
        if (pSuccessor->FindDeclarator(pDecl->GetName()))
            return true;
    }
    return CBESndFunction::DoExchangeParameters(pPrecessor, pSuccessor);
}

/** \brief write the IPC code
 *  \param pFile the file to write to
 */
void CL4BESndFunction::WriteIPC(CBEFile *pFile)
{
    assert(m_pTrace);
    m_pTrace->BeforeCall(pFile, this);

    assert(m_pComm);
    m_pComm->WriteSend(pFile, this);
    
    m_pTrace->AfterCall(pFile, this);
}
