/**
 *    \file    dice/src/be/l4/L4BEMsgBufferType.cpp
 *    \brief   contains the implementation of the class CL4BEMsgBufferType
 *
 *    \date    02/13/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/l4/L4BEMsgBufferType.h"
#include "be/l4/L4BENameFactory.h"
#include "be/l4/L4BESizes.h"
#include "be/BEContext.h"
#include "be/BETypedDeclarator.h"
#include "be/BEFunction.h"
#include "be/BEType.h"
#include "be/BEUserDefinedType.h"
#include "be/BEDeclarator.h"
#include "be/BEAttribute.h"
#include "be/BEHeaderFile.h"

#include "fe/FETaggedStructType.h"
#include "fe/FESimpleType.h"
#include "fe/FEUserDefinedType.h"
#include "fe/FEInterface.h"
#include "fe/FEOperation.h"
#include "fe/FEExpression.h"
#include "fe/FEPrimaryExpression.h"
#include "fe/FEArrayDeclarator.h"

#include "Compiler.h"


CL4BEMsgBufferType::CL4BEMsgBufferType()
: CBEMsgBufferType()
{
    m_bCountAllVarsAsMax = false;
    m_vMaxima[0].clear();
    m_vMaxima[1].clear();
}

CL4BEMsgBufferType::CL4BEMsgBufferType(CL4BEMsgBufferType & src)
: CBEMsgBufferType(src)
{
    m_bCountAllVarsAsMax = src.m_bCountAllVarsAsMax;
    m_vMaxima[0] = src.m_vMaxima[0];
    m_vMaxima[1] = src.m_vMaxima[1];
}

/** \brief destructor of this instance */
CL4BEMsgBufferType::~CL4BEMsgBufferType()
{

}

/** \brief creates the message buffer type
 *  \param pFEInterface the respective front-end interface
 *  \param pFEDeclarator the type's name
 *  \return the new message buffer type
 *
 * This function creates, depending of the members of the interface a new
 * message buffer for that interface. This message buffer is then used as
 * parameter to wait-any and unmarshal functions of that interface.
 *
 * The L4 message buffer consists of:
 * - l4_fpage_t (a possible flexpage descriptor)
 * - l4_msgdope_t (the size of the total message structure)
 * - l4_msgdope_t (the size of the actually send message)
 * - l4_umword_t[] (an message buffer, containing direct values or flexpages)
 * - l4_strdope_t[] (string dopes, describing the strings to send)
 *
 */
CFETypeSpec *CL4BEMsgBufferType::GetMsgBufferType(CFEInterface *pFEInterface,
     CFEDeclarator* &pFEDeclarator)
{
    VERBOSE("CL4BEMsgBufferType::GetMsgBufferType(interface %s)\n",
	pFEInterface->GetName().c_str());
    vector<CFETypedDeclarator*> *pMembers = new vector<CFETypedDeclarator*>();

    // add "l4_fpage_t fpage" to type
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sTypeName = pNF->GetTypeName(TYPE_RCV_FLEXPAGE,
        true);
    CFETypeSpec *pType = new CFEUserDefinedType(sTypeName);
    string sFpageBuf = pNF->GetMessageBufferMember(TYPE_RCV_FLEXPAGE);
    CFEDeclarator *pDecl = new CFEDeclarator(DECL_IDENTIFIER, sFpageBuf);
    vector<CFEDeclarator*> *pDeclarators = new vector<CFEDeclarator*>();
    pDeclarators->push_back(pDecl);
    pDecl = 0;
    CFETypedDeclarator *pMember = new CFETypedDeclarator(TYPEDECL_FIELD, pType, pDeclarators);
    pType = 0;
    delete pDeclarators;
    pMembers->push_back(pMember);
    pMember = 0;

    // add "l4_msgdope_t size" to type
    sTypeName = pNF->GetTypeName(TYPE_MSGDOPE_SIZE,
        true);
    pType = new CFEUserDefinedType(sTypeName);
    string sSize = pNF->GetMessageBufferMember(TYPE_MSGDOPE_SIZE);
    pDecl = new CFEDeclarator(DECL_IDENTIFIER, sSize);
    pDeclarators = new vector<CFEDeclarator*>();
    pDeclarators->push_back(pDecl);
    pDecl = 0;
    pMember = new CFETypedDeclarator(TYPEDECL_FIELD, pType, pDeclarators);
    pType = 0;
    delete pDeclarators;
    pMembers->push_back(pMember);
    pMember = 0;

    // add "l4_msgdope_t send" to type
    sTypeName = pNF->GetTypeName(TYPE_MSGDOPE_SEND, true);
    pType = new CFEUserDefinedType(sTypeName);
    string sSend = pNF->GetMessageBufferMember(TYPE_MSGDOPE_SEND);
    pDecl = new CFEDeclarator(DECL_IDENTIFIER, sSend);
    pDeclarators = new vector<CFEDeclarator*>();
    pDeclarators->push_back(pDecl);
    pDecl = 0;
    pMember = new CFETypedDeclarator(TYPEDECL_FIELD, pType, pDeclarators);
    pType = 0;
    delete pDeclarators;
    pMembers->push_back(pMember);
    pMember = 0;

    CL4BESizes *pSizes = dynamic_cast<CL4BESizes*>(CCompiler::GetSizes());
    assert(pSizes);
    // we need the size for the IN and OUT parameters.
    int nSizeIn = GetCount(TYPE_FIXED, DIRECTION_IN);
    int nSizeOut = GetCount(TYPE_FIXED, DIRECTION_OUT);
    int nFlexpageIn = GetCount(TYPE_FLEXPAGE, DIRECTION_IN);
    int nFlexpageOut = GetCount(TYPE_FLEXPAGE, DIRECTION_OUT);
    int nFlexpageSize = pSizes->GetSizeOfType(TYPE_FLEXPAGE);
    if (nFlexpageIn > 0)
        nSizeIn += (nFlexpageIn+1)*nFlexpageSize;
    if (nFlexpageOut > 0)
        nSizeOut += (nFlexpageOut+1)*nFlexpageSize;
    int nSize = (nSizeIn >  nSizeOut) ? nSizeIn :  nSizeOut;
    // get minimum size
    int nMin = pSizes->GetMaxShortIPCSize(DIRECTION_IN);
    if (nSize < nMin)
        nSize = nMin;
    if ((GetCount(TYPE_VARSIZED, DIRECTION_IN) + 
	    GetCount(TYPE_VARSIZED, DIRECTION_OUT)) > 0)
    {
        nSize = 0;
        pFEDeclarator->SetStars(1);
    }
    // dword-align size
    nSize = pSizes->WordRoundUp(nSize);
    CFEExpression *pValue = new CFEPrimaryExpression(EXPR_INT, (long int) nSize);

    // add "char buffer[]" to type
    string sCharBuf = pNF->GetMessageBufferMember(TYPE_CHAR);
    pType = new CFESimpleType(TYPE_CHAR);
    pDecl = new CFEArrayDeclarator(sCharBuf, pValue);
    pValue = 0;
    pDeclarators = new vector<CFEDeclarator*>();
    pDeclarators->push_back( pDecl);
    pDecl = 0;
    pMember = new CFETypedDeclarator(TYPEDECL_FIELD, pType, pDeclarators);
    pType = 0;
    delete pDeclarators;
    pMembers->push_back(pMember);
    pMember = 0;

    // get number of strings
    int nStringIn = GetCount(TYPE_STRING, DIRECTION_IN);
    int nStringOut = GetCount(TYPE_STRING, DIRECTION_OUT);
    int nStrings = (nStringIn >  nStringOut) ? nStringIn :  nStringOut;
    if (nStrings > 0)
    {
        string sStrBuf = pNF->GetMessageBufferMember(TYPE_REFSTRING);
        pValue = new CFEPrimaryExpression(EXPR_INT, (long int) nStrings);
        pDecl = new CFEArrayDeclarator(sStrBuf, pValue);
        pValue = 0;
        pDeclarators = new vector<CFEDeclarator*>();
        pDeclarators->push_back(pDecl);
        pDecl = 0;

        sTypeName = pNF->GetTypeName(TYPE_REFSTRING, true);
        pType = new CFEUserDefinedType(sTypeName);
        pMember = new CFETypedDeclarator(TYPEDECL_FIELD, pType, pDeclarators);
        pType = 0;
        delete pDeclarators;
        pMembers->push_back(pMember);
        pMember = 0;
    }

    // add tag to be able to declare the message buffer before defining it
    string sName = pNF->GetMessageBufferTypeName(pFEInterface);
    sName = "_" + sName;
    CFETaggedStructType *pReturn = new CFETaggedStructType(sName, pMembers);
    delete pMembers;
    pReturn->SetParent(pFEInterface);
    pReturn->SetSourceLine(pFEInterface->GetSourceLine());
    return pReturn;
}

/** \brief creates the message buffer type for a specific function
 *  \param pFEOperation the repsective front-end function
 *  \param pFEDeclarator the name of the message buffer (for manipulation)
 *  \return the type of the message buffer
 */
CFETypeSpec * CL4BEMsgBufferType::GetMsgBufferType(CFEOperation * pFEOperation,
    CFEDeclarator* &pFEDeclarator)
{
    VERBOSE("CL4BEMsgBufferType::GetMsgBufferType(operation %s)\n",
	pFEOperation->GetName().c_str());
    vector<CFETypedDeclarator*> *pMembers = new vector<CFETypedDeclarator*>();

    // add "l4_fpage_t fpage" to type
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sTypeName = pNF->GetTypeName(TYPE_RCV_FLEXPAGE, true);
    CFETypeSpec *pType = new CFEUserDefinedType(sTypeName);
    string sFpageBuf = pNF->GetMessageBufferMember(TYPE_RCV_FLEXPAGE);
    CFEDeclarator *pDecl = new CFEDeclarator(DECL_IDENTIFIER, sFpageBuf);
    vector<CFEDeclarator*> *pDeclarators = new vector<CFEDeclarator*>();
    pDeclarators->push_back(pDecl);
    pDecl = 0;
    CFETypedDeclarator *pMember = new CFETypedDeclarator(TYPEDECL_FIELD, pType, pDeclarators);
    pType = 0;
    delete pDeclarators;
    pMembers->push_back(pMember);
    pMember = 0;

    // add "l4_msgdope_t size" to type
    sTypeName = pNF->GetTypeName(TYPE_MSGDOPE_SIZE, true);
    pType = new CFEUserDefinedType(sTypeName);
    string sSize = pNF->GetMessageBufferMember(TYPE_MSGDOPE_SIZE);
    pDecl = new CFEDeclarator(DECL_IDENTIFIER, sSize);
    pDeclarators = new vector<CFEDeclarator*>();
    pDeclarators->push_back(pDecl);
    pDecl = 0;
    pMember = new CFETypedDeclarator(TYPEDECL_FIELD, pType, pDeclarators);
    pType = 0;
    delete pDeclarators;
    pMembers->push_back(pMember);
    pMember = 0;

    // add "l4_msgdope_t send" to type
    sTypeName = pNF->GetTypeName(TYPE_MSGDOPE_SEND, true);
    pType = new CFEUserDefinedType(sTypeName);
    string sSend = pNF->GetMessageBufferMember(TYPE_MSGDOPE_SEND);
    pDecl = new CFEDeclarator(DECL_IDENTIFIER, sSend);
    pDeclarators = new vector<CFEDeclarator*>();
    pDeclarators->push_back(pDecl);
    pDecl = 0;
    pMember = new CFETypedDeclarator(TYPEDECL_FIELD, pType, pDeclarators);
    pType = 0;
    delete pDeclarators;
    pMembers->push_back(pMember);
    pMember = 0;

    // we need the size for the IN and OUT parameters.
    #warning size instead of count
    int nBytesIn = GetCount(TYPE_FIXED, DIRECTION_IN);
    int nBytesOut = GetCount(TYPE_FIXED, DIRECTION_OUT);
    DTRACE("%s for %s bytes-in:%d, bytes-out:%d\n", __FUNCTION__, 
	pFEOperation->GetName().c_str(), nBytesIn, nBytesOut);

    // we add the number of flexpages
    int nFlexpageSize = CCompiler::GetSizes()->GetSizeOfType(TYPE_FLEXPAGE);
    int nFlexpagesIn = GetCount(TYPE_FLEXPAGE, DIRECTION_IN);
    int nFlexpagesOut = GetCount(TYPE_FLEXPAGE, DIRECTION_OUT);
    if (nFlexpagesIn > 0)
        nBytesIn += (nFlexpagesIn+1)*nFlexpageSize;
    if (nFlexpagesOut > 0)
        nBytesOut += (nFlexpagesOut+1)*nFlexpageSize;

    int nSize = (nBytesIn >  nBytesOut) ? nBytesIn :  nBytesOut;
    // get minimum size
    CL4BESizes *pSizes = (CL4BESizes*)CCompiler::GetSizes();
    int nMin = pSizes->GetMaxShortIPCSize(DIRECTION_IN);
    if (nSize < nMin)
        nSize = nMin;
    if ((GetCount(TYPE_VARSIZED, DIRECTION_IN) +
         GetCount(TYPE_VARSIZED, DIRECTION_OUT)) > 0)
    {
        nSize = 0;
        pFEDeclarator->SetStars(1);
    }
    // dword-align size
    nSize = pSizes->WordRoundUp(nSize);
    CFEExpression *pValue = new CFEPrimaryExpression(EXPR_INT, (long int) nSize);

    // add "char buffer[]" to type
    string sCharBuf = pNF->GetMessageBufferMember(TYPE_CHAR);
    pType = new CFESimpleType(TYPE_CHAR);
    pDecl = new CFEArrayDeclarator(sCharBuf, pValue);
    pValue = 0;
    pDeclarators = new vector<CFEDeclarator*>();
    pDeclarators->push_back(pDecl);
    pDecl = 0;
    pMember = new CFETypedDeclarator(TYPEDECL_FIELD, pType, pDeclarators);
    pType = 0;
    delete pDeclarators;
    pMembers->push_back(pMember);
    pMember = 0;

    // get number of strings
    int nStringsIn = GetCount(TYPE_STRING, DIRECTION_IN);
    int nStringsOut = GetCount(TYPE_STRING, DIRECTION_OUT);
    int nStrings = (nStringsIn >  nStringsOut) ? nStringsIn :  nStringsOut;
    if ((nStrings > 0) && !IsVariableSized())
    {
        string sStrBuf = pNF->GetMessageBufferMember(TYPE_REFSTRING);
        pValue = new CFEPrimaryExpression(EXPR_INT, (long int) nStrings);
        sTypeName = pNF->GetTypeName(TYPE_REFSTRING,
            true);
        pType = new CFEUserDefinedType(sTypeName);
        pDecl = new CFEArrayDeclarator(sStrBuf, pValue);
        pValue = 0;
        pDeclarators = new vector<CFEDeclarator*>();
        pDeclarators->push_back(pDecl);
        pDecl = 0;
        pMember = new CFETypedDeclarator(TYPEDECL_FIELD, pType, pDeclarators);
        pType = 0;
        delete pDeclarators;
        pMembers->push_back(pMember);
        pMember = 0;
    }

    // add tag to be able to declare the message buffer before defining it
    string sName = pNF->GetMessageBufferTypeName(string());
    sName = "_" + sName;

    CFEStructType *pReturn = new CFETaggedStructType(sName, pMembers);
    delete pMembers;
    pReturn->SetParent(pFEOperation);
    pReturn->SetSourceLine(pFEOperation->GetSourceLine());
    return pReturn;
}

/** \brief counts the L4 specific members of the function
 *  \param pFunction the function to count

 *
 * The sizes we need are the size for the byte-buffer, which contains
 * fixed sized parameters and varaiable sized parameters, count for the
 * indirect string elements, which are variable sized parameters with
 * the ATTR_REF attribute, and the count of flexpages.
 *
 * The byte buffer has to be sized according to these rules:
 * # if only fixed sized parameters, count their size in bytes
 * # if variable sized parameters are send and _NOT_ received,
 *   use a dynamically allocated buffer on the stack
 * # if variable sized parameters are received and _NOT_ send,
 *   determine their maximum size
 * # if variable sized parameter are send _AND_ received, determine
 *   which side is larger: if received maximum is larger, then
 *   use fixed size; if send maximum is larger, then use dynamically
 *   allocated, but at least the receivers maximum
 * # indirect string are simply counted
 * # so are flexpages
 *
 * Because the maximum size of variable sized parameters has such a
 * great impact on the fixed sized parameters, we check them first.
 * To determine which direction is the send direction by using the
 * function's GetSendDirection() and GetReceiveDirection() methods.
 * Still, we store the values using the DIRECTION_IN and DIRECTION_OUT
 * constants. The caller has to be aware of this. In plain this means:
 * if the marshaller wants to determine how big the send buffer has to be
 * it first calls the functions GetSendDirection() and then using that
 * value the message buffers GetFixedSize(nDirection).
 */
void 
CL4BEMsgBufferType::InitCounts(CBEFunction * pFunction)
{
    assert(pFunction);
    // get directions
    int nSendDir = pFunction->GetSendDirection();
    int nRecvDir = pFunction->GetReceiveDirection();
    // count flexpages
    unsigned int nFlexpagesSend = 
	pFunction->GetParameterCount(TYPE_FLEXPAGE, nSendDir);
    unsigned int nFlexpagesRecv = 
	pFunction->GetParameterCount(TYPE_FLEXPAGE, nRecvDir);
    int nFlexpageSize = CCompiler::GetSizes()->GetSizeOfType(TYPE_FLEXPAGE);
    vector<struct TypeCount>::iterator iterSend, iterRecv;
    iterSend = GetCountIter(TYPE_FLEXPAGE, nSendDir);
    iterRecv = GetCountIter(TYPE_FLEXPAGE, nRecvDir);
    (*iterSend).nCount = (nFlexpagesSend >  (*iterSend).nCount) ? nFlexpagesSend :  (*iterSend).nCount;
    (*iterRecv).nCount = (nFlexpagesRecv >  (*iterRecv).nCount) ? nFlexpagesRecv :  (*iterRecv).nCount;
    // since the flexpages are counted as fixed sized parameters as well,
    // we have to substract them from the fixed size now.
    // get fixed sized parameters
    unsigned int nTempSend = pFunction->GetFixedSize(nSendDir);
    unsigned int nTempRecv = pFunction->GetFixedSize(nRecvDir);
    if (nFlexpagesSend > 0)
        nTempSend -= nFlexpageSize*nFlexpagesSend; // NO terminating zero flexpage!
    if (nFlexpagesRecv > 0)
        nTempRecv -= nFlexpageSize*nFlexpagesRecv; // NO terminating zero flexpage!
    iterSend = GetCountIter(TYPE_FIXED, nSendDir);
    iterRecv = GetCountIter(TYPE_FIXED, nRecvDir);
    (*iterSend).nCount = (nTempSend >  (*iterSend).nCount) ? nTempSend :  (*iterSend).nCount;
    (*iterRecv).nCount = (nTempRecv >  (*iterRecv).nCount) ? nTempRecv :  (*iterRecv).nCount;
    // now get var sized parameter counts
    nTempSend = pFunction->GetVariableSizedParameterCount(nSendDir);
    nTempSend += pFunction->GetStringParameterCount(nSendDir, 0, ATTR_REF); // strings without ref
    nTempRecv = pFunction->GetVariableSizedParameterCount(nRecvDir);
    nTempRecv += pFunction->GetStringParameterCount(nRecvDir, 0, ATTR_REF); // strings without ref
    // if there are variable sized parameters for send and NOT for receive
    // -> add var of send to var sized count
    if ((nTempSend > 0) && (nTempRecv == 0))
    {
        if (m_bCountAllVarsAsMax)
        {
            unsigned int nMaxSend = pFunction->GetMaxSize(nSendDir);// + pFunction->GetFixedSize(nSendDir);
            (*iterSend).nCount = (nMaxSend >  (*iterSend).nCount) ? nMaxSend :  (*iterSend).nCount;
        }
        else
        {
	    vector<struct TypeCount>::iterator iter =
		GetCountIter(TYPE_VARSIZED, nSendDir);
            (*iter).nCount = (nTempSend >  (*iter).nCount) ? nTempSend :  (*iter).nCount;
        }
    }
    // if there are variable sized parameters for recv and NOT for send
    // -> get max of receive (dont't forget "normal" fixed) and max with fixed
    // pFunction->GetMaxSize includes the normal fixed size already
    if ((nTempSend == 0) && (nTempRecv > 0))
    {
        unsigned int nMaxRecv = pFunction->GetMaxSize(nRecvDir);// + pFunction->GetFixedSize(nRecvDir);
        (*iterRecv).nCount = (nMaxRecv >  (*iterRecv).nCount) ? nMaxRecv :  (*iterRecv).nCount;
    }
    // if there are variable sized parameters for send AND recveive
    // -> add recv max to fixed AND if send max bigger, set var sized of send
    if ((nTempSend > 0) && (nTempRecv > 0))
    {
        unsigned int nMaxSend = pFunction->GetMaxSize(nSendDir);
        unsigned int nMaxRecv = pFunction->GetMaxSize(nRecvDir);
        (*iterRecv).nCount = (nMaxRecv >  (*iterRecv).nCount) ? nMaxRecv :  (*iterRecv).nCount;
        if (nMaxSend > nMaxRecv)
        {
            if (m_bCountAllVarsAsMax)
                (*iterSend).nCount = (nMaxSend >  (*iterSend).nCount) ? nMaxSend :  (*iterSend).nCount;
            else
            {
                vector<struct TypeCount>::iterator iter =
                    GetCountIter(TYPE_VARSIZED, nSendDir);
                (*iter).nCount = (nTempSend >  (*iter).nCount) ? nTempSend :  (*iter).nCount;
            }
        }
        else
            // count send max as fixed, so we do not forget this value
            (*iterSend).nCount = (nMaxSend >  (*iterSend).nCount) ? nMaxSend :  (*iterSend).nCount;
    }

    // count indirect strings (we count all parameters with the ref attribute)
    nTempSend = pFunction->GetParameterCount(ATTR_REF, 0, nSendDir);
    nTempRecv = pFunction->GetParameterCount(ATTR_REF, 0, nRecvDir);
    iterSend = GetCountIter(TYPE_STRING, nSendDir);
    iterRecv = GetCountIter(TYPE_STRING, nRecvDir);
    (*iterSend).nCount = (nTempSend >  (*iterSend).nCount) ? nTempSend :  (*iterSend).nCount;
    (*iterRecv).nCount = (nTempRecv >  (*iterRecv).nCount) ? nTempRecv :  (*iterRecv).nCount;
    // resize the maxima vector, if necessary
    if (m_vMaxima[nSendDir-1].size() < (nTempSend >  nTempRecv) ? nTempSend :  nTempRecv )
    {
        m_vMaxima[nSendDir-1].resize((nTempSend >  nTempRecv) ? nTempSend :  nTempRecv, 0);
        m_vMaxima[nRecvDir-1].resize((nTempSend >  nTempRecv) ? nTempSend :  nTempRecv, 0);
    }
    // get maxima for function
    int nCurrent = 0;
    vector<CBETypedDeclarator*>::iterator iter = pFunction->GetFirstSortedParameter();
    CBETypedDeclarator *pParameter;
    while ((pParameter = pFunction->GetNextSortedParameter(iter)) != 0)
    {
        if (!(pParameter->FindAttribute(ATTR_REF)))
            continue;
        CBEAttribute *pAttr = pParameter->FindAttribute(ATTR_MAX_IS);
        if (!pAttr)
            continue;
        nTempSend = pAttr->GetIntValue();
        if (pParameter->IsDirection(nSendDir))
            m_vMaxima[nSendDir-1][nCurrent] = 
		(nTempSend >  (unsigned int)m_vMaxima[nSendDir-1][nCurrent]) ? 
		nTempSend :  
		(unsigned int)m_vMaxima[nSendDir-1][nCurrent];
        if (pParameter->IsDirection(nRecvDir))
            m_vMaxima[nRecvDir-1][nCurrent] = 
		(nTempSend >  (unsigned int)m_vMaxima[nRecvDir-1][nCurrent]) ?
		nTempSend :
		(unsigned int)m_vMaxima[nRecvDir-1][nCurrent];
        nCurrent++;
    }

    // set reference to function
    m_pFunction = pFunction;
}

/** \brief writes the initialization code for the message buffer
 *  \param pFile the file to write to

 *
 * This initialization does mostly initialize a variable size message buffer.
 * To initialize it we first have to calculate the size using the size_is,
 * length_is or max_is attributes of the variable sized parameters. The size
 * is the maximum of the IN and the OUT buffer size (don't forget the opcode
 * size) plus the base size for size, send and fpage dopes.
 *
 * Then we allocate the message buffer on the stack using 'alloca()' (The
 * whole buffer has to be allocated on the stack, not just the byte array!)
 *
 * First we count all fixed sized values, which we obtain using GetSize() and
 * then we dword align this value and store it in the offset variable.
 *
 * The we add all variable sized parameters' sizes.
 *
 * Regard that OUT variable sized parameters need max storage amount in msg
 * buffer!!!
 */
void CL4BEMsgBufferType::WriteInitialization(CBEFile * pFile)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sOffset = pNF->GetOffsetVariable();
    string sMsgBuffer = pNF->GetMessageBufferVariable();

    if (IsVariableSized())
    {
        // use offset var for size calculation
        pFile->PrintIndent("%s = ", sOffset.c_str());
        // get size of payload
        WriteSizeOfPayload(pFile);
        // add basic size
	CBESizes *pSizes = CCompiler::GetSizes();
        int nStartOfMsgBuffer = pSizes->GetSizeOfEnvType(string("l4_fpage_t"))
                              + pSizes->GetSizeOfEnvType(string("l4_msgdope_t"))*2;
        pFile->Print("+%d;\n", nStartOfMsgBuffer);
        // dword align size
        pFile->PrintIndent("%s = %s;\n", sOffset.c_str(),
	    pSizes->WordRoundUpStr(sOffset).c_str());
        // allocate message data structure
        pFile->PrintIndent("%s = ", sMsgBuffer.c_str());
        m_pType->WriteCast(pFile, true);
        pFile->Print("_dice_alloca(%s);\n", sOffset.c_str());
    }
    // if we receive flexpages, we have to initialize the fpage member
    // this is done in the receive/wait functions explicetly. Therefore we
    // can remove it here
}

/** \brief initializes members of the message buffer
 *  \param pFile the file to write to
 *  \param nType the type of the member to initialize
 *  \param nDirection the direction of the type to initialize

 *
 * This implementation does nothing
 */
void
CL4BEMsgBufferType::WriteInitialization(CBEFile *pFile,
    unsigned int nType,
    int nDirection)
{
    switch (nType)
    {
    case TYPE_REFSTRING:
        WriteReceiveIndirectStringInitialization(pFile);
        break;
    case TYPE_RCV_FLEXPAGE:
    case TYPE_FLEXPAGE:
        WriteReceiveFlexpageInitialization(pFile, nDirection);
        break;
    case TYPE_MSGDOPE_SIZE:
        WriteSizeDopeInit(pFile);
        break;
    case TYPE_MSGDOPE_SEND:
        if (nDirection == 0)
            WriteSendDopeInit(pFile);
        else
            WriteSendDopeInit(pFile, nDirection);
        break;
    default:
        break;
    }
}

/** \brief writes the string needed to obtain the size of the byte buffer
 *  \param pFile the file to write to

 */
void CL4BEMsgBufferType::WriteSizeOfBytes(CBEFile *pFile)
{
    int nFixedIn = GetCount(TYPE_FIXED, DIRECTION_IN);
    int nFixedOut = GetCount(TYPE_FIXED, DIRECTION_OUT);
    int nBufferSize = (nFixedIn >  nFixedOut) ? nFixedIn :  nFixedOut;
    // flexpages?
    CBESizes *pSizes = CCompiler::GetSizes();
    int nFlexIn = GetCount(TYPE_FLEXPAGE, DIRECTION_IN);
    int nFlexOut = GetCount(TYPE_FLEXPAGE, DIRECTION_OUT);
    int nFlexpages = (nFlexIn >  nFlexOut) ? nFlexIn :  nFlexOut;
    if (nFlexpages > 0)
        nBufferSize += (nFlexpages+1)*pSizes->GetSizeOfType(TYPE_FLEXPAGE);
    // add space for opcode / exception
    /* if we have a reference function, test if both attributes
     * (noopcode and noexceptions) are set. If so, do not add
     * opcode size.
     */
    bool bHaveBothAttrs = false;
    if (m_pFunction &&
        (m_pFunction->FindAttribute(ATTR_NOOPCODE) &&
         m_pFunction->FindAttribute(ATTR_NOEXCEPTIONS)))
        bHaveBothAttrs = true;
    if (!bHaveBothAttrs)
    {
        int nOpcodeSize = pSizes->GetOpcodeSize();
        int nExceptionSize = pSizes->GetExceptionSize();
        nBufferSize += (nOpcodeSize > nExceptionSize) ? nOpcodeSize : 
	    nExceptionSize;
    }
    // make dword aligned
    nBufferSize = pSizes->WordRoundUp(nBufferSize);
    // declare constant size (at least opcode size)
    pFile->Print("%d", nBufferSize);
    // iterate over variable sized parameters
    WriteInitializationVarSizedParameters(pFile);
}

/** \brief writes the string needed to obtain the size of the indirect strings in bytes
 *  \param pFile the file to write to

 */
void CL4BEMsgBufferType::WriteSizeOfRefStrings(CBEFile *pFile)
{
    // get number of indirect strings
    int nStringsIn = GetCount(TYPE_STRING, DIRECTION_IN);
    int nStringsOut = GetCount(TYPE_STRING, DIRECTION_OUT);
    int nStrings = (nStringsIn >  nStringsOut) ? nStringsIn :  nStringsOut;
    if (nStrings > 0)
    {
        // add size of strings
        pFile->Print("+sizeof(l4_strdope_t)");
        if (nStrings > 1)
            pFile->Print("*%d", nStrings);
    }
}

/** \brief write the string needed to obtain the size of the payload in bytes
 *  \param pFile the file to write to
 */
void CL4BEMsgBufferType::WriteSizeOfPayload(CBEFile *pFile)
{
    // get number of indirect strings
    int nStringsIn = GetCount(TYPE_STRING, DIRECTION_IN);
    int nStringsOut = GetCount(TYPE_STRING, DIRECTION_OUT);
    int nStrings = (nStringsIn >  nStringsOut) ? nStringsIn :  nStringsOut;
    // if variable sized, we have to align the size of the dwords to the
    // size of l4_strdope_t
    if (nStrings > 0)
        pFile->Print("((");
    WriteSizeOfBytes(pFile);
    if (nStrings > 0)
    {
        // align bytes
        pFile->Print("+0xf) & ~0xf)");
    }
    // add strings
    WriteSizeOfRefStrings(pFile);
}

/** \brief initializes the flexpage member of the message buffer
 *  \param pFile the file to write to
 *  \param nDirection the direction which is receive
 *
 * If flexpage can be received by this function, we have to inialize the
 * receive window.  We will not have a TYPE_RCV_FLEXPAGE parameter, because
 * the receive window is hidden in the CORBA environment. Ths we have to find
 * flexpages and check whether we might receive some.  This is easy here,
 * because call can only be at client side, so flexpages with out count.
 *
 * Before using it, test if it is NULL. If it is we have to init the fpage
 * descriptor with a default value
 */
void CL4BEMsgBufferType::WriteReceiveFlexpageInitialization(CBEFile *pFile, int nDirection)
{
    if (GetCount(TYPE_FLEXPAGE, nDirection) == 0)
    {
	pFile->PrintIndent("");
	WriteMemberAccess(pFile, TYPE_RCV_FLEXPAGE, nDirection);
	pFile->Print(" = l4_fpage(0, 0, 0, 0);\n");
        return;
    }
    string sEnv = CCompiler::GetNameFactory()->GetCorbaEnvironmentVariable();
    // test on NULL
    pFile->PrintIndent("if (%s)\n", sEnv.c_str());
    // use env
    pFile->IncIndent();
    pFile->PrintIndent("");
    WriteMemberAccess(pFile, TYPE_RCV_FLEXPAGE, nDirection);
    pFile->Print(" = %s->rcv_fpage;\n", sEnv.c_str());
    pFile->DecIndent();
    // else use default
    pFile->PrintIndent("else\n");
    pFile->IncIndent();
    pFile->PrintIndent("");
    WriteMemberAccess(pFile, TYPE_RCV_FLEXPAGE, nDirection);
    pFile->Print(" = l4_fpage(0, L4_WHOLE_ADDRESS_SPACE, L4_FPAGE_RW, L4_FPAGE_GRANT);\n");
    pFile->DecIndent();
}

/** \brief initialize indirect strings if they can be received
 *  \param pFile the file to write to

 *
 * We have to iterate over the parameters and initialize for each the receive string.
 *
 * This function is either called by a "normal" function - the call-function or
 * the receive-any-function or the server loop function.
 *
 * \todo make receive string allocation fixed size
 * \todo if variable sized msg-buffer, where do we start to set strings (need receive sizes all params, which are not refstr)
 * \todo calculate the maximum all possible string for each indirect string and use this to allocate memory
 */
void CL4BEMsgBufferType::WriteReceiveIndirectStringInitialization(CBEFile *pFile)
{
    CBEFunction *pFunction = dynamic_cast<CBEFunction*>(GetParent());
    int nDirection = 0;
    if (pFunction)
        nDirection = pFunction->GetReceiveDirection();
    // get an array of values, which contain the maxima
    int nStrings = GetCount(TYPE_STRING, nDirection);
    int nMaxStrSize = CCompiler::GetSizes()->GetMaxSizeOfType(TYPE_CHAR);
    // get the offset variable (just in case we need it)
    string sOffset = CCompiler::GetNameFactory()->GetOffsetVariable();
    // if we have an variable sized string, we need to offset into the bytes
    // member of the message buffer. For this we need an offset. This function
    // is called after the size dope of the message has been set, which is the
    // number of dwords used for variable sized parameters.  we get the offset
    // for the cast by multiplying this number with the size of an mword
    if (IsVariableSized())
    {
        char sNumber[5];
        snprintf(sNumber, 5, "*%d",
            CCompiler::GetSizes()->GetSizeOfType(TYPE_MWORD));
        sOffset += sNumber;
    }
    else
        sOffset = "";
    // iterate over the number of indirect strings, and init them
    vector<CBETypedDeclarator*>::iterator iter;
    CBETypedDeclarator *pParameter = 0;
    if (pFunction)
        iter = pFunction->GetFirstSortedParameter();
    string sMWord = CCompiler::GetNameFactory()->GetTypeName(TYPE_MWORD, true);
    for (int i=0; i < nStrings; i++)
    {
        // get next indirect string parameter (with ATTR_REF)
        if (pFunction)
        {
            do {
                pParameter = pFunction->GetNextSortedParameter(iter);
            } while ((pParameter && !pParameter->FindAttribute(ATTR_REF)) ||
                     (pParameter && !pParameter->IsDirection(nDirection)));
        }
        // get message buffer variables
        // allocate maximum memory
        // check if we use the user-provided function
        CBEClass *pClass = 0;
        if (pFunction)
            pClass = pFunction->GetClass();
        if (CCompiler::IsOptionSet(PROGRAM_INIT_RCVSTRING) ||
            (pClass &&
            (pClass->FindAttribute(ATTR_INIT_RCVSTRING) ||
             pClass->FindAttribute(ATTR_INIT_RCVSTRING_CLIENT) ||
             pClass->FindAttribute(ATTR_INIT_RCVSTRING_SERVER))))
        {
	    // call is <function-name>(<string-number>, <pointer to string>,
	    // <pointer to size>)
	    string sFuncName;
            if (pClass &&
                (pClass->FindAttribute(ATTR_INIT_RCVSTRING) ||
                 pClass->FindAttribute(ATTR_INIT_RCVSTRING_CLIENT) ||
                 pClass->FindAttribute(ATTR_INIT_RCVSTRING_SERVER)))
            {
                CBEAttribute *pAttr = 0;
                if ((pAttr = pClass->FindAttribute(ATTR_INIT_RCVSTRING)) != NULL)
                    sFuncName = pAttr->GetString();
                if (((pAttr = pClass->FindAttribute(ATTR_INIT_RCVSTRING_CLIENT)) != NULL) &&
                    pFile->IsOfFileType(FILETYPE_CLIENT))
                    sFuncName = pAttr->GetString();
                if (((pAttr = pClass->FindAttribute(ATTR_INIT_RCVSTRING_SERVER)) != NULL) &&
                    pFile->IsOfFileType(FILETYPE_COMPONENT))
                    sFuncName = pAttr->GetString();
            }
	    CBENameFactory *pNF = CCompiler::GetNameFactory();
            if (sFuncName.empty())
                sFuncName = pNF->GetString(STR_INIT_RCVSTRING_FUNC);
            else
                sFuncName = pNF->GetString(STR_INIT_RCVSTRING_FUNC,
		    (void*)&sFuncName);
            // get env variable
            string sEnv = pNF->GetCorbaEnvironmentVariable();
            // call the init function for the indorect string
            pFile->PrintIndent("%s( %d, &(", sFuncName.c_str(), i);
            WriteMemberAccess(pFile, TYPE_REFSTRING, DIRECTION_OUT, sOffset);
            pFile->Print("[%d].rcv_str), &(", i);
            WriteMemberAccess(pFile, TYPE_REFSTRING, DIRECTION_OUT, sOffset);
            pFile->Print("[%d].rcv_size), ", i);
            // only if parent is server-loop, we have to test if the environment is a pointer
            pFile->Print("%s);\n", sEnv.c_str());
        }
        else
        {
            if (pParameter && pParameter->FindAttribute(ATTR_PREALLOC))
            {
                pFile->PrintIndent("");
                WriteMemberAccess(pFile, TYPE_REFSTRING, DIRECTION_OUT, sOffset);
                *pFile << "[" << i << "].rcv_str = (" << sMWord << ")(";
                vector<CBEDeclarator*>::iterator iterD = pParameter->GetFirstDeclarator();
                CBEDeclarator *pD = *iterD;
                int nStart = 1, nStars;
                CBEType *pType = pParameter->GetType();
                if (pType->IsPointerType())
                    nStart--;
                for (nStars=nStart; nStars<pD->GetStars(); nStars++)
                    *pFile << "*";
                *pFile << pD->GetName() << ");\n";
                // set receive size
                pFile->PrintIndent("");
                WriteMemberAccess(pFile, TYPE_REFSTRING, DIRECTION_OUT, sOffset);
                *pFile << "[" << i << "].rcv_size = ";
                if ((pParameter->FindAttribute(ATTR_SIZE_IS)) ||
                    (pParameter->FindAttribute(ATTR_LENGTH_IS)))
                {
                    if (pType->GetSize() > 1)
                        pFile->Print("(");
                    pParameter->WriteGetSize(pFile, NULL);
                    if (pType->GetSize() > 1)
                    {
                        pFile->Print(")");
                        pFile->Print("*sizeof");
                        pType->WriteCast(pFile, false);
                    }
                }
                else
                    pParameter->WriteGetSize(pFile, NULL);
                *pFile << ";\n";
            }
            else
            {
		int nStrSize = 0;
		if (!m_vMaxima[nDirection-1].empty())
		    nStrSize = m_vMaxima[nDirection-1][i];
		// only if there is none defining a maximum for this string
		// we use the string maximum
		if (nStrSize == 0)
		{
		    nStrSize = nMaxStrSize;
		    if (CCompiler::IsWarningSet(PROGRAM_WARNING_NO_MAXSIZE))
		    {
			if (pFunction)
			    CCompiler::Warning("refstring nb %d in %s has no maximum size (guessing size %d)",
				i, pFunction->GetName().c_str(), nStrSize);
			else
			    CCompiler::Warning("refstring nb %d in interface %s has no maximum size (guessing size %d)",
				i, GetSpecificParent<CBEClass>()->GetName().c_str(), nStrSize);
		    }
		}		

                pFile->PrintIndent("");
                WriteMemberAccess(pFile, TYPE_REFSTRING, DIRECTION_OUT, sOffset);
                pFile->Print("[%d].rcv_str = (%s)", i, sMWord.c_str());
                CBEContext::WriteMalloc(pFile, pFunction);
                pFile->Print("(%d);\n", nStrSize);
                // set receive size
                pFile->PrintIndent("");
                WriteMemberAccess(pFile, TYPE_REFSTRING, DIRECTION_OUT, sOffset);
                pFile->Print("[%d].rcv_size = %d;\n", i, nStrSize);
            }
        }
    }
}

/** \brief sets the receive indirect strings to zero
 *  \param pFile the file to write to

 *
 * This function is used instead of the init function to zero the memory
 * of the receive buffer.
 *
 * We assume that rcv_str points to valid memory, and rcv_size is the correct
 * size of that memory.
 */
void CL4BEMsgBufferType::WriteReceiveIndirectStringSetZero(CBEFile *pFile)
{
    CBEFunction *pFunction = dynamic_cast<CBEFunction*>(GetParent());
    int nDirection = 0;
    if (pFunction)
        nDirection = pFunction->GetReceiveDirection();
    // iterate over the number of indirect strings, and init them
    int nStrings = GetCount(TYPE_STRING, nDirection);
    for (int i=0; i < nStrings; i++)
    {
        // get message buffer variables
        pFile->PrintIndent("memset((void*)((");
        WriteMemberAccess(pFile, TYPE_REFSTRING, DIRECTION_OUT);
        pFile->Print("[%d]).rcv_str), 0, (", i);
        WriteMemberAccess(pFile, TYPE_REFSTRING, DIRECTION_OUT);
        pFile->Print("[%d]).rcv_size);\n", i);
    }
}

/** \brief fills the message buffer with zeros
 *  \param pFile the file to write to
 *  \param nType the type of the member to initialize with zero
 *  \param nDirection the direction of the message
 */
void
CL4BEMsgBufferType::WriteSetZero(CBEFile *pFile,
    unsigned int nType,
    int nDirection)
{
    switch (nType)
    {
    case TYPE_FLEXPAGE:
        WriteReceiveIndirectStringSetZero(pFile);
        break;
    default:
        break;
    }
}

/** \brief init the internal count variables
 *  \param pClass the class to count for
 *
 * We have to check the results of the counting after the functions have been
 * counted. The resons is this: this method is called for a server's message
 * buffer, which is NOT dynamically allocated, but has always to be the
 * maximum size to fit all possible messages. Therefore, we cannot use
 * variable sized parameters.
 *
 * We circumwent this by setting an internal status flag, which tells the
 * function counting to count ALL variable sized parameters with their max
 * values.
 */
void CL4BEMsgBufferType::InitCounts(CBEClass * pClass)
{
    m_bCountAllVarsAsMax = true;
    CBEMsgBufferType::InitCounts(pClass);
    m_bCountAllVarsAsMax = false;
}

/** \brief wraps the access to indirect parts
 *  \param pFile the file to write to
 *  \param nIndex the index of the indirect part to access

 *
 * Since the message buffer can be of a base type,the string member might
 * point to a location which is not the location we want. Therefore we have
 * to use the size dope of the message buffer to find the correct location
 * of the string.
 *
 * It looks like this:
 * msgbuf->_buffer[msgbuf->size.md.dwords*4+%d*sizeof(l4strdope_t)]
 */
/*void CL4BEMsgBufferType::WriteIndirectPartAccess(CBEFile* pFile, int nIndex)
{
    pFile->Print("(*(l4_strdope_t*)(&");
    WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    pFile->Print("[");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(".md.dwords*4");
    if (nIndex > 0)
        pFile->Print(" + %d*sizeof(l4_strdope_t)", nIndex);
    pFile->Print("]))");
}*/

/** \brief writes the access to the members of this type
 *  \param pFile the file to write to
 *  \param nMemberType the type of the member to access
 *  \param nDirection determines the direction of the message

 *  \param sOffset a possible offset variable if the member access has to be done using a cast of another type
 *
 * We need to implement a special case for TYPE_REFSTRING. They are not members
 * of the message buffer structure if it is variable size. Therefore we have to
 * cast the TYPE_INTEGER member to the l4_strdope_t type and use this.
 */
void
CL4BEMsgBufferType::WriteMemberAccess(CBEFile * pFile,
    int nMemberType,
    int nDirection,
    string sOffset)
{
    if ((nMemberType == TYPE_REFSTRING) && IsVariableSized())
    {
        // create l4_strdope_t
	CBEClassFactory *pCF = CCompiler::GetClassFactory();
        CBEUserDefinedType *pType = pCF->GetNewUserDefinedType();
        pType->SetParent(this);
        if (!pType->CreateBackEnd(string("l4_strdope_t")))
        {
            pFile->Print(")");
            return;
        }
        // ((l4_strdope_t*)(&(_buffer->_bytes[offset])))
        pFile->Print("(");
        // cast of l4_strdope_t type
        pType->WriteCast(pFile, true);
        // get pointer to offset in message buffer
        pFile->Print("(&");
        WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
        pFile->Print("[%s]))", sOffset.c_str());
        // done
        delete pType;
        return;
    }
    // print variable name
    string sName = CCompiler::GetNameFactory()->GetMessageBufferVariable();
    pFile->Print("%s", sName.c_str());
    CBEDeclarator *pDecl = GetAlias();
    if ((pDecl->GetStars() > 0) || IsVariableSized())
        pFile->Print("->");
    else
        pFile->Print(".");
    sName = CCompiler::GetNameFactory()->GetMessageBufferMember(nMemberType);
    pFile->Print("%s", sName.c_str());
}

/** \brief writes the access to the members of this type
 *  \param pFile the file to write to
 *  \param pParameter the parameter to access
 *  \param nDirection determines the direction of the message

 *  \param sOffset a possible offset variable if the member access has to be done using a cast of another type
 *
 * Using this function, a specific member is accessed.
 */
void CL4BEMsgBufferType::WriteMemberAccess(CBEFile * pFile,
    CBETypedDeclarator *pParameter,
    int nDirection,
    string sOffset)
{
    assert(pParameter);
    assert(pParameter->GetType());
    int nMemberType = pParameter->GetType()->GetFEType();
    if ((nMemberType == TYPE_REFSTRING) && IsVariableSized())
    {
        // create l4_strdope_t
	CBEClassFactory *pCF = CCompiler::GetClassFactory();
        CBEUserDefinedType *pType = pCF->GetNewUserDefinedType();
        pType->SetParent(this);
        if (!pType->CreateBackEnd(string("l4_strdope_t")))
        {
            pFile->Print(")");
            return;
        }
        // ((l4_strdope_t*)(&(_buffer->_bytes[offset])))
        pFile->Print("(");
        // cast of l4_strdope_t type
        pType->WriteCast(pFile, true);
        // get pointer to offset in message buffer
        pFile->Print("(&");
        WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
        pFile->Print("[%s]))", sOffset.c_str());
        // done
        delete pType;
        return;
    }
    // print variable name
    string sName = CCompiler::GetNameFactory()->GetMessageBufferVariable();
    pFile->Print("%s", sName.c_str());
    CBEDeclarator *pDecl = GetAlias();
    if ((pDecl->GetStars() > 0) || IsVariableSized())
        pFile->Print("->");
    else
        pFile->Print(".");
    sName = CCompiler::GetNameFactory()->GetMessageBufferMember(nMemberType);
    pFile->Print("%s", sName.c_str());
}

/** \brief creates a new instance of this class */
CObject * CL4BEMsgBufferType::Clone()
{
    return new CL4BEMsgBufferType(*this);
}

/** \brief writes the init string for the size dope
 *  \param pFile the file to write to

 */
void CL4BEMsgBufferType::WriteSizeDopeInit(CBEFile *pFile)
{
    // get minimum size
    CL4BESizes *pSizes = (CL4BESizes*)CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);
    int nMin = pSizes->GetMaxShortIPCSize(DIRECTION_IN);
    nMin = pSizes->WordsFromBytes(nMin);
    // calculate dwords
    int nDwordsIn = GetCount(TYPE_FIXED, DIRECTION_IN);
    int nDwordsOut = GetCount(TYPE_FIXED, DIRECTION_OUT);
    // calculate flexpages
    int nFlexpageIn = GetCount(TYPE_FLEXPAGE, DIRECTION_IN);
    int nFlexpageOut = GetCount(TYPE_FLEXPAGE, DIRECTION_OUT);
    int nFlexpageSize = CCompiler::GetSizes()->GetSizeOfType(TYPE_FLEXPAGE);
    if (nFlexpageIn > 0)
        nDwordsIn += (nFlexpageIn+1)*nFlexpageSize;
    if (nFlexpageOut > 0)
        nDwordsOut += (nFlexpageOut+1)*nFlexpageSize;
    // make dwords
    nDwordsIn = pSizes->WordsFromBytes(nDwordsIn);
    nDwordsOut = pSizes->WordsFromBytes(nDwordsOut);
    // check minimum value
    if (nDwordsIn < nMin)
        nDwordsIn = nMin;
    int nDwords = (nDwordsIn >  nDwordsOut) ? nDwordsIn :  nDwordsOut;

    // calculate strings
    int nStringsIn = GetCount(TYPE_STRING, DIRECTION_IN);
    int nStringsOut = GetCount(TYPE_STRING, DIRECTION_OUT);
    int nStrings = (nStringsIn >  nStringsOut) ? nStringsIn :  nStringsOut;

    string sOffset = CCompiler::GetNameFactory()->GetOffsetVariable();
    if (IsVariableSized())
    {
        // declare constant size
        pFile->PrintIndent("%s = ", sOffset.c_str());
        WriteSizeOfBytes(pFile);
        pFile->Print(";\n");
        // make dwords
        // if we have indirect strings, we have to set the size
        // of the dwords to all the dwords before the strings, because
        // the access to the strings start at the offset defined by the
        // dwords
        if (nStrings > 0)
        {
            int nStartOfMsgBuffer = CCompiler::GetSizes()->GetSizeOfEnvType(string("l4_fpage_t"))
                                  + CCompiler::GetSizes()->GetSizeOfEnvType(string("l4_msgdope_t"))*2;
            pFile->PrintIndent("%s = (((%s+0x%x) & ~0xf) - 0x%x) / %d;\n",
		sOffset.c_str(), sOffset.c_str(),
		nStartOfMsgBuffer+0xf, nStartOfMsgBuffer,
		nWordSize);
        }
        else
            pFile->PrintIndent("%s = %s;\n",
		sOffset.c_str(), pSizes->WordsFromBytesStr(sOffset).c_str());
    }

    // size dope
    pFile->PrintIndent("");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(" = L4_IPC_DOPE(");
    if (IsVariableSized())
        pFile->Print("%s", sOffset.c_str());
    else
        pFile->Print("%d", nDwords);
    pFile->Print(", %d);\n", nStrings);
}

/** \brief writes the init string for the send dope
 *  \param pFile the file to write to
 *  \param nSendDirection the direction for IN

 */
void 
CL4BEMsgBufferType::WriteSendDopeInit(CBEFile *pFile,
    int nSendDirection)
{
    // get minimum size
    CL4BESizes *pSizes = (CL4BESizes*)CCompiler::GetSizes();
    int nMin = pSizes->GetMaxShortIPCSize(DIRECTION_IN);
    nMin = pSizes->WordsFromBytes(nMin); 
    // check if we use offset variable
    assert(m_pFunction);
    bool bHasSizeIsParams =
        (m_pFunction->GetParameterCount(ATTR_SIZE_IS, ATTR_REF, nSendDirection) > 0) ||
        (m_pFunction->GetParameterCount(ATTR_LENGTH_IS, ATTR_REF, nSendDirection) > 0) ||
        (m_pFunction->GetParameterCount(ATTR_STRING, ATTR_REF, nSendDirection) > 0);
    bool bUseOffset = IsVariableSized(nSendDirection) || bHasSizeIsParams;
    // after marshalling set the message dope
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    if (bUseOffset)
    {
        string sOffset = pNF->GetOffsetVariable();
        // dword align size
        pFile->PrintIndent("%s = %s;\n",
                           sOffset.c_str(),
			   pSizes->WordsFromBytesStr(sOffset).c_str());
        // check minimum
        pFile->PrintIndent("%s = (%s<%d)?%d:%s;\n",
                           sOffset.c_str(),
                           sOffset.c_str(), nMin,
                           nMin, sOffset.c_str());
    }
    // send dope
    pFile->PrintIndent("");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
    pFile->Print(" = L4_IPC_DOPE(");
    if (bUseOffset)
    {
        string sOffset = pNF->GetOffsetVariable();
        pFile->Print("%s", sOffset.c_str());
    }
    else
    {
        // calculate bytes
        int nDwordsIn = GetCount(TYPE_FIXED, nSendDirection);
        if (GetCount(TYPE_FLEXPAGE, nSendDirection) > 0)
            nDwordsIn += (GetCount(TYPE_FLEXPAGE, nSendDirection) + 1) *
                pSizes->GetSizeOfType(TYPE_FLEXPAGE);
        // make dwords
        nDwordsIn = pSizes->WordsFromBytes(nDwordsIn);
        // check minimum value
        if (nDwordsIn < nMin)
            nDwordsIn = nMin;
        pFile->Print("%d", nDwordsIn);
    }
    pFile->Print(", %d);\n", GetCount(TYPE_STRING, nSendDirection));
}

/** \brief writes the init string for the send dope if nothing to send
 *  \param pFile the file to write to

 *
 * This inits the send dope to a "zero" send dope, which contains (by default)
 * 2 dwords and no ref-strings
 */
void CL4BEMsgBufferType::WriteSendDopeInit(CBEFile *pFile)
{
    // send dope
    pFile->PrintIndent("");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
    CL4BESizes *pSizes = (CL4BESizes*)CCompiler::GetSizes();
    int nSize = pSizes->GetMaxShortIPCSize(DIRECTION_IN) / 
	pSizes->GetSizeOfType(TYPE_MWORD);
    pFile->Print(" = L4_IPC_DOPE(%d, 0);\n", nSize);
}

/** \brief tests if this message buffer is suited for a short IPC
 *  \param nDirection the direction to test
 *  \param nWords the number of words allowed for a short IPC
 *  \return true if short IPC
 */
bool 
CL4BEMsgBufferType::IsShortIPC(int nDirection, 
    int nWords)
{
    if (nDirection == 0)
        return IsShortIPC(DIRECTION_IN, nWords) &&
               IsShortIPC(DIRECTION_OUT, nWords);

    if (GetCount(TYPE_VARSIZED, nDirection) > 0)
        return false;
    CL4BESizes *pSizes = (CL4BESizes*)(CCompiler::GetSizes());
    if (nWords == 0)
        nWords = pSizes->GetMaxShortIPCSize(nDirection);
    else
        nWords *= pSizes->GetSizeOfType(TYPE_MWORD);
    if (GetCount(TYPE_FIXED, nDirection) > (unsigned int)nWords)
        return false;
    if (GetCount(TYPE_STRING, nDirection) > 0)
        return false;
    if (GetCount(TYPE_FLEXPAGE, nDirection) > 0)
        return false;
    return true;
}

/** \brief uses the values of the given message buffer
 *  \param pMsgBuffer the message buffer to use the values from

 */
void CL4BEMsgBufferType::InitCounts(CBEMsgBufferType * pMsgBuffer)
{
    CBEMsgBufferType::InitCounts(pMsgBuffer);
    CL4BEMsgBufferType *pL4Buffer = dynamic_cast<CL4BEMsgBufferType*>(pMsgBuffer);
    assert(pL4Buffer);
    m_vMaxima[0] = pL4Buffer->m_vMaxima[0];
    m_vMaxima[1] = pL4Buffer->m_vMaxima[1];
}

/** \brief writes a dumper function for the message buffer
 *  \param pFile the file to write to
 *  \param sResult the result dope string (if empty the send dope is used)

 */
void CL4BEMsgBufferType::WriteDump(CBEFile *pFile, string sResult)
{
    string sFunc = CCompiler::GetTraceMsgBufFunc();
    bool bSend = sResult.empty();
    // printf fpage
    pFile->PrintIndent("%s(\"", sFunc.c_str());
    WriteMemberAccess(pFile, TYPE_RCV_FLEXPAGE, DIRECTION_IN);
    pFile->Print(".fp = {\\n\\t grant:%%x,\\n\\t write:%%x,\\n\\t size:%%x,\\n\\t zero:%%x,\\n\\t page:%%x }\\n\", ");
    WriteMemberAccess(pFile, TYPE_RCV_FLEXPAGE, DIRECTION_IN);
    pFile->Print(".fp.grant, ");
    WriteMemberAccess(pFile, TYPE_RCV_FLEXPAGE, DIRECTION_IN);
    pFile->Print(".fp.write, ");
    WriteMemberAccess(pFile, TYPE_RCV_FLEXPAGE, DIRECTION_IN);
    pFile->Print(".fp.size, ");
    WriteMemberAccess(pFile, TYPE_RCV_FLEXPAGE, DIRECTION_IN);
    pFile->Print(".fp.zero, ");
    WriteMemberAccess(pFile, TYPE_RCV_FLEXPAGE, DIRECTION_IN);
    pFile->Print(".fp.page);\n");
    // print size dope
    pFile->PrintIndent("%s(\"", sFunc.c_str());
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(".md = {\\n\\t msg_deceited:%%x,\\n\\t fpage_received:%%x,\\n\\t msg_redirected:%%x,\\n\\t "
        "src_inside:%%x,\\n\\t snd_error:%%x,\\n\\t error_code:%%x,\\n\\t strings:%%x,\\n\\t dwords:%%x }\\n\", ");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(".md.msg_deceited, ");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(".md.fpage_received, ");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(".md.msg_redirected, ");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(".md.src_inside, ");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(".md.snd_error, ");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(".md.error_code, ");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(".md.strings, ");
    WriteMemberAccess(pFile, TYPE_MSGDOPE_SIZE, DIRECTION_IN);
    pFile->Print(".md.dwords);\n");
    // print send dope
    if (bSend)
    {
        pFile->PrintIndent("%s(\"", sFunc.c_str());
        WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        pFile->Print(".md = { msg_deceited:%%x,\\n\\t fpage_received:%%x,\\n\\t msg_redirected:%%x,\\n\\t "
            "src_inside:%%x,\\n\\t snd_error:%%x,\\n\\t error_code:%%x,\\n\\t strings:%%x,\\n\\t dwords:%%x }\\n\", ");
        WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        pFile->Print(".md.msg_deceited, ");
        WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        pFile->Print(".md.fpage_received, ");
        WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        pFile->Print(".md.msg_redirected, ");
        WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        pFile->Print(".md.src_inside, ");
        WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        pFile->Print(".md.snd_error, ");
        WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        pFile->Print(".md.error_code, ");
        WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        pFile->Print(".md.strings, ");
        WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        pFile->Print(".md.dwords);\n");
    }
    else
    {
	*pFile << "\t" << sFunc << " (\"" << sResult << 
	    ".md = { msg_deceited:%%x,\\n" <<
	    "\\t fpage_received=%%x,\\n" << 
	    "\\t msg_redirected=%%x,\\n" <<
	    "\\t src_inside=%%x,\\n" <<
	    "\\t snd_error=%%x,\\n" <<
	    "\\t error_code=%%x,\\n" <<
	    "\\t strings=%%x,\\n" <<
	    "\\t dwords=%%x }\\n\", " <<
            sResult << ".md.msg_deceited, " <<
	    sResult << ".md.fpage_received, " << 
	    sResult << ".md.msg_redirected, " <<
	    sResult << ".md.src_inside, " <<
	    sResult << ".md.snd_error, " <<
	    sResult << ".md.error_code, " <<
	    sResult << ".md.strings, " <<
	    sResult << ".md.dwords);\n";
    }
    // print dwords
    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);
	
    pFile->PrintIndent("for (_i=0; _i<");
    if (CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF_DWORDS))
    {
        int nBytes = CCompiler::GetTraceMsgBufDwords()*nWordSize;
        pFile->Print("((%d <= ", nBytes);
        if (bSend)
            WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        else
            pFile->Print("%s", sResult.c_str());
        pFile->Print(".md.dwords*%d) ? %d : ", nWordSize, nBytes);
        if (bSend)
            WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        else
            pFile->Print("%s", sResult.c_str());
        pFile->Print(".md.dwords*%d)", nWordSize);
    }
    else
    {
        if (bSend)
            WriteMemberAccess(pFile, TYPE_MSGDOPE_SEND, DIRECTION_IN);
        else
            pFile->Print("%s", sResult.c_str());
        pFile->Print(".md.dwords*%d", nWordSize);
    }
    pFile->Print("; _i++)\n");
    pFile->PrintIndent("{\n");
    pFile->IncIndent();
    pFile->PrintIndent("if (_i% %d == 0)\n", nWordSize);
    pFile->IncIndent();
    pFile->PrintIndent("%s(\"dwords[%%d]:\", _i/%d);\n", sFunc.c_str(), nWordSize);
    pFile->DecIndent();
    pFile->PrintIndent("%s(\"%%02x \", ", sFunc.c_str());
    WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    pFile->Print("[_i]);\n");
    pFile->PrintIndent("if (_i% %d == %d)\n", nWordSize, nWordSize-1);
    pFile->IncIndent();
    pFile->PrintIndent("%s(\"\\n\");\n", sFunc.c_str());
    pFile->DecIndent();
    pFile->DecIndent();
    pFile->PrintIndent("}\n");
}

/** \brief writes the definition of the message buffer
 *  \param pFile the file to write to
 *  \param bTypedef true if a type definition should be written

 */
void CL4BEMsgBufferType::WriteDefinition(CBEFile* pFile,  bool bTypedef)
{
    if (bTypedef &&
        dynamic_cast<CBEHeaderFile*>(pFile) &&
        !IsVariableSized())
    {
        // get minimum size
        CL4BESizes *pSizes = (CL4BESizes*)CCompiler::GetSizes();
        int nMin = pSizes->GetMaxShortIPCSize(DIRECTION_IN);
	nMin = pSizes->WordsFromBytes(nMin);;
        // calculate dwords
        int nDwordsIn = GetCount(TYPE_FIXED, DIRECTION_IN);
        int nDwordsOut = GetCount(TYPE_FIXED, DIRECTION_OUT);
        // calculate flexpages
        int nFlexpageIn = GetCount(TYPE_FLEXPAGE, DIRECTION_IN);
        int nFlexpageOut = GetCount(TYPE_FLEXPAGE, DIRECTION_OUT);
        int nFlexpageSize = CCompiler::GetSizes()->GetSizeOfType(TYPE_FLEXPAGE);
        if (nFlexpageIn > 0)
            nDwordsIn += (nFlexpageIn+1)*nFlexpageSize;
        if (nFlexpageOut > 0)
            nDwordsOut += (nFlexpageOut+1)*nFlexpageSize;
        // make dwords from bytes
        nDwordsIn = pSizes->WordsFromBytes(nDwordsIn);
        nDwordsOut = pSizes->WordsFromBytes(nDwordsOut);
        // check minimum value
        if (nDwordsIn < nMin)
            nDwordsIn = nMin;
        int nDwords = (nDwordsIn >  nDwordsOut) ? nDwordsIn :  nDwordsOut;

        // calculate strings
	int nStringsIn = GetCount(TYPE_STRING, DIRECTION_IN);
	int nStringsOut = GetCount(TYPE_STRING, DIRECTION_OUT);
        int nStrings = (nStringsIn >  nStringsOut) ? nStringsIn :  nStringsOut;
        // get const name
        string sName = CCompiler::GetNameFactory()->GetString(STR_MSGBUF_SIZE_CONST, this);
        // write size dope constant
        pFile->Print("#ifndef %s\n", sName.c_str());
        pFile->Print("#define %s L4_IPC_DOPE(%d, %d)\n", sName.c_str(), 
	    nDwords, nStrings);
        pFile->Print("#endif /* %s */\n\n", sName.c_str());

    }
    CBEMsgBufferType::WriteDefinition(pFile, bTypedef);
}

/** \brief checks if a property is fulfilled by this message buffer
 *  \param nProperty the property to check for
 *  \param nDirection the direction to check
 *  \return true if property if fulfilled.
 */
bool CL4BEMsgBufferType::CheckProperty(int nProperty,
    int nDirection)
{
    switch (nProperty)
    {
    case MSGBUF_PROP_SHORT_IPC:
        return IsShortIPC(nDirection, 2);
        break;
    }
    return false;
}
