/**
 *    \file    dice/src/be/l4/L4BEIPC.cpp
 *    \brief   contains the implementation of the class CL4BEIPC
 *
 *    \date    02/25/2003
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/l4/L4BEIPC.h"
#include "be/l4/L4BENameFactory.h"
#include "be/l4/L4BEMsgBufferType.h"
#include "be/BEContext.h"
#include "be/BEFile.h"
#include "be/BEFunction.h"
#include "be/BEDeclarator.h"
#include "be/BESizes.h"
#include "Compiler.h"
#include "TypeSpec-Type.h"
#include "Attribute-Type.h"

CL4BEIPC::CL4BEIPC()
{
}

/** \brief destructor of target class */
CL4BEIPC::~CL4BEIPC()
{
}

/** \brief write an IPC call
 *  \param pFile the file to write to
 *  \param pFunction the function to write it for
 */
void 
CL4BEIPC::WriteCall(CBEFile *pFile, 
    CBEFunction* pFunction,
    bool bSendFlexpage, 
    bool bSendShortIPC)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sServerID = pNF->GetComponentIDVariable();
    string sResult = pNF->GetString(STR_RESULT_VAR);
    string sTimeout = pNF->GetTimeoutClientVariable();
    string sScheduling = pNF->GetScheduleClientVariable();
    string sMWord = pNF->GetTypeName(TYPE_MWORD, true);
    string sMsgBuffer = pNF->GetMessageBufferVariable();
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    int nDirection = pFunction->GetSendDirection();
    bool bScheduling = pFunction->FindAttribute(ATTR_SCHED_DECEIT); 

    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    // XXX FIXME:
    // not implemented, because X0 adaption has no 3 word bindings
    // CL4BESizes *pSizes = (CL4BESizes*)CCompiler::GetSizes();
    // bool bWord3 = (pSizes->GetMaxShortIPCSize(DIRECTION_IN) / pSizes->GetSizeOfType(TYPE_MWORD)) == 3;
    // if (bWord3)
    //   pFile->PrintIndent("l4_ipc_call_w3(*%s,\n");
    // else
    //   pFile->PrintIndent("l4_ipc_call(*%s,\n");

    pFile->PrintIndent("l4_ipc_call(*%s,\n", sServerID.c_str());
    pFile->IncIndent();
    pFile->PrintIndent("");
    if (IsShortIPC(pFunction, nDirection))
    {
        pFile->Print("L4_IPC_SHORT_MSG");
        if (bScheduling)
            *pFile << " | " << sScheduling;
    }
    else
    {
        if ((pMsgBuffer->GetCount(TYPE_FLEXPAGE, nDirection) > 0) || bScheduling)
            pFile->Print("(%s*)((%s)", sMWord.c_str(), sMWord.c_str());

        if (pMsgBuffer->HasReference())
            pFile->Print("%s", sMsgBuffer.c_str());
        else
            pFile->Print("&%s", sMsgBuffer.c_str());

        if (pMsgBuffer->GetCount(TYPE_FLEXPAGE, nDirection) > 0)
            pFile->Print("|2");
        if (bScheduling)
            *pFile << "|" << sScheduling;
        if ((pMsgBuffer->GetCount(TYPE_FLEXPAGE, nDirection) > 0) || bScheduling)
            *pFile << ")";
    }
    pFile->Print(",\n");

    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[0]))),\n");
    pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[%d]))),\n", nWordSize);

//  if (bWord3)
//     pFile->PrintIndent("*((%s*)(&(", sMWord.c_str());
//     pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER);
//     pFile->Print("[8]))),\n");

    nDirection = pFunction->GetReceiveDirection();
    if (IsShortIPC(pFunction, nDirection))
        pFile->PrintIndent("L4_IPC_SHORT_MSG,\n");
    else
    {
        if (pMsgBuffer->HasReference())
            pFile->PrintIndent("%s,\n", sMsgBuffer.c_str());
        else
            pFile->PrintIndent("&%s,\n", sMsgBuffer.c_str());
    }

    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[0])),\n");
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[%d])),\n", nWordSize);

//  if (bWord3)
//     pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
//     pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER);
//     pFile->Print("[8])),\n");

    pFile->PrintIndent("%s, &%s);\n", sTimeout.c_str(), sResult.c_str());

    pFile->DecIndent();
}

/** \brief write an IPC receive operation
 *  \param pFile the file to write to
 *  \param pFunction the function to write it for
 */
void
CL4BEIPC::WriteReceive(CBEFile* pFile,
    CBEFunction* pFunction)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sServerID = pNF->GetComponentIDVariable();
    string sResult = pNF->GetString(STR_RESULT_VAR);
    string sTimeout;
    if (pFunction->IsComponentSide())
        sTimeout = pNF->GetTimeoutServerVariable();
    else
        sTimeout = pNF->GetTimeoutClientVariable();
    string sMsgBuffer = pNF->GetMessageBufferVariable();
    string sMWord = pNF->GetTypeName(TYPE_MWORD, true);
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);

    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    pFile->PrintIndent("l4_ipc_receive(*%s,\n", sServerID.c_str());
    pFile->IncIndent();

    if (IsShortIPC(pFunction, pFunction->GetReceiveDirection()))
        pFile->PrintIndent("L4_IPC_SHORT_MSG,\n ");
    else
    {
        if (pMsgBuffer->HasReference())
            pFile->PrintIndent("%s,\n", sMsgBuffer.c_str());
        else
            pFile->PrintIndent("&%s,\n", sMsgBuffer.c_str());
    }

    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER,
	pFunction->GetReceiveDirection());
    pFile->Print("[0])),\n");
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER,
	pFunction->GetReceiveDirection());
    pFile->Print("[%d])),\n", nWordSize);

    pFile->PrintIndent("%s, &%s);\n", sTimeout.c_str(), sResult.c_str());

    pFile->DecIndent();
}

/** \brief write an IPC wait operation
 *  \param pFile the file to write to
 *  \param pFunction the function to write it for
 */
void
CL4BEIPC::WriteWait(CBEFile* pFile,
    CBEFunction *pFunction)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sServerID = pNF->GetComponentIDVariable();
    string sResult = pNF->GetString(STR_RESULT_VAR);
    string sTimeout;
    if (pFunction->IsComponentSide())
        sTimeout = pNF->GetTimeoutServerVariable();
    else
        sTimeout = pNF->GetTimeoutClientVariable();
    string sMsgBuffer = pNF->GetMessageBufferVariable();
    string sMWord = pNF->GetTypeName(TYPE_MWORD, true);
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    int nDirection = pFunction->GetReceiveDirection();
    bool bVarBuffer = pMsgBuffer->IsVariableSized(nDirection) || 
	(pMsgBuffer->GetAlias()->GetStars() > 0);

    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    pFile->PrintIndent("l4_ipc_wait(%s,\n", sServerID.c_str());
    pFile->IncIndent();
    if (IsShortIPC(pFunction, nDirection))
        pFile->PrintIndent("L4_IPC_SHORT_MSG,\n");
    else
    {
        if (bVarBuffer)
            pFile->PrintIndent("%s,\n", sMsgBuffer.c_str());
        else
            pFile->PrintIndent("&%s,\n", sMsgBuffer.c_str());
    }
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[0])),\n");
    pFile->PrintIndent("(%s*)(&(", sMWord.c_str());
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    pFile->Print("[%d])),\n", nWordSize);
    pFile->PrintIndent("%s, &%s);\n", sTimeout.c_str(), sResult.c_str());
    pFile->DecIndent();
}

/** \brief write an IPC reply and receive operation
 *  \param pFile the file to write to
 *  \param pFunction the function to write it for
 *  \param bSendFlexpage true if a flexpage should be send (false, if the message buffer should determine this)
 *  \param bSendShortIPC true if a short IPC should be send (false, if message buffer should determine this)
 */
void 
CL4BEIPC::WriteReplyAndWait(CBEFile* pFile, 
    CBEFunction* pFunction, 
    bool bSendFlexpage, 
    bool bSendShortIPC)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sResult = pNF->GetString(STR_RESULT_VAR);
    string sTimeout;
    if (pFunction->IsComponentSide())
        sTimeout = pNF->GetTimeoutServerVariable();
    else
        sTimeout = pNF->GetTimeoutClientVariable();
    string sServerID = pNF->GetComponentIDVariable();
    string sMsgBuffer = pNF->GetMessageBufferVariable();
    string sMWord = pNF->GetTypeName(TYPE_MWORD, true);
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);

    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);
    
    *pFile << "\tl4_ipc_reply_and_wait(*" << sServerID << ",\n";
    pFile->IncIndent();
    *pFile << "\t(";
    if (bSendShortIPC)
    {
        *pFile << sMWord << "*)(L4_IPC_SHORT_MSG";
        if (bSendFlexpage)
            *pFile << "|2";
    }
    else
    {
        if (bSendFlexpage)
            *pFile << sMWord << "*)((" << sMWord << ")";
        *pFile << sMsgBuffer;
        if (bSendFlexpage)
            *pFile << "|2";
    }
    pFile->Print("),\n");

    int nRcvDir = pFunction->GetReceiveDirection();
    *pFile << "\t*((" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nRcvDir);
    *pFile << "[0]))),\n";
    *pFile << "\t*((" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nRcvDir);
    *pFile << "[" << nWordSize << "]))),\n";

    *pFile << "\t" << sServerID << ",\n";
    *pFile << "\t" << sMsgBuffer << ",\n";

    *pFile << "\t(" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    *pFile << "[0])),\n";
    *pFile << "\t(" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, DIRECTION_IN);
    *pFile << "[" << nWordSize << "])),\n";

    *pFile << "\t" << sTimeout << ", &" << sResult << ");\n";

    pFile->DecIndent();
}

/** \brief write an IPC send operation
 *  \param pFile the file to write to
 *  \param pFunction the function to write it for
 */
void 
CL4BEIPC::WriteSend(CBEFile* pFile, 
    CBEFunction* pFunction)
{
    int nDirection = pFunction->GetSendDirection();
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sServerID = pNF->GetComponentIDVariable();
    string sResult = pNF->GetString(STR_RESULT_VAR);
    string sTimeout;
    if (pFunction->IsComponentSide())
        sTimeout = pNF->GetTimeoutServerVariable();
    else
        sTimeout = pNF->GetTimeoutClientVariable();
    string sMsgBuffer = pNF->GetMessageBufferVariable();
    string sMWord = pNF->GetTypeName(TYPE_MWORD, true);
    string sScheduling = pNF->GetScheduleClientVariable();
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);

    CBESizes *pSizes = CCompiler::GetSizes();
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);

    *pFile << "\tl4_ipc_send(*" << sServerID << ",\n";
    pFile->IncIndent();
    *pFile << "\t";
    bool bVarBuffer = pMsgBuffer->IsVariableSized(nDirection) ||
                     (pMsgBuffer->GetAlias()->GetStars() > 0);
    bool bScheduling = pFunction->FindAttribute(ATTR_SCHED_DECEIT); 
    /* OR further attributes */

    if (IsShortIPC(pFunction, nDirection))
    {
        *pFile << "L4_IPC_SHORT_MSG";
        if (bScheduling)
            *pFile << "|" << sScheduling;
        if (pMsgBuffer->GetCount(TYPE_FLEXPAGE, nDirection) > 0)
            *pFile << "|2";
    }
    else
    {
        if ((pMsgBuffer->GetCount(TYPE_FLEXPAGE, nDirection) > 0) || 
	    bScheduling)
            pFile->Print("(%s*)((%s)", sMWord.c_str(), sMWord.c_str());

        if (!bVarBuffer)
            *pFile << "&";
        *pFile << sMsgBuffer;

        if (pMsgBuffer->GetCount(TYPE_FLEXPAGE, nDirection) > 0)
            pFile->Print(")|2");
        if (bScheduling)
            *pFile << "|" << sScheduling;
        if ((pMsgBuffer->GetCount(TYPE_FLEXPAGE, nDirection) > 0) || 
	    bScheduling)
            *pFile << ")";
    }
    pFile->Print(",\n");

    *pFile << "\t*((" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    *pFile << "[0]))),\n";
    *pFile << "\t*((" << sMWord << "*)(&(";
    pMsgBuffer->WriteMemberAccess(pFile, TYPE_INTEGER, nDirection);
    *pFile << "[" << nWordSize << "]))),\n";

    *pFile << "\t" << sTimeout << ", &" << sResult << ");\n";

    pFile->DecIndent();
}

/** \brief write an IPC reply operation
 *  \param pFile the file to write to
 *  \param pFunction the function to write it for

 *
 * In the generic L4 case this is a send operation. We have to be careful though
 * with ASM code, which can push parameters directly into registers, since the
 * parameters for reply (exception) are not the same as for send (opcode).
 */
void CL4BEIPC::WriteReply(CBEFile* pFile, CBEFunction* pFunction)
{
    WriteSend(pFile, pFunction);
}

/** \brief determine if we should use assembler for the IPCs
 *  \param pFunction the function to write the call for

 *  \return true if assembler code should be written
 *
 * This implementation currently always returns false, because assembler code
 * is always ABI specific.
 */
bool CL4BEIPC::UseAssembler(CBEFunction *pFunction)
{
    return false;
}

/** \brief helper function to test for short IPC
 *  \param pFunction the function to test

 *  \param nDirection the direction to test
 *  \return true if the function uses short IPC in the specified direction
 *
 * This is a simple helper function, which just delegates the call to
 * the function's message buffer.
 */
bool CL4BEIPC::IsShortIPC(CBEFunction *pFunction, int nDirection)
{
    if (nDirection == 0)
        return IsShortIPC(pFunction, pFunction->GetSendDirection()) &&
               IsShortIPC(pFunction, pFunction->GetReceiveDirection());
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    return  pMsgBuffer->CheckProperty(MSGBUF_PROP_SHORT_IPC, nDirection);
}

/** \brief check if the property is fulfilled for this communication
 *  \param pFunction the function using the communication
 *  \param nProperty the property to check

 *  \return true if the property if fulfilled
 */
bool CL4BEIPC::CheckProperty(CBEFunction *pFunction, int nProperty)
{
    switch (nProperty)
    {
    case COMM_PROP_USE_ASM:
        return UseAssembler(pFunction);
        break;
    }
    return false;
}
