/**
 *    \file   dice/src/be/BEWaitAnyFunction.cpp
 *    \brief  contains the implementation of the class CBEWaitAnyFunction
 *
 *    \date   01/21/2002
 *    \author Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/BEWaitAnyFunction.h"
#include "be/BEContext.h"
#include "be/BEFile.h"
#include "be/BEOpcodeType.h"
#include "be/BETypedDeclarator.h"
#include "be/BEDeclarator.h"
#include "be/BEComponent.h"
#include "be/BEImplementationFile.h"
#include "be/BEHeaderFile.h"
#include "be/BEMsgBufferType.h"
#include "be/BEUserDefinedType.h"
#include "be/BETrace.h"
#include "Compiler.h"
#include "TypeSpec-Type.h"
#include "fe/FEInterface.h"

CBEWaitAnyFunction::CBEWaitAnyFunction(bool bOpenWait, bool bReply)
{
    m_bOpenWait = bOpenWait;
    m_bReply = bReply;
    /* reply is only allowed with open wait */
    if (bReply)
        assert(bOpenWait);
}

CBEWaitAnyFunction::CBEWaitAnyFunction(CBEWaitAnyFunction & src)
: CBEInterfaceFunction(src)
{
    m_bOpenWait = src.m_bOpenWait;
    m_bReply = src.m_bReply;
}

/** \brief destructor of target class */
CBEWaitAnyFunction::~CBEWaitAnyFunction()
{

}

/** \brief creates the wait-any function for the given interface
 *  \param pFEInterface the respective front-end interface

 *  \return true if successful
 *
 * A function which waits for any message from any sender, does return the
 * opcode of the received message and has as a parameter a reference to the
 * message buffer.
 */
bool
CBEWaitAnyFunction::CreateBackEnd(CFEInterface * pFEInterface)
{
    int nFunctionType = 0;
    if (m_bOpenWait)
    {
        if (m_bReply)
	    nFunctionType = FUNCTION_REPLY_ANY_WAIT_ANY;
        else
	    nFunctionType = FUNCTION_WAIT_ANY;
    }
    else
	nFunctionType = FUNCTION_RECV_ANY;
    // set target file name
    SetTargetFileName(pFEInterface);
    // name of the function
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sName = pNF->GetFunctionName(pFEInterface, nFunctionType);

    if (!CBEInterfaceFunction::CreateBackEnd(pFEInterface))
        return false;
    // set source line number to last number of interface
    SetSourceLine(pFEInterface->GetSourceLineEnd());

    // return type -> set to opcode
    // if return var is parameter do not delete it
    string sOpcodeVar = pNF->GetOpcodeVariable();
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    if (!SetReturnVar(pCF->GetNewOpcodeType(), sOpcodeVar))
    {
        VERBOSE("%s failed because return var could not be created\n",
            __PRETTY_FUNCTION__);
        return false;
    }
    // add parameters
    if (!AddMessageBuffer(pFEInterface))
    {
        VERBOSE("%s failed because message buffer could not be created\n",
            __PRETTY_FUNCTION__);
        return false;
    }

    return true;
}

/** \brief adds the message buffer parameter to this function
 *  \param pFEInterface the front-end interface to use as reference

 *  \return true if successful
 */
bool
CBEWaitAnyFunction::AddMessageBuffer(CFEInterface * pFEInterface)
{
    // get class's message buffer
    CBEClass *pClass = GetClass();
    assert(pClass);
    // get message buffer type
    CBEMsgBufferType *pMsgBuffer = pClass->GetMessageBuffer();
    assert(pMsgBuffer);
    // msg buffer not initialized yet
    pMsgBuffer->InitCounts(pClass);
    // create own message buffer
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    m_pMsgBuffer = pCF->GetNewMessageBufferType(true);
    m_pMsgBuffer->SetParent(this);
    if (!m_pMsgBuffer->CreateBackEnd(pMsgBuffer))
    {
        delete m_pMsgBuffer;
        m_pMsgBuffer = 0;
        VERBOSE("%s failed because message buffer could not be created\n",
            __PRETTY_FUNCTION__);
        return false;
    }
    return true;
}

/** \brief writes the variable declarations of this function
 *  \param pFile the file to write to

 *
 * No message buffer variable - its a parameter.
 */
void
CBEWaitAnyFunction::WriteVariableDeclaration(CBEFile * pFile)
{
    m_pReturnVar->WriteZeroInitDeclaration(pFile);
}

/** \brief writes the variable initializations of this function
 *  \param pFile the file to write to

 *
 * This implementation should initialize the message buffer and the pointers of
 * the out variables. We init the message buffer, because we have nothing to
 * send and want the message buffer to be in a defined state.
 */
void
CBEWaitAnyFunction::WriteVariableInitialization(CBEFile * pFile)
{
}

/** \brief writes the invocation of the message transfer
 *  \param pFile the file to write to

 *
 * This implementation calls the underlying message trasnfer mechanisms
 */
void
CBEWaitAnyFunction::WriteInvocation(CBEFile * pFile)
{
    pFile->PrintIndent("/* invoke */\n");
}

/** \brief writes the unmarshalling of the message
 *  \param pFile the file to write to
 *  \param nStartOffset the offset where to start unmarshalling
 *  \param bUseConstOffset true if a constant offset should be used, set it to \
 *         false if not possible
 *
 * This implementation should unpack the out parameters from the returned
 * message structure.
 *
 * This implementation unmarshals the "return variable", which is the opcode.
 */
void
CBEWaitAnyFunction::WriteUnmarshalling(CBEFile * pFile,
    int nStartOffset,
    bool& bUseConstOffset)
{
    assert (m_pTrace);
    bool bLocalTrace = false;
    if (!m_bTraceOn)
    {
	m_pTrace->BeforeUnmarshalling(pFile, this);
	m_bTraceOn = bLocalTrace = true;
    }
    
    WriteUnmarshalReturn(pFile, nStartOffset, bUseConstOffset);

    if (bLocalTrace)
    {
	m_pTrace->AfterUnmarshalling(pFile, this);
	m_bTraceOn = false;
    }
}

/** \brief check if this parameter is marshalled
 *  \param pParameter the parameter to marshal
 *  \return true if if is marshalled
 *
 * Always return false, because this function does not marshal any parameters
 */
bool
CBEWaitAnyFunction::DoMarshalParameter(CBETypedDeclarator * pParameter)
{
    return false;
}

/** \brief test if this function should be written
 *  \param pFile the file to write to
 *  \return true if should be written
 *
 * A wait-any function is written at the component's side.
 *
 * A receive-any function is only written if the PROGRAM_GENERATE_MESSAGE
 * option is set. Then it is created for the client's as well as the
 * component's side.
 */
bool
CBEWaitAnyFunction::DoWriteFunction(CBEHeaderFile * pFile)
{
    if (!IsTargetFile(pFile))
        return false;
    if (m_bOpenWait)
        // this test is true for m_bReply (true or false)
        return pFile->IsOfFileType(FILETYPE_COMPONENT);
    else
        return CCompiler::IsOptionSet(PROGRAM_GENERATE_MESSAGE);
}

/** \brief test if this function should be written
 *  \param pFile the file to write to
 *  \return true if should be written
 *
 * A wait-any function is written at the component's side.
 *
 * A receive-any function is only written if the PROGRAM_GENERATE_MESSAGE
 * option is set. Then it is created for the client's as well as the
 * component's side.
 */
bool
CBEWaitAnyFunction::DoWriteFunction(CBEImplementationFile * pFile)
{
    if (!IsTargetFile(pFile))
        return false;
    if (m_bOpenWait)
        // this test is true for m_bReply (true or false)
        return pFile->IsOfFileType(FILETYPE_COMPONENT);
    else
        return CCompiler::IsOptionSet(PROGRAM_GENERATE_MESSAGE);
}

/** \brief write the message buffer parameter
 *  \param pFile the file to write to

 *  \param bComma true if a comma has to be written before the parameter
 */
void
CBEWaitAnyFunction::WriteAfterParameters(CBEFile * pFile,
    bool bComma)
{
    if (bComma)
    {
        pFile->Print(",\n");
        pFile->PrintIndent("");
    }
    // write message buffer
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    WriteParameter(pFile, pMsgBuffer, false /* no const msgbuf */);
    // base class
    CBEInterfaceFunction::WriteAfterParameters(pFile, true);
}

/** \brief write the message buffer call parameter
 *  \param pFile the file to write to

 *  \param bComma true if a comma has to be written before the declarators
 */
void
CBEWaitAnyFunction::WriteCallAfterParameters(CBEFile * pFile,
    bool bComma)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (bComma)
    {
        pFile->Print(",\n");
        pFile->PrintIndent("");
    }
    if (pMsgBuffer->HasReference())
        pFile->Print("&");
    WriteCallParameter(pFile, pMsgBuffer);
    CBEInterfaceFunction::WriteCallAfterParameters(pFile, true);
}

/** \brief writes additional parameters before the parameter list
 *  \param pFile the file to print to

 *  \return true if this function wrote something
 *
 * The CORBA C mapping specifies a CORBA_object to appear as first parameter.
 */
bool
CBEWaitAnyFunction::WriteBeforeParameters(CBEFile *pFile)
{
    if (m_pCorbaObject)
    {
        WriteParameter(pFile, m_pCorbaObject, false /* no const */);
        return true;
    }
    return false;
}

/** \brief tries to find a parameter by its type
 *  \param sTypeName the name of the type
 *  \return a reference to the found parameter
 */
CBETypedDeclarator * CBEWaitAnyFunction::FindParameterType(string sTypeName)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    if (pMsgBuffer)
    {
        CBEType *pType = pMsgBuffer->GetType();
        if (dynamic_cast<CBEUserDefinedType*>(pType))
        {
            if (((CBEUserDefinedType*)pType)->GetName() == sTypeName)
                return pMsgBuffer;
        }
        if (pType->HasTag(sTypeName))
            return pMsgBuffer;
    }
    return CBEInterfaceFunction::FindParameterType(sTypeName);
}

/** \brief gets the direction for the sending data
 *  \return if at client's side DIRECTION_IN, else DIRECTION_OUT
 *
 * Since this function ignores the send part, this value should be not
 * interesting
 */
int CBEWaitAnyFunction::GetSendDirection()
{
    return IsComponentSide() ? DIRECTION_OUT : DIRECTION_IN;
}

/** \brief gets the direction for the receiving data
 *  \return if at client's side DIRECTION_OUT, else DIRECTION_IN
 */
int CBEWaitAnyFunction::GetReceiveDirection()
{
    return IsComponentSide() ? DIRECTION_IN : DIRECTION_OUT;
}
