/**
 *    \file    dice/src/be/BEUserDefinedType.cpp
 *    \brief    contains the implementation of the class CBEUserDefinedType
 *
 *    \date    02/13/2002
 *    \author    Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/BEUserDefinedType.h"
#include "be/BEContext.h"
#include "be/BERoot.h"
#include "be/BETypedef.h"
#include "be/BEDeclarator.h"
#include "be/BEExpression.h"
#include "be/BESizes.h"

#include "Compiler.h"
#include "TypeSpec-Type.h"
#include "fe/FEUserDefinedType.h"
#include "fe/FEFile.h"

CBEUserDefinedType::CBEUserDefinedType()
{
}

CBEUserDefinedType::CBEUserDefinedType(CBEUserDefinedType & src)
 : CBEType(src)
{
    m_sOriginalName = src.m_sOriginalName;
}

/** \brief destructor of this instance */
CBEUserDefinedType::~CBEUserDefinedType()
{

}

/** \brief creates a user defined type
 *  \param sName the name of the type

 *  \return true if successful
 */
bool CBEUserDefinedType::CreateBackEnd(string sName)
{
    if (sName.empty())
    {
        VERBOSE("%s failed because user defined name is empty\n",
	    __PRETTY_FUNCTION__);
        return false;
    }
    m_sName = sName;
    m_sOriginalName = sName;
    // set size
    m_nSize = GetSizeOfTypedef(sName);
    // check if size could be found; if not test environment types
    if (m_nSize == 0)
    {
        CBESizes *pSizes = CCompiler::GetSizes();
        m_nSize = pSizes->GetSizeOfEnvType(sName);
    }
    // set type
    m_nFEType = TYPE_USER_DEFINED;
    return true;
}

/** \brief creates a user defined type
 *  \param pFEType the front-end type

 *  \return true if successful
 *
 * This overloads the CBEType::CreateBE function to implement user-defined type specific behaviour.
 */
bool CBEUserDefinedType::CreateBackEnd(CFETypeSpec * pFEType)
{
    // call CBEObject's CreateBackEnd method
    if (!CBEObject::CreateBackEnd(pFEType))
        return false;

    if (!pFEType)
    {
        VERBOSE("%s failed because FE Type is 0\n", __PRETTY_FUNCTION__);
        return false;
    }

    if (!dynamic_cast<CFEUserDefinedType*>(pFEType))
    {
        VERBOSE("%s failed because FE Type is not 'user defined'\n", __PRETTY_FUNCTION__);
        return false;
    }

    CFEUserDefinedType *pUserType = (CFEUserDefinedType *) pFEType;
    if (!pUserType->GetName().empty())
    {
        // find original type
        CFEFile *pFERoot = dynamic_cast<CFEFile*>(pFEType->GetRoot());
        assert(pFERoot);
        string sName, sUserName = pUserType->GetName();
        CFETypedDeclarator *pFETypedef = pFERoot->FindUserDefinedType(sUserName);
        if (pFETypedef)
            sName = CCompiler::GetNameFactory()->GetTypeName((CFEBase*)pFETypedef, sUserName);
        else
        {
            CFEInterface *pFEInterface = pFERoot->FindInterface(sUserName);
            if (pFEInterface)
                sName = CCompiler::GetNameFactory()->GetTypeName((CFEBase*)pFEInterface, sUserName);
        }
        if (sName.empty())
            sName = sUserName;
        if (!CreateBackEnd(sName))
        {
            VERBOSE("%s failed because no name set\n", __PRETTY_FUNCTION__);
            return false;
        }
        // reset original name
        m_sOriginalName = sUserName;
    }
    m_nFEType = TYPE_USER_DEFINED;
    return true;
}

/** \brief finds the size of a user defined type
 *  \param sTypeName the name of the type, we are looking for
 *  \return the size of the type in bytes
 *
 * This implementation finds the root of the tree and then searches downward
 * for the type.
 */
int CBEUserDefinedType::GetSizeOfTypedef(string sTypeName)
{
    CBERoot *pRoot = GetSpecificParent<CBERoot>();
    assert(pRoot);
    CBETypedef *pTypedef = pRoot->FindTypedef(sTypeName);
    if (!pTypedef)
        return 0;
    /* since the typedef is a CBETypedDeclarator, it would evaluate the size
       of it's base type and sum it for all it's declarators. We only want it
       for the declarator we are using. That's why we use a specific
       GetSize function instead of the generic one. */
    int nSize = pTypedef->GetSize(sTypeName);
    return nSize;
}

/** \brief generates an exact copy of this class
 *  \return a reference to the new object
 */
CObject *CBEUserDefinedType::Clone()
{
    return new CBEUserDefinedType(*this);
}

/** \brief get the user defined name
 *  \return a reference to the name
 */
string CBEUserDefinedType::GetName()
{
    return m_sName;
}

/** \brief calculate the size of thise type
 *  \return the size in bytes.
 *
 * The size of the user defined type can be calculated by searching for the
 * typedef with this name and calling its type GetSize() function.
 */
int CBEUserDefinedType::GetSize()
{
    if (m_nSize == 0)
        m_nSize = GetSizeOfTypedef(m_sName);
    /* if it is still zero, use original name */
    if (m_nSize == 0)
        m_nSize = GetSizeOfTypedef(m_sOriginalName);
    return m_nSize;
}


/** \brief writes cod eto initialize a variable of this type with a zero value
 *  \param pFile the file to write to

 *
 * To initialize a user defined type with zero values, means to find the typedef and use its type
 * to write this initialization.
 */
void CBEUserDefinedType::WriteZeroInit(CBEFile * pFile)
{
    CBEType *pType = GetRealType();
    if (pType)
    {
        // test for array types, something like: typedef int five_ints[5];
        CBEDeclarator *pAlias = GetRealName();
        if (pAlias && pAlias->IsArray())
        {
            WriteZeroInitArray(pFile, pType, pAlias, pAlias->GetFirstArrayBound());
            return;
        }
        // no array
        pType->WriteZeroInit(pFile);
        return;
    }
    CBEType::WriteZeroInit(pFile);
}

/** \brief checks if this is a constructed type
 *  \return true if it is
 *
 * This call is redirected to the "real" type
 */
bool CBEUserDefinedType::IsConstructedType()
{
    CBEType *pType = GetRealType();
    if (pType)
        return pType->IsConstructedType();
    return false;
}

/** \brief determines if this function writes zero init code
 *  \return the delegate result
 */
bool CBEUserDefinedType::DoWriteZeroInit()
{
    CBEType *pType = GetRealType();
    if (pType)
        return pType->DoWriteZeroInit();;
    return CBEType::DoWriteZeroInit();
}

/** \brief calls the WriteGetSize function of the original type
 *  \param pFile the file to write to
 *  \param pStack contains the declarator stack of constructed typed var-sized parameters

 */
void CBEUserDefinedType::WriteGetSize(CBEFile *pFile,
    vector<CDeclaratorStackLocation*> *pStack)
{
    CBEType *pType = GetRealType();
    CBEDeclarator *pAlias = GetRealName();
    // if the type is simple, but the alias is variable sized
    // then we need to the get max-size of the type
    // e.g. typedef small *small_var_array;
    if (pType && pType->IsSimpleType() &&
        pAlias && (pAlias->GetStars() > 0))
    {
        int nMaxSize = CCompiler::GetSizes()->GetMaxSizeOfType(pType->GetFEType());
        pFile->Print("%d", nMaxSize);
    }
    else
    {
        if (pType)
            pType->WriteGetSize(pFile, pStack);
    }
}

/** \brief test if this is a simple type
 *  \return false
 */
bool CBEUserDefinedType::IsSimpleType()
{
    return false;
}

/** \brief checks if this type has array dimensions
 *  \return true if it has
 */
bool CBEUserDefinedType::IsArrayType()
{
    CBEDeclarator *pAlias = GetRealName();
    // if the type is simple, but the alias is variable sized
    // then we have at least one array dimensions
    if (pAlias && (pAlias->GetStars() > 0))
        return true;
    if (pAlias && (pAlias->GetArrayDimensionCount() > 0))
        return true;
    // check base type
    CBEType *pType = GetRealType();
    if (pType)
        return pType->IsArrayType();
    return false;
}

/** \brief calculates the number of array dimensions for this type
 *  \return the number of found array dimensions
 */
int CBEUserDefinedType::GetArrayDimensionCount()
{
    CBEType *pType = GetRealType();
    CBEDeclarator *pAlias = GetRealName();
    // if the type is simple, but the alias is variable sized
    // then we have at least one array dimensions
    if (pType && pAlias)
        return pType->GetArrayDimensionCount() + pAlias->GetArrayDimensionCount();
    return 0;
}

/** \brief test if this type is a pointer type
 *  \return true if it is
 *
 * The CORBA_Object type is a pointer type
 */
bool CBEUserDefinedType::IsPointerType()
{
    CBEDeclarator *pAlias = GetRealName();
    // if the type is simple, but the alias is variable sized
    // then we have at least one array dimensions
    if (pAlias && (pAlias->GetStars() > 0))
        return true;
    // check the aliased type
    CBEType *pType = GetRealType();
    if (pType)
        return pType->IsPointerType();
    // no alias; fallback to base type
    return CBEType::IsPointerType();
}

/** \brief if this is the alias of another type, get this type
 *  \return the type of the typedef
 */
CBEType* CBEUserDefinedType::GetRealType()
{
    CBERoot *pRoot = GetSpecificParent<CBERoot>();
    assert(pRoot);
    CBETypedef *pTypedef = pRoot->FindTypedef(m_sName);
    if (pTypedef)
        return pTypedef->GetType();
    return 0;
}

/** \brief if this is the alias of another type, get the alias of that type
 *  \return the alias of the typedef
 */
CBEDeclarator* CBEUserDefinedType::GetRealName()
{
    CBERoot *pRoot = GetSpecificParent<CBERoot>();
    assert(pRoot);
    CBETypedef *pTypedef = pRoot->FindTypedef(m_sName);
    if (pTypedef)
        return pTypedef->GetAlias();
    return 0;
}

/** \brief writes the type without indirections
 *  \param pFile the file to write to

 */
void CBEUserDefinedType::WriteIndirect(CBEFile* pFile)
{
    if (IsPointerType())
    {
        CBEType *pType = GetRealType();
        if (pType)
        {
            pType->WriteIndirect(pFile);
            return;
        }
    }
    CBEType::WriteIndirect(pFile);
}

/** \brief returns the number of indirections
 *  \return the number of stars in the typedef alias
 */
int CBEUserDefinedType::GetIndirectionCount()
{
    if (IsPointerType())
    {
        int nIndirect = 0;
        CBEType *pType = GetRealType();
        if (pType)
        {
            CBEDeclarator *pAlias = GetRealName();
            if (pAlias)
                nIndirect = pAlias->GetStars();
            return nIndirect + pType->GetIndirectionCount();
        }
    }
    return CBEType::GetIndirectionCount();
}
