/**
 *    \file    dice/src/be/BEUnmarshalFunction.cpp
 *    \brief   contains the implementation of the class CBEUnmarshalFunction
 *
 *    \date    01/20/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/BEUnmarshalFunction.h"
#include "be/BEContext.h"
#include "be/BEComponent.h"
#include "be/BEFile.h"
#include "be/BEType.h"
#include "be/BEOpcodeType.h"
#include "be/BETypedDeclarator.h"
#include "be/BEDeclarator.h"
#include "be/BEAttribute.h"
#include "be/BEClient.h"
#include "be/BEComponent.h"
#include "be/BEImplementationFile.h"
#include "be/BEHeaderFile.h"
#include "be/BEMsgBufferType.h"
#include "be/BEUserDefinedType.h"
#include "be/BETrace.h"
#include "Compiler.h"
#include "TypeSpec-Type.h"
#include "fe/FEInterface.h"
#include "fe/FEOperation.h"
#include "fe/FETypedDeclarator.h"

CBEUnmarshalFunction::CBEUnmarshalFunction()
{
}

CBEUnmarshalFunction::CBEUnmarshalFunction(CBEUnmarshalFunction & src)
 : CBEOperationFunction(src)
{
}

/** \brief destructor of target class */
CBEUnmarshalFunction::~CBEUnmarshalFunction()
{

}

/** \brief creates the back-end unmarshal function
 *  \param pFEOperation the corresponding front-end operation

 *  \return true if successful
 *
 * This function should only contain IN parameters if it is on the component's side an
 * OUT parameters if it is on the client's side.
 */
bool CBEUnmarshalFunction::CreateBackEnd(CFEOperation * pFEOperation)
{
    // set target file name
    SetTargetFileName(pFEOperation);
    // set name
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sName = pNF->GetFunctionName(pFEOperation, FUNCTION_UNMARSHAL);

    if (!CBEOperationFunction::CreateBackEnd(pFEOperation))
    {
        VERBOSE("%s failed because base function could not be created\n", __PRETTY_FUNCTION__);
        return false;
    }

    // set return type
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    if (IsComponentSide())
    {
        CBEType *pReturnType = pCF->GetNewType(TYPE_VOID);
        pReturnType->SetParent(this);
        if (!pReturnType->CreateBackEnd(false, 0, TYPE_VOID))
        {
            VERBOSE("%s failed because return var could not be created\n", __PRETTY_FUNCTION__);
            delete pReturnType;
            return false;
        }
        CBEType *pOldType = m_pReturnVar->ReplaceType(pReturnType);
        delete pOldType;
    }
    // add parameters
    if (!AddMessageBuffer(pFEOperation->GetSpecificParent<CFEInterface>()))
    {
        VERBOSE("%s failed because message buffer could not be created\n", __PRETTY_FUNCTION__);
        return false;
    }

    return true;
}

/** \brief writes the variable declarations of this function
 *  \param pFile the file to write to

 *
 * The variable declarations contains the return variable if needed. And a temporary variable if
 * we have any variable sized arrays. No message buffer definition (its an parameter).
 */
void CBEUnmarshalFunction::WriteVariableDeclaration(CBEFile * pFile)
{
    m_pReturnVar->WriteZeroInitDeclaration(pFile);
    // check for temp
    if (HasVariableSizedParameters(GetReceiveDirection()) ||
        HasArrayParameters(GetReceiveDirection()))
    {
        string sTmpVar = CCompiler::GetNameFactory()->GetTempOffsetVariable();
        string sOffsetVar = CCompiler::GetNameFactory()->GetOffsetVariable();
        pFile->PrintIndent("unsigned %s __attribute__ ((unused));\n", sTmpVar.c_str());
        pFile->PrintIndent("unsigned %s __attribute__ ((unused));\n", sOffsetVar.c_str());
    }
}

/** \brief writes the variable initializations of this function
 *  \param pFile the file to write to

 *
 * This implementation should initialize the pointers of the out variables. Do not
 * initialize the message buffer - this may overwrite the values we try to unmarshal.
 */
void CBEUnmarshalFunction::WriteVariableInitialization(CBEFile * pFile)
{
}

/** \brief writes the invocation of the message transfer
 *  \param pFile the file to write to

 *
 * This implementation does nothing, because the unmarshalling does not contain a
 * message transfer.
 */
void CBEUnmarshalFunction::WriteInvocation(CBEFile * pFile)
{

}

/** \brief writes the unmarshalling of the message
 *  \param pFile the file to write to
 *  \param nStartOffset the offset where to start unmarshalling
 *  \param bUseConstOffset true if a constant offset should be used, set \
 *    it to false if not possible
 *
 * This implementation iterates over the parameters (except the message buffer
 * itself) and unmarshals them.
 */
void
CBEUnmarshalFunction::WriteUnmarshalling(CBEFile * pFile, int nStartOffset,
    bool& bUseConstOffset)
{
    assert(m_pTrace);
    bool bLocalTrace = false;
    if (!m_bTraceOn)
    {
	m_pTrace->BeforeUnmarshalling(pFile, this);
	m_bTraceOn = bLocalTrace = true;
    }
    
    if (IsComponentSide())
    {
        if (!FindAttribute(ATTR_NOOPCODE))
        {
            // start after opcode
	    CBEClassFactory *pCF = CCompiler::GetClassFactory();
            CBEOpcodeType *pOpcodeType = pCF->GetNewOpcodeType();
            pOpcodeType->SetParent(this);
            if (pOpcodeType->CreateBackEnd())
                nStartOffset += pOpcodeType->GetSize();
            delete pOpcodeType;
        }
    }
    else
    {
        // unmarshal exception
        nStartOffset += WriteUnmarshalException(pFile, nStartOffset, 
	    bUseConstOffset);
        // first unmarshl return value
        nStartOffset += WriteUnmarshalReturn(pFile, nStartOffset,
	    bUseConstOffset);
    }
    // now unmarshal rest
    CBEOperationFunction::WriteUnmarshalling(pFile, nStartOffset,
	bUseConstOffset);

    if (bLocalTrace)
    {
	m_pTrace->AfterUnmarshalling(pFile, this);
	m_bTraceOn = false;
    }
}

/** \brief adds a single parameter to this function
 *  \param pFEParameter the parameter to add

 *  \return true if successful
 *
 * This function decides, which parameters to add and which don't. The parameters to unmarshal are
 * for client-to-component transfer the IN parameters and for component-to-client transfer the OUT
 * and return parameters. We depend on the information set in m_bComponentSide.
 */
bool CBEUnmarshalFunction::AddParameter(CFETypedDeclarator * pFEParameter)
{
    if (IsComponentSide())
    {
        if (!(pFEParameter->FindAttribute(ATTR_IN)))
            return true;
    }
    else
    {
        if (!(pFEParameter->FindAttribute(ATTR_OUT)))
            return true;
    }
    return CBEOperationFunction::AddParameter(pFEParameter);
}

/** \brief checks if this parameter should be marshalled or not
 *  \param pParameter the parameter to check

 *  \return true if this parameter should be marshalled
 *
 * Always return false, because this function does only unmarshal parameters
 */
bool
CBEUnmarshalFunction::DoMarshalParameter(CBETypedDeclarator * pParameter)
{
    return false;
}

/** \brief checks if this parameter should be unmarshalled or not
 *  \param pParameter the parameter to check

 *  \return true if this parameter should be unmarshalled
 */
bool
CBEUnmarshalFunction::DoUnmarshalParameter(CBETypedDeclarator * pParameter)
{
    if (IsComponentSide())
    {
        if (pParameter->FindAttribute(ATTR_IN))
            return true;
    }
    else
    {
        if (pParameter->FindAttribute(ATTR_OUT))
            return true;
    }
    return false;
}

/** \brief checks whether a given parameter needs an additional reference pointer
 *  \param pDeclarator the decl to check

 *  \param bCall true if the parameter is a call parameter
 *  \return true if we need a reference
 *
 * This implementation checks for the special condition to give an extra
 * reference.  Since the declarator may also belong to an attribute, we have
 * to check this as well.  (The size_is declarator belongs to a parameter, but
 * has to be checked as well).
 *
 * Another possibility is, that members of structures or union are checked. To
 * avoid giving them unwanted references, we search for the parameter.
 *
 * An additional reference is also given to the [in, string] parameters.
 *
 * (The message buffer parameter needs no additional reference, it is itself a
 * pointer.)
 *
 *
 * Because every parameter is set inside this functions, all parameters should
 * be referenced.  Since OUT parameters are already referenced, we only need
 * to add an asterisk to IN parameters.  This function is only called when
 * iterating over the declarators of a typed declarator. Thus the direction of
 * a parameter can be found out by checking the attributes of the declarator's
 * parent.  To be sure, whether this parameter needs an additional star, we
 * check the existing number of stars.
 *
 * We also need to add an asterisk to the message buffer parameter.
 */
bool 
CBEUnmarshalFunction::HasAdditionalReference(CBEDeclarator * pDeclarator,
    bool bCall)
{
    CBETypedDeclarator *pParameter = GetParameter(pDeclarator, bCall);
    if (!pParameter)
        return false;
    assert(dynamic_cast<CBETypedDeclarator*>(pParameter));
    if (pParameter->FindAttribute(ATTR_IN))
    {
        CBEType *pType = pParameter->GetType();
        CBEAttribute *pAttr;
        if ((pAttr = pParameter->FindAttribute(ATTR_TRANSMIT_AS)) != 0)
            pType = pAttr->GetAttrType();
        int nArrayDimensions = pDeclarator->GetArrayDimensionCount() - pType->GetArrayDimensionCount();
        if ((pDeclarator->GetStars() == 0) && (nArrayDimensions <= 0))
            return true;
        if ((pParameter->FindAttribute(ATTR_STRING)) &&
             pType->IsOfType(TYPE_CHAR) &&
            (pDeclarator->GetStars() < 2))
            return true;
        if ((pParameter->FindAttribute(ATTR_SIZE_IS) ||
            pParameter->FindAttribute(ATTR_LENGTH_IS) ||
            pParameter->FindAttribute(ATTR_MAX_IS)) &&
            (nArrayDimensions <= 0))
            return true;
    }
    return CBEOperationFunction::HasAdditionalReference(pDeclarator, bCall);
}

/** \brief test if this function should be written
 *  \param pFile the file to write to
 *  \return true if should be written
 *
 * An unmarshal function is written if client's side and OUT or if component's
 * side and one of the parameters has an IN.
 */
bool CBEUnmarshalFunction::DoWriteFunction(CBEHeaderFile * pFile)
{
    if (!IsTargetFile(pFile))
        return false;
    if (pFile->IsOfFileType(FILETYPE_CLIENT) &&
        (FindAttribute(ATTR_OUT)))
        return true;
    if (dynamic_cast<CBEComponent*>(pFile->GetTarget()))
    {
        vector<CBETypedDeclarator*>::iterator iter = GetFirstParameter();
        CBETypedDeclarator *pParameter;
        while ((pParameter = GetNextParameter(iter)) != 0)
        {
            if (pParameter->FindAttribute(ATTR_IN))
                return true;
        }
    }
    return false;
}

/** \brief test if this function should be written
 *  \param pFile the file to write to
 *  \return true if should be written
 *
 * An unmarshal function is written if client's side and OUT or if component's
 * side and one of the parameters has an IN.
 */
bool CBEUnmarshalFunction::DoWriteFunction(CBEImplementationFile * pFile)
{
    if (!IsTargetFile(pFile))
        return false;
    if (pFile->IsOfFileType(FILETYPE_CLIENT) &&
        (FindAttribute(ATTR_OUT)))
        return true;
    if (dynamic_cast<CBEComponent*>(pFile->GetTarget()))
    {
        vector<CBETypedDeclarator*>::iterator iter = GetFirstParameter();
        CBETypedDeclarator *pParameter;
        while ((pParameter = GetNextParameter(iter)) != 0)
        {
            if (pParameter->FindAttribute(ATTR_IN))
                return true;
        }
    }
    return false;
}

/** \brief writes the message buffer parameter
 *  \param pFile the file to write to

 *  \param bComma true if a comma has to be written before the parameter
 */
void CBEUnmarshalFunction::WriteAfterParameters(CBEFile * pFile, bool bComma)
{
    if (bComma)
    {
        pFile->Print(",\n");
        pFile->PrintIndent("");
    }
    // write message buffer
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    WriteParameter(pFile, pMsgBuffer);
    // base class
    CBEOperationFunction::WriteAfterParameters(pFile, true);
}

/** \brief writes the message buffer call parameter
 *  \param pFile the file to write to

 *  \param bComma true if a comma has to be written before the declarators
 *
 * This is also where the environment variable is written. If the server has a
 * parameter of type Corba-Environment, it is a pointer in the server loop and
 * when calling the unmarshal function, needs no reference making '&'.
 */
void CBEUnmarshalFunction::WriteCallAfterParameters(CBEFile * pFile, bool bComma)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (bComma)
    {
        pFile->Print(",\n");
        pFile->PrintIndent("");
    }
    /* this should be unset if the message buffer type stays the
     * same for derived interfaces (e.g. char[] for socket Backend)
     */
    if (m_bCastMsgBufferOnCall)
        pMsgBuffer->GetType()->WriteCast(pFile, true);
    WriteCallParameter(pFile, pMsgBuffer);
    CBEOperationFunction::WriteCallAfterParameters(pFile, true);
}

/** \brief tries to find a parameter by its type
 *  \param sTypeName the name of the type
 *  \return a reference to the found parameter
 */
CBETypedDeclarator * CBEUnmarshalFunction::FindParameterType(string sTypeName)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    if (pMsgBuffer)
    {
        CBEType *pType = pMsgBuffer->GetType();
        if (dynamic_cast<CBEUserDefinedType*>(pType))
        {
            if (((CBEUserDefinedType*)pType)->GetName() == sTypeName)
                return pMsgBuffer;
        }
        if (pType->HasTag(sTypeName))
            return pMsgBuffer;
    }
    return CBEOperationFunction::FindParameterType(sTypeName);
}

/** \brief adds the specific message buffer parameter for this function
 *  \param pFEInterface the respective front-end interface to use as reference

 *  \return true if the create process was successful
 *
 * Instead of creating a whole new message buffer type, we use the existing type
 * of the class as a user defined type.
 */
bool CBEUnmarshalFunction::AddMessageBuffer(CFEInterface * pFEInterface)
{
    // get class's message buffer
    CBEClass *pClass = GetClass();
    assert(pClass);
    // get message buffer type
    CBEMsgBufferType *pMsgBuffer = pClass->GetMessageBuffer();
    assert(pMsgBuffer);
    // we have to sort parameters before the message buffer extracts the sizes
    SortParameters(0);
    // msg buffer not yet initialized
    pMsgBuffer->InitCounts(pClass);
    // create own message buffer
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    m_pMsgBuffer = pCF->GetNewMessageBufferType(true);
    m_pMsgBuffer->SetParent(this);
    if (!m_pMsgBuffer->CreateBackEnd(pMsgBuffer))
    {
        delete m_pMsgBuffer;
        m_pMsgBuffer = 0;
        VERBOSE("%s failed because message buffer could not be created\n", __PRETTY_FUNCTION__);
        return false;
    }
    return true;
}

/** \brief gets the direction, which the marshal-parameters have
 *  \return if at client's side DIRECTION_IN, if at server's side DIRECTION_OUT
 *
 * Since this function ignores marshalling parameter this value should be irrelevant
 */
int CBEUnmarshalFunction::GetSendDirection()
{
    return IsComponentSide() ? DIRECTION_OUT : DIRECTION_IN;
}

/** \brief gets the direction of the unmarshal-parameters
 *  \return if at client's side DIRECTION_OUT, if at server's side DIRECTION_IN
 */
int CBEUnmarshalFunction::GetReceiveDirection()
{
    return IsComponentSide() ? DIRECTION_IN : DIRECTION_OUT;
}
