/**
 *    \file    dice/src/be/BETrace.cpp
 *    \brief   contains the implementation of the class CBETrace
 *
 *    \date    12/05/2005
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2005
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "BETrace.h"
#include "BEContext.h"
#include "BEFunction.h"
#include "Compiler.h"
#include "BEMsgBufferType.h"

CBETrace::CBETrace()
{
}

CBETrace::~CBETrace()
{
}

/** \brief writes necessary variable declarations
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::VariableDeclaration(CBEFile *pFile,
    CBEFunction *pFunction)
{
    // write loop variable for msg buffer dump
    if (CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF))
        *pFile << "\tint _i;\n";
}

/** \brief write the code required to setup at server
 *  \param pFile the file to write to
 *  \param pFunction the server loop function calling this hook
 */
void
CBETrace::InitServer(CBEFile *pFile,
    CBEFunction *pFunction)
{
}

/** \brief writes default includes required for tracing
 *  \param pFile the file to write to
 */
void
CBETrace::DefaultIncludes(CBEFile *pFile)
{
    if (!pFile->IsOfFileType(FILETYPE_HEADER))
	return;

    if (CCompiler::IsOptionSet(PROGRAM_TRACE_SERVER))
	*pFile << "#define DICE_TRACE_SERVER 1\n";
    if (CCompiler::IsOptionSet(PROGRAM_TRACE_CLIENT))
	*pFile << "#define DICE_TRACE_CLIENT 1\n";
    if (CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF))
	*pFile << "#define DICE_TRACE_MSGBUF 1\n";
}

/** \brief prints the tracing message before a call
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::BeforeCall(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF))
	return;

    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    // get tracing function
    string sFunc = CCompiler::GetTraceClientFunc();
    
    *pFile << "\t" << sFunc << " (\"" << pFunction->GetName() <<
	": before call\\n\");\n";
    pMsgBuffer->WriteDump(pFile, string());
}

/** \brief write the tracing code after the call
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::AfterCall(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF))
	return;
    
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    // get tracing function
    string sFunc = CCompiler::GetTraceClientFunc();

    *pFile << "\t" << sFunc << " (\"" << pFunction->GetName() <<
	": after call\\n\");\n";
    pMsgBuffer->WriteDump(pFile, string());
}

/** \brief write the tracing code shortly before calling dispatch function
 *  \param pFile the file to write to
 *  \param pFunction the dispatch function to write for
 */
void
CBETrace::BeforeDispatch(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!pFunction->IsComponentSide())
	return;
    if (!CCompiler::IsOptionSet(PROGRAM_TRACE_SERVER))
	return;
    
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sOpcodeVar = pNF->GetOpcodeVariable();
    string sFunc = CCompiler::GetTraceServerFunc();

    *pFile << "\t" << sFunc << " (\"opcode %%x received\\n\", "
	<< sOpcodeVar << ");\n";
}

/** \brief write the tracing code shortly after the call to the dispatch func
 *  \param pFile the file to write to
 *  \param pFunction the dispatch function to write for
 */
void
CBETrace::AfterDispatch(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!pFunction->IsComponentSide())
	return;
    if (!CCompiler::IsOptionSet(PROGRAM_TRACE_SERVER))
	return;
    
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sFunc = CCompiler::GetTraceServerFunc();
    string sReply = pNF->GetReplyCodeVariable();

    *pFile << "\t" << sFunc << " (\"reply %%s\\n\", (" <<
	sReply << "==DICE_REPLY)?\"DICE_REPLY\":" <<
	"((" << sReply << "==DICE_DEFERRED_REPLY)?\"DICE_DEFERRED_REPLY\":" <<
	"\"DICE_NEVER_REPLY\"));\n";
}

/** \brief write the tracing the code before reply IPC
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::BeforeReplyOnly(CBEFile *pFile,
    CBEFunction *pFunction)
{
    BeforeCall(pFile, pFunction);
}

/** \brief write tracing the code after reply IPC
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::AfterReplyOnly(CBEFile *pFile,
    CBEFunction *pFunction)
{
    AfterCall(pFile, pFunction);
}

/** \brief writes the tracing code before the server replies to request
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::BeforeReplyWait(CBEFile *pFile,
    CBEFunction *pFunction)
{
}

/** \brief writes the tracing code after the server waited for a message
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::AfterReplyWait(CBEFile *pFile,
    CBEFunction *pFunction)
{
    if (!CCompiler::IsOptionSet(PROGRAM_TRACE_MSGBUF))
	return;
    
    CBEMsgBufferType *pMsgBuffer = pFunction->GetMessageBuffer();
    assert(pMsgBuffer);
    // get tracing function
    string sFunc = CCompiler::GetTraceClientFunc();

    *pFile << "\t" << sFunc << " (\"" << pFunction->GetName() <<
	": after wait\\n\");\n";
    pMsgBuffer->WriteDump(pFile, string());
}

/** \brief write the tracing the code before calling component function
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::BeforeComponent(CBEFile *pFile,
    CBEFunction *pFunction)
{
}

/** \brief writes the tracing code after calling the component function
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::AfterComponent(CBEFile *pFile,
    CBEFunction *pFunction)
{
}

/** \brief write the tracing the code before marshalling
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::BeforeMarshalling(CBEFile *pFile,
    CBEFunction *pFunction)
{
}

/** \brief writes the tracing code after marshalling
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::AfterMarshalling(CBEFile *pFile,
    CBEFunction *pFunction)
{
}

/** \brief write the tracing the code before unmarshalling
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::BeforeUnmarshalling(CBEFile *pFile,
    CBEFunction *pFunction)
{
}

/** \brief writes the tracing code after unmarshalling
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void
CBETrace::AfterUnmarshalling(CBEFile *pFile,
    CBEFunction *pFunction)
{
}
