/**
 *    \file    dice/src/be/BETarget.cpp
 * \brief   contains the implementation of the class CBETarget
 *
 *    \date    01/11/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/BETarget.h"
#include "be/BENameSpace.h"
#include "be/BEContext.h"
#include "be/BEHeaderFile.h"
#include "be/BEImplementationFile.h"
#include "be/BEHeaderFile.h"
#include "be/BEImplementationFile.h"
#include "be/BESndFunction.h"
#include "be/BEWaitFunction.h"
#include "be/BEWaitAnyFunction.h"
#include "be/BEConstant.h"
#include "be/BETypedef.h"
#include "be/BEMsgBufferType.h"
#include "be/BEFunction.h"
#include "be/BERoot.h"
#include "be/BEDeclarator.h"

#include "fe/FEFile.h"
#include "fe/FELibrary.h"
#include "fe/FEInterface.h"
#include "fe/FEOperation.h"
#include "fe/FEFile.h"
#include "fe/FEConstDeclarator.h"
#include "fe/FETypedDeclarator.h"
#include "fe/FEDeclarator.h"

#include "Compiler.h"

CBETarget::CBETarget()
{
}

CBETarget::CBETarget(CBETarget & src)
: CBEObject(src)
{
    COPY_VECTOR(CBEHeaderFile, m_vHeaderFiles, iterH);
    COPY_VECTOR(CBEImplementationFile, m_vImplementationFiles, iterI);
}

/** \brief destructor of target class */
CBETarget::~CBETarget()
{
    DEL_VECTOR(m_vHeaderFiles);
    DEL_VECTOR(m_vImplementationFiles);
}

/** \brief generates the output files and code

 */
void CBETarget::Write()
{
    assert(false);
}

/** \brief adds another header file to the respective vector
 *  \param pHeaderFile the header file to add
 */
void CBETarget::AddFile(CBEHeaderFile * pHeaderFile)
{
    if (!pHeaderFile)
        return;
    m_vHeaderFiles.push_back(pHeaderFile);
    pHeaderFile->SetParent(this);
}

/** \brief removes a header file from the respective vector
 *  \param pHeader the header file to remove
 */
void CBETarget::RemoveFile(CBEHeaderFile * pHeader)
{
    if (!pHeader)
        return;
    vector<CBEHeaderFile*>::iterator iter;
    for (iter = m_vHeaderFiles.begin(); iter != m_vHeaderFiles.end(); iter++)
    {
        if (*iter == pHeader)
        {
            m_vHeaderFiles.erase(iter);
            return;
        }
    }
}

/** \brief adds another implementation file to the respective vector
 *  \param pImplementationFile the implementation file to add
 */
void CBETarget::AddFile(CBEImplementationFile * pImplementationFile)
{
    if (!pImplementationFile)
        return;
    m_vImplementationFiles.push_back(pImplementationFile);
    pImplementationFile->SetParent(this);
}

/** \brief removes an implementation file from the respective vectpr
 *  \param pImplementation the implementation file to remove
 */
void CBETarget::RemoveFile(CBEImplementationFile * pImplementation)
{
    if (!pImplementation)
        return;
    vector<CBEImplementationFile*>::iterator iter;
    for (iter = m_vImplementationFiles.begin(); iter != m_vImplementationFiles.end(); iter++)
    {
        if (*iter == pImplementation)
        {
            m_vImplementationFiles.erase(iter);
            return;
        }
    }
}

/** \brief retrievs a pointer to the first header file
 *  \return a pointer to the first header file
 */
vector<CBEHeaderFile*>::iterator CBETarget::GetFirstHeaderFile()
{
    return m_vHeaderFiles.begin();
}

/** \brief retrieves the next header file using the given pointer
 *  \param iter the pointer to the next header file
 *  \return a reference to the next header file
 */
CBEHeaderFile *CBETarget::GetNextHeaderFile(vector<CBEHeaderFile*>::iterator &iter)
{
    if (iter == m_vHeaderFiles.end())
        return 0;
    return *iter++;
}

/** \brief retrieves a pointer to the first implementation file
 *  \return a pointer to the first implementation file
 */
vector<CBEImplementationFile*>::iterator CBETarget::GetFirstImplementationFile()
{
    return m_vImplementationFiles.begin();
}

/** \brief retrieves a reference to the next implementation file
 *  \param iter the pointer to the next implementation file
 *  \return a reference to the next implementation file
 */
CBEImplementationFile *CBETarget::GetNextImplementationFile(vector<CBEImplementationFile*>::iterator &iter)
{
    if (iter == m_vImplementationFiles.end())
        return 0;
    return *iter++;
}

/** \brief writes the header files

 */
void CBETarget::WriteHeaderFiles()
{
    vector<CBEHeaderFile*>::iterator iter = GetFirstHeaderFile();
    CBEHeaderFile *pFile;
    while ((pFile = GetNextHeaderFile(iter)) != 0)
    {
        pFile->Write();
    }
}

/** \brief writes the implementation files

 */
void CBETarget::WriteImplementationFiles()
{
    vector<CBEImplementationFile*>::iterator iter = GetFirstImplementationFile();
    CBEImplementationFile *pFile;
    while ((pFile = GetNextImplementationFile(iter)) != 0)
    {
        pFile->Write();
    }
}

/** \brief adds the constant of the front-end file to the back-end file
 *  \param pFile the back-end file
 *  \param pFEFile the front-end file

 *  \return true if successful
 *
 * This implementation iterates over the interfaces and libs of the current file. It
 * also iterates over the included files if the program options allow it. And the
 * implementation also call AddConstantToFile for all constants of the file.
 */
bool CBETarget::AddConstantToFile(CBEFile * pFile, CFEFile * pFEFile)
{
    if (!pFEFile)
    {
        VERBOSE("CBETarget::AddConstantToFile aborted because front-end file is 0\n");
        return true;
    }
    if (!pFEFile->IsIDLFile())
    {
        VERBOSE("CBETarget::AddConstantToFile aborted because front-end file is not IDL file\n");
        return true;
    }

    vector<CFEInterface*>::iterator iterI = pFEFile->GetFirstInterface();
    CFEInterface *pInterface;
    while ((pInterface = pFEFile->GetNextInterface(iterI)) != 0)
    {
        if (!AddConstantToFile(pFile, pInterface))
            return false;
    }

    vector<CFELibrary*>::iterator iterL = pFEFile->GetFirstLibrary();
    CFELibrary *pLib;
    while ((pLib = pFEFile->GetNextLibrary(iterL)) != 0)
    {
        if (!AddConstantToFile(pFile, pLib))
            return false;
    }

    vector<CFEConstDeclarator*>::iterator iterC = pFEFile->GetFirstConstant();
    CFEConstDeclarator *pFEConstant;
    while ((pFEConstant = pFEFile->GetNextConstant(iterC)) != 0)
    {
        if (!AddConstantToFile(pFile, pFEConstant))
            return false;
    }

    if (DoAddIncludedFiles())
    {
        vector<CFEFile*>::iterator iterF = pFEFile->GetFirstChildFile();
        CFEFile *pIncFile;
        while ((pIncFile = pFEFile->GetNextChildFile(iterF)) != 0)
        {
            if (!AddConstantToFile(pFile, pIncFile))
                return false;
        }
    }

    return true;
}

/** \brief adds the constants of the front-end library to the back-end file
 *  \param pFile the back-end file
 *  \param pFELibrary the front-end library

 *  \return true if successful
 *
 * This implementation iterates over the interfaces and nested libs of the current library.
 * It also calls AddConstantToFile for every constant of the library.
 */
bool CBETarget::AddConstantToFile(CBEFile * pFile, CFELibrary * pFELibrary)
{
    vector<CFEInterface*>::iterator iterI = pFELibrary->GetFirstInterface();
    CFEInterface *pInterface;
    while ((pInterface = pFELibrary->GetNextInterface(iterI)) != 0)
    {
        if (!AddConstantToFile(pFile, pInterface))
            return false;
    }

    vector<CFELibrary*>::iterator iterL = pFELibrary->GetFirstLibrary();
    CFELibrary *pNestedLib;
    while ((pNestedLib = pFELibrary->GetNextLibrary(iterL)) != 0)
    {
        if (!AddConstantToFile(pFile, pNestedLib))
            return false;
    }

    vector<CFEConstDeclarator*>::iterator iterC = pFELibrary->GetFirstConstant();
    CFEConstDeclarator *pFEConstant;
    while ((pFEConstant = pFELibrary->GetNextConstant(iterC)) != 0)
    {
        if (!AddConstantToFile(pFile, pFEConstant))
            return false;
    }

    return true;
}

/** \brief adds the constants of the front-end interface to the back-end file
 *  \param pFile the back-end file
 *  \param pFEInterface the front-end interface

 *  \return true if successful
 *
 * This implementation iterates over the constants of the current interface
 */
bool CBETarget::AddConstantToFile(CBEFile * pFile, CFEInterface * pFEInterface)
{
    if (!pFEInterface)
    {
        VERBOSE("CBETarget::AddConstantToFile (interface) aborted because front-end interface is 0\n");
        return true;
    }

    vector<CFEConstDeclarator*>::iterator iter = pFEInterface->GetFirstConstant();
    CFEConstDeclarator *pFEConstant;
    while ((pFEConstant = pFEInterface->GetNextConstant(iter)) != 0)
    {
        if (!AddConstantToFile(pFile, pFEConstant))
            return false;
    }

    return true;
}

/** \brief adds the constants of the front-end to the back-end file
 *  \param pFile the file to add the function to
 *  \param pFEConstant the constant to generate the back-end constant from

 *  \return true if generation went alright
 *
 * This implementation generates a back-end constant and adds it to the header file.
 */
bool CBETarget::AddConstantToFile(CBEFile * pFile, CFEConstDeclarator * pFEConstant)
{
    if (dynamic_cast<CBEHeaderFile*>(pFile))
    {
        CBERoot *pRoot = GetSpecificParent<CBERoot>();
        assert(pRoot);
        CBEConstant *pConstant = pRoot->FindConstant(pFEConstant->GetName());
        if (!pConstant)
        {
	    CBEClassFactory *pCF = CCompiler::GetClassFactory();
            pConstant = pCF->GetNewConstant();
            ((CBEHeaderFile *) pFile)->AddConstant(pConstant);
            if (!pConstant->CreateBackEnd(pFEConstant))
            {
                ((CBEHeaderFile *) pFile)->RemoveConstant(pConstant);
                delete pConstant;
                VERBOSE("CBETarget::AddConstantToFile (constant) failed because could not create BE constant\n");
                return false;
            }
        }
        else
            ((CBEHeaderFile *) pFile)->AddConstant(pConstant);
    }

    return true;
}

/** \brief adds the type definition of the front-end file to the back-end file
 *  \param pFile the back-end file
 *  \param pFEFile the front-end file

 *  \return true if successful
 *
 * This implementation iterates over the interfaces and libs of the current file. It
 * also iterates over the included files if the program options allow it. And it calls
 * AddTypedefToFile for the type definitions of the file
 */
bool CBETarget::AddTypedefToFile(CBEFile * pFile, CFEFile * pFEFile)
{
    if (!pFEFile)
        return true;
    if (!pFEFile->IsIDLFile())
        return true;

    vector<CFEInterface*>::iterator iterI = pFEFile->GetFirstInterface();
    CFEInterface *pInterface;
    while ((pInterface = pFEFile->GetNextInterface(iterI)) != 0)
    {
        if (!AddTypedefToFile(pFile, pInterface))
            return false;
    }

    vector<CFELibrary*>::iterator iterL = pFEFile->GetFirstLibrary();
    CFELibrary *pLib;
    while ((pLib = pFEFile->GetNextLibrary(iterL)) != 0)
    {
        if (!AddTypedefToFile(pFile, pLib))
            return false;
    }

    vector<CFETypedDeclarator*>::iterator iterT = pFEFile->GetFirstTypedef();
    CFETypedDeclarator *pFETypedef;
    while ((pFETypedef = pFEFile->GetNextTypedef(iterT)) != 0)
    {
        if (!AddTypedefToFile(pFile, pFETypedef))
            return false;
    }

    if (DoAddIncludedFiles())
    {
        vector<CFEFile*>::iterator iterF = pFEFile->GetFirstChildFile();
        CFEFile *pIncFile;
        while ((pIncFile = pFEFile->GetNextChildFile(iterF)) != 0)
        {
            if (!AddTypedefToFile(pFile, pIncFile))
                return false;
        }
    }

    return true;
}

/** \brief adds the type definitions of the front-end library to the back-end file
 *  \param pFile the back-end file
 *  \param pFELibrary the front-end library

 *  \return true if successful
 *
 * This implementation iterates over the interfaces and nested libs of the current library.
 * And it calls AddTypedefToFile for the type definitions in the library.
 */
bool CBETarget::AddTypedefToFile(CBEFile * pFile, CFELibrary * pFELibrary)
{
    vector<CFEInterface*>::iterator iterI = pFELibrary->GetFirstInterface();
    CFEInterface *pInterface;
    while ((pInterface = pFELibrary->GetNextInterface(iterI)) != 0)
    {
        if (!AddTypedefToFile(pFile, pInterface))
            return false;
    }

    vector<CFELibrary*>::iterator iterL = pFELibrary->GetFirstLibrary();
    CFELibrary *pNestedLib;
    while ((pNestedLib = pFELibrary->GetNextLibrary(iterL)) != 0)
    {
        if (!AddTypedefToFile(pFile, pNestedLib))
            return false;
    }

    vector<CFETypedDeclarator*>::iterator iterT = pFELibrary->GetFirstTypedef();
    CFETypedDeclarator *pFETypedef;
    while ((pFETypedef = pFELibrary->GetNextTypedef(iterT)) != 0)
    {
        if (!AddTypedefToFile(pFile, pFETypedef))
            return false;
    }

    return true;
}

/** \brief adds the type definitions of the front-end interface to the back-end file
 *  \param pFile the back-end file
 *  \param pFEInterface the front-end interface

 *  \return true if successful
 *
 * This implementation iterates over the type definitions of the current interface.
 * It also adds one message buffer per interface to the header file.
 */
bool CBETarget::AddTypedefToFile(CBEFile * pFile, CFEInterface * pFEInterface)
{
    vector<CFETypedDeclarator*>::iterator iter = pFEInterface->GetFirstTypeDef();
    CFETypedDeclarator *pFETypedef;
    while ((pFETypedef = pFEInterface->GetNextTypeDef(iter)) != 0)
    {
        if (!AddTypedefToFile(pFile, pFETypedef))
            return false;
    }

    return true;
}

/** \brief adds the type definitions of the front-end to the back-end file
 *  \param pFile the file to add the function to
 *  \param pFETypedDeclarator the typedef to generate the back-end typedef from

 *  \return true if generation went alright
 *
 * This implementation adds the type definition to the header file, but skips
 * the implementation file.  It searches for the typedef at the root and then
 * adds a reference to its own collection.
 */
bool CBETarget::AddTypedefToFile(CBEFile * pFile,
                 CFETypedDeclarator * pFETypedDeclarator)
{
    if (dynamic_cast<CBEHeaderFile*>(pFile))
    {
        CBERoot *pRoot = GetSpecificParent<CBERoot>();
        assert(pRoot);
        vector<CFEDeclarator*>::iterator iter = pFETypedDeclarator->GetFirstDeclarator();
        CFEDeclarator *pDecl = pFETypedDeclarator->GetNextDeclarator(iter);
        CBETypedef *pTypedef = pRoot->FindTypedef(pDecl->GetName());
        assert(pTypedef);
        ((CBEHeaderFile *) pFile)->AddTypedef(pTypedef);
    }
    return true;
}

/** \brief finds a header file which belongs to a certain front-end file
 *  \param pFEFile the front-end file
 *  \return a reference to the respective file; 0 if not found
 *
 * The search uses the file name to find a file. The name-factory generates a
 * file-name based on the front-end file and this name is used to find the
 * file.
 */
CBEHeaderFile*
CBETarget::FindHeaderFile(CFEFile * pFEFile,
    int nFileType)
{
    DTRACE("%s for file %s called\n", __PRETTY_FUNCTION__,
	pFEFile->GetFileName().c_str());
    // get file name
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sFileName = pNF->GetFileName(pFEFile, nFileType);
    // search file
    return FindHeaderFile(sFileName);
}

/** \brief finds a header file which belongs to a certain front-end library
 *  \param pFELibrary the front-end library
 *  \return a reference to the respective file; 0 if not found
 *
 * The search uses the file name to find a file. The name-factory generates a
 * file-name based on the front-end library and this name is used to find the
 * file.
 */
CBEHeaderFile*
CBETarget::FindHeaderFile(CFELibrary * pFELibrary,
    int nFileType)
{
    DTRACE("%s for library %s called\n", __PRETTY_FUNCTION__,
	pFELibrary->GetName().c_str());
    // get file name
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sFileName = pNF->GetFileName(
	pFELibrary->GetSpecificParent<CFEFile>(0), nFileType);
    // search file
    return FindHeaderFile(sFileName);
}

/** \brief finds a header file which belongs to a certain front-end interface
 *  \param pFEInterface the front-end interface
 *  \return a reference to the respective file; 0 if not found
 *
 * The search uses the file name to find a file. The name-factory generates a
 * file-name based on the front-end interface and this name is used to find
 * the file.
 */
CBEHeaderFile*
CBETarget::FindHeaderFile(CFEInterface * pFEInterface,
    int nFileType)
{
    DTRACE("%s for interface %s called\n", __PRETTY_FUNCTION__,
	pFEInterface->GetName().c_str());
    // get file name
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sFileName = pNF->GetFileName(
	pFEInterface->GetSpecificParent<CFEFile>(0), nFileType);
    // search file
    return FindHeaderFile(sFileName);
}

/** \brief finds a header file which belongs to a certain front-end operation
 *  \param pFEOperation the front-end operation
 *  \return a reference to the respective file; 0 if not found
 *
 * The search uses the file name to find a file. The name-factory generates a
 * file-name based on the front-end operation and this name is used to find
 * the file.
 */
CBEHeaderFile*
CBETarget::FindHeaderFile(CFEOperation * pFEOperation,
    int nFileType)
{
    DTRACE("%s for operation %s called\n", __PRETTY_FUNCTION__,
	pFEOperation->GetName().c_str());
    // get file name
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sFileName = pNF->GetFileName(
	pFEOperation->GetSpecificParent<CFEFile>(0), nFileType);
    // search file
    return FindHeaderFile(sFileName);
}

/** \brief finds a header file which belongs to a certain name
 *  \param sFileName the file name to search for
 *  \return a reference to the respective file; 0 if not found
 */
CBEHeaderFile*
CBETarget::FindHeaderFile(string sFileName)
{
    DTRACE("%s (%s) called\n", __PRETTY_FUNCTION__, sFileName.c_str());
    
    vector<CBEHeaderFile*>::iterator iter = GetFirstHeaderFile();
    CBEHeaderFile *pHeader;
    while ((pHeader = GetNextHeaderFile(iter)) != 0)
    {
	DTRACE("%s: compare header \"%s\" with given \"%s\"\n",
	    __PRETTY_FUNCTION__,
	    pHeader->GetFileName().c_str(), sFileName.c_str());
        if (pHeader->GetFileName() == sFileName)
            return pHeader;
    }
    DTRACE("%s returns 0 (not found)\n", __PRETTY_FUNCTION__);
    return 0;
}

/** \brief tries to find the typedef with a type of the given name
 *  \param sTypeName the name to search for
 *  \return a reference to the searched typedef or 0
 *
 * To find a typedef, we iterate over the header files and check them.
 * (Implementation file cannot contain typedefs).
 */
CBETypedef *CBETarget::FindTypedef(string sTypeName)
{
    CBETypedef *pRet = 0;
    vector<CBEHeaderFile*>::iterator iter = GetFirstHeaderFile();
    CBEHeaderFile *pHeader;
    while (((pHeader = GetNextHeaderFile(iter)) != 0) && (!pRet))
    {
        pRet = pHeader->FindTypedef(sTypeName);
    }
    return pRet;
}

/** \brief tries to find the function with the given function name
 *  \param sFunctionName the name to search for
 *  \return a reference to the found function or 0 if not found
 *
 * To find a function, we iterate over the header files and the implementation files.
 */
CBEFunction *CBETarget::FindFunction(string sFunctionName)
{
    CBEFunction *pRet = 0;
    // search header files
    vector<CBEHeaderFile*>::iterator iterH = GetFirstHeaderFile();
    CBEHeaderFile *pHeader;
    while (((pHeader = GetNextHeaderFile(iterH)) != 0) && (!pRet))
    {
        pRet = pHeader->FindFunction(sFunctionName);
    }
    // search implementation files
    vector<CBEImplementationFile*>::iterator iterI = GetFirstImplementationFile();
    CBEImplementationFile *pImplementation;
    while (((pImplementation = GetNextImplementationFile(iterI)) != 0) && (!pRet))
    {
        pRet = pImplementation->FindFunction(sFunctionName);
    }
    return pRet;
}

/** \brief creates the back-end classes for this target
 *  \param pFEFile is the respective front-end file to use as reference

 *  \return true if successful
 */
bool CBETarget::CreateBackEnd(CFEFile *pFEFile)
{
    // if argument is 0, we assume a mistaken include file
    if (!pFEFile)
    {
        VERBOSE("CBETarget::CreateBackEnd aborted because front-end file is 0\n");
        return true;
    }
    // if file is not IDL file we simply return "no error", because C files might also be included files
    if (!pFEFile->IsIDLFile())
    {
        VERBOSE("CBETarget::CreateBackEnd aborted because front-end file (%s) is not IDL file\n", pFEFile->GetFileName().c_str());
        return true;
    }

    // create header file(s)
    if (!CreateBackEndHeader(pFEFile))
        return false;

    // create implementation file(s)
    if (!CreateBackEndImplementation(pFEFile))
        return false;

    return true;
}

/** \brief prints all generated target file name to the given output
 *  \param output the output stream to write to
 *  \param nCurCol the current column where to start output (indent)
 *  \param nMaxCol the maximum number of columns
 */
void CBETarget::PrintTargetFiles(FILE *output, int &nCurCol, int nMaxCol)
{
    // iterate over implementation files
    vector<CBEHeaderFile*>::iterator iterH = GetFirstHeaderFile();
    CBEHeaderFile *pHeader;
    while ((pHeader = GetNextHeaderFile(iterH)) != 0)
    {
        PrintTargetFileName(output, pHeader->GetFileName(), nCurCol, nMaxCol);
    }
    // iterate over header files
    vector<CBEImplementationFile*>::iterator iterI = GetFirstImplementationFile();
    CBEImplementationFile *pImpl;
    while ((pImpl = GetNextImplementationFile(iterI)) != 0)
    {
        PrintTargetFileName(output, pImpl->GetFileName(), nCurCol, nMaxCol);
    }
}

/** \brief prints the current filename
 *  \param output the output stream to print to
 *  \param sFilename the filename to print
 *  \param nCurCol the current output column
 *  \param nMaxCol the maximum output column
 */
void CBETarget::PrintTargetFileName(FILE *output, string sFilename, int &nCurCol, int nMaxCol)
{
    nCurCol += sFilename.length();
    if (nCurCol > nMaxCol)
    {
        fprintf(output, "\\\n ");
        nCurCol = sFilename.length()+1;
    }
    fprintf(output, "%s ", sFilename.c_str());
}

/** \brief tests if this target has a function with a given type
 *  \param sTypeName the name of the user defined type

 *  \return true if the function has this type
 *
 * The methods searches for a function, which has a parameter of the given
 * user defined type. Since all functions are declared in the header files,
 * we only need to search those.
 */
bool CBETarget::HasFunctionWithUserType(string sTypeName)
{
    vector<CBEHeaderFile*>::iterator iter = GetFirstHeaderFile();
    CBEHeaderFile *pFile;
    while ((pFile = GetNextHeaderFile(iter)) != 0)
    {
        if (pFile->HasFunctionWithUserType(sTypeName))
            return true;
    }
    return false;
}

/** \brief check if we add the object of the included files to this target

 *  \return true if we add the objects of the included files
 *
 * The default implementation tests for PROGRAM_FILE_ALL.
 */
bool CBETarget::DoAddIncludedFiles()
{
    return CCompiler::IsOptionSet(PROGRAM_FILE_ALL);
}
