/**
 *    \file    dice/src/be/BESrvLoopFunction.cpp
 *    \brief   contains the implementation of the class CBESrvLoopFunction
 *
 *    \date    01/21/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "BESrvLoopFunction.h"
#include "BEContext.h"
#include "BEFile.h"
#include "BEType.h"
#include "BETypedDeclarator.h"
#include "BEOpcodeType.h"
#include "BEReplyCodeType.h"
#include "BESwitchCase.h"
#include "BEWaitAnyFunction.h"
#include "BEDispatchFunction.h"
#include "BERoot.h"
#include "BEComponent.h"
#include "BEImplementationFile.h"
#include "BEHeaderFile.h"
#include "BEMsgBufferType.h"
#include "BEDeclarator.h"
#include "BETrace.h"

#include "Compiler.h"
#include "TypeSpec-Type.h"
#include "fe/FEInterface.h"
#include "fe/FEOperation.h"
#include "fe/FEStringAttribute.h"

CBESrvLoopFunction::CBESrvLoopFunction()
{
    m_pWaitAnyFunction = 0;
    m_pReplyAnyWaitAnyFunction = 0;
    m_pDispatchFunction = 0;
}

CBESrvLoopFunction::CBESrvLoopFunction(CBESrvLoopFunction & src)
: CBEInterfaceFunction(src)
{
    m_pWaitAnyFunction = src.m_pWaitAnyFunction;
    m_pReplyAnyWaitAnyFunction = src.m_pReplyAnyWaitAnyFunction;
    m_pDispatchFunction = src.m_pDispatchFunction;
}

/** \brief destructor of target class */
CBESrvLoopFunction::~CBESrvLoopFunction()
{
}

/** \brief creates the server loop function for the given interface
 *  \param pFEInterface the respective front-end interface

 *  \return true if successful
 *
 * A server loop function does usually not return anything. However, it might
 * be possible to return a status code or something similar. As parameters one
 * might use timeouts or similar.
 *
 * After we created the switch cases, we force them to reset the message
 * buffer type of their functions.  This way, we use in all functions the
 * message buffer type of this server loop.
 */
bool CBESrvLoopFunction::CreateBackEnd(CFEInterface * pFEInterface)
{
    // set target file name
    SetTargetFileName(pFEInterface);
    // set name
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sName = pNF->GetFunctionName(pFEInterface, FUNCTION_SRV_LOOP);

    if (!CBEInterfaceFunction::CreateBackEnd(pFEInterface))
        return false;
    // set source line number to last number of interface
    SetSourceLine(pFEInterface->GetSourceLineEnd());

    // set own message buffer
    if (!AddMessageBuffer(pFEInterface))
        return false;

    // CORBA_Object should not have any pointers (its a pointer type itself)
    // CORBA_Environment keeps its pointer

    // no parameters added

    CBERoot *pRoot = GetSpecificParent<CBERoot>();
    assert(pRoot);
    // search for wait any function
    string sFuncName = pNF->GetFunctionName(pFEInterface, FUNCTION_WAIT_ANY);
    m_pWaitAnyFunction = (CBEWaitAnyFunction*)pRoot->FindFunction(sFuncName);
    assert(m_pWaitAnyFunction);
    if (m_pCorbaObject)
    {
        vector<CBEDeclarator*>::iterator iterCO = m_pCorbaObject->GetFirstDeclarator();
        CBEDeclarator *pDecl = *iterCO;
        m_pWaitAnyFunction->SetCallVariable(pDecl->GetName(), pDecl->GetStars(), pDecl->GetName());
    }
    if (m_pCorbaEnv)
    {
        vector<CBEDeclarator*>::iterator iterCE = m_pCorbaEnv->GetFirstDeclarator();
        CBEDeclarator *pDecl = *iterCE;
        m_pWaitAnyFunction->SetCallVariable(pDecl->GetName(), pDecl->GetStars(), pDecl->GetName());
    }

    // search for reply-any-wait-any function
    sFuncName = pNF->GetFunctionName(pFEInterface, FUNCTION_REPLY_ANY_WAIT_ANY);
    m_pReplyAnyWaitAnyFunction = (CBEWaitAnyFunction*)pRoot->FindFunction(sFuncName);
    assert(m_pReplyAnyWaitAnyFunction);
    if (m_pCorbaObject)
    {
        vector<CBEDeclarator*>::iterator iterCO = m_pCorbaObject->GetFirstDeclarator();
        CBEDeclarator *pDecl = *iterCO;
        m_pReplyAnyWaitAnyFunction->SetCallVariable(pDecl->GetName(), pDecl->GetStars(), pDecl->GetName());
    }
    if (m_pCorbaEnv)
    {
        vector<CBEDeclarator*>::iterator iterCE = m_pCorbaEnv->GetFirstDeclarator();
        CBEDeclarator *pDecl = *iterCE;
        m_pReplyAnyWaitAnyFunction->SetCallVariable(pDecl->GetName(), pDecl->GetStars(), pDecl->GetName());
    }

    // search for dispatch function
    sFuncName = pNF->GetFunctionName(pFEInterface, FUNCTION_DISPATCH);
    m_pDispatchFunction = (CBEDispatchFunction*)pRoot->FindFunction(sFuncName);
    assert(m_pDispatchFunction);
    if (m_pCorbaObject)
    {
        vector<CBEDeclarator*>::iterator iterCO = m_pCorbaObject->GetFirstDeclarator();
        CBEDeclarator *pDecl = *iterCO;
        m_pDispatchFunction->SetCallVariable(pDecl->GetName(), pDecl->GetStars(), pDecl->GetName());
    }
    if (m_pCorbaEnv)
    {
        vector<CBEDeclarator*>::iterator iterCE = m_pCorbaEnv->GetFirstDeclarator();
        CBEDeclarator *pDecl = *iterCE;
        m_pDispatchFunction->SetCallVariable(pDecl->GetName(), pDecl->GetStars(), pDecl->GetName());
    }
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    if (pMsgBuffer)
    {
        vector<CBEDeclarator*>::iterator iterM = pMsgBuffer->GetFirstDeclarator();
        CBEDeclarator *pDecl = *iterM;
        m_pDispatchFunction->SetCallVariable(pDecl->GetName(), pDecl->GetStars(), pDecl->GetName());
    }

    return true;
}

/** \brief write the declaration of the CORBA_Object variable
 *  \param pFile the file to write to

 */
void CBESrvLoopFunction::WriteCorbaObjectDeclaration(CBEFile *pFile)
{
    if (m_pCorbaObject)
    {
        pFile->PrintIndent("");
        m_pCorbaObject->WriteDeclaration(pFile);
        pFile->Print("; // is client id\n");
    }
}

/** \brief write the declaration of the CORBA_Environment variable
 *  \param pFile the file to write to

 */
void CBESrvLoopFunction::WriteCorbaEnvironmentDeclaration(CBEFile *pFile)
{
    // we always have a CORBA_Environment, but sometimes
    // it is set as a cast from the server parameter.
    if (m_pCorbaEnv)
    {
        pFile->PrintIndent("");
        m_pCorbaEnv->WriteDeclaration(pFile);
        pFile->Print(";\n");
    }
}

/** \brief writes the variable declarations of this function
 *  \param pFile the file to write to

 *
 * The variable declarations of the call function include the message buffer for send and receive.
 * Variables declared for the server loop include:
 * - the CORBA_Object
 * - the CORBA_Environment
 * - the opcode
 */
void CBESrvLoopFunction::WriteVariableDeclaration(CBEFile * pFile)
{
    // write CORBA stuff
    WriteCorbaObjectDeclaration(pFile);
    WriteCorbaEnvironmentDeclaration(pFile);
    // clean up

    // write message buffer
    pFile->PrintIndent("");
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    pMsgBuffer->WriteDeclaration(pFile);
    pFile->Print(";\n");

    // write reply code
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEReplyCodeType *pReplyType = pCF->GetNewReplyCodeType();
    if (!pReplyType->CreateBackEnd())
    {
        delete pReplyType;
        return;
    }
    string sReply = CCompiler::GetNameFactory()->GetReplyCodeVariable();
    pFile->PrintIndent("");
    pReplyType->Write(pFile);
    pFile->Print(" %s;\n", sReply.c_str());
    delete pReplyType;

    // write opcode
    CBEOpcodeType *pOpcodeType = pCF->GetNewOpcodeType();
    if (!pOpcodeType->CreateBackEnd())
    {
        delete pOpcodeType;
        return;
    }
    string sOpcodeVar = CCompiler::GetNameFactory()->GetOpcodeVariable();
    pFile->PrintIndent("");
    pOpcodeType->Write(pFile);
    pFile->Print(" %s;\n", sOpcodeVar.c_str());

    delete pOpcodeType;
}

/** \brief writes the variable initializations of this function
 *  \param pFile the file to write to
 *
 * This implementation should initialize the message buffer and the pointers
 * of the out variables.  The CROBA stuff does not need to be set, because it
 * is set by the first wait function.  This function is also used to cast the
 * server loop parameter to the CORBA_Environment if it is used.
 */
void CBESrvLoopFunction::WriteVariableInitialization(CBEFile * pFile)
{
    assert(m_pTrace);
    m_pTrace->InitServer(pFile, this);

    // do CORBA_ENvironment cast before message buffer init, because it might
    // contain values used to init message buffer
    WriteEnvironmentInitialization(pFile);
    // init message buffer
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    pMsgBuffer->WriteInitialization(pFile);
}

/** \brief writes the invocation of the message transfer
 *  \param pFile the file to write to

 *
 * This implementation calls the underlying message trasnfer mechanisms
 */
void CBESrvLoopFunction::WriteInvocation(CBEFile * pFile)
{
    pFile->PrintIndent("/* invoke */\n");
}

/** \brief writes the server loop's function body
 *  \param pFile the target file

 */
void CBESrvLoopFunction::WriteBody(CBEFile * pFile)
{
    // write variable declaration and initialization
    WriteVariableDeclaration(pFile);
    WriteVariableInitialization(pFile);
    // write loop (contains switch)
    WriteLoop(pFile);
    // write clean up
    WriteCleanup(pFile);
    // write return
    WriteReturn(pFile);
}

/** \brief do not write any additional parameters
 *  \param pFile the file to write to

 *  \return true if a parameter has been written
 */
bool CBESrvLoopFunction::WriteBeforeParameters(CBEFile * pFile)
{
    return false;
}

/** \brief do not write any additional parameters
 *  \param pFile the file to write to

 *  \param bComma true if we have to write a comma first
 *
 * The server loop has a special parameter, which is a void pointer.
 * If it is NOT NULL, we will cast it to a CORBA_Environment, which
 * is used to set the server loop local CORBA_Environment. If it is
 * NULL, we use a default CORBA_Environment in the server loop.
 */
void CBESrvLoopFunction::WriteAfterParameters(CBEFile * pFile, bool bComma)
{
    if (bComma) // should be false, but who knows...
        pFile->Print(", ");
    string sServerParam = CCompiler::GetNameFactory()->GetServerParameterName();
    pFile->Print("void* %s", sServerParam.c_str());
}

/** \brief writes the loop
 *  \param pFile the file to write to

 */
void CBESrvLoopFunction::WriteLoop(CBEFile * pFile)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sOpcodeVar = pNF->GetOpcodeVariable();
    pFile->PrintIndent("");
    m_pWaitAnyFunction->WriteCall(pFile, sOpcodeVar);
    pFile->Print("\n");

    pFile->PrintIndent("while (1)\n");
    pFile->PrintIndent("{\n");
    pFile->IncIndent();

    // write switch
    WriteDispatchInvocation(pFile);

    // check if we should reply or not
    string sReply = pNF->GetReplyCodeVariable();
    pFile->PrintIndent("if (%s == DICE_REPLY)\n", sReply.c_str());
    pFile->IncIndent();
    pFile->PrintIndent("");
    m_pReplyAnyWaitAnyFunction->WriteCall(pFile, sOpcodeVar);
    pFile->Print("\n");
    pFile->DecIndent();
    pFile->PrintIndent("else\n");
    pFile->IncIndent();
    pFile->PrintIndent("");
    m_pWaitAnyFunction->WriteCall(pFile, sOpcodeVar);
    pFile->Print("\n");
    pFile->DecIndent();

    pFile->DecIndent();
    pFile->PrintIndent("}\n");
}

/** \brief writes the dispatcher invocation
 *  \param pFile the file to write to
 */
void CBESrvLoopFunction::WriteDispatchInvocation(CBEFile *pFile)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sReply = pNF->GetReplyCodeVariable();
    if (m_pDispatchFunction)
    {
	assert(m_pTrace);
	m_pTrace->BeforeDispatch(pFile, this);

        pFile->PrintIndent("");
        m_pDispatchFunction->WriteCall(pFile, sReply);
        pFile->Print("\n");

    	// tracing
	m_pTrace->AfterDispatch(pFile, this);
    }
}

/** \brief test if this function should be written
 *  \param pFile the file to write to
 *  \return  true if this function should be written
 *
 * A server loop is only written at the component's side.
 */
bool CBESrvLoopFunction::DoWriteFunction(CBEHeaderFile * pFile)
{
    if (!IsTargetFile(pFile))
        return false;
    return dynamic_cast<CBEComponent*>(pFile->GetTarget());
}

/** \brief test if this function should be written
 *  \param pFile the file to write to
 *  \return  true if this function should be written
 *
 * A server loop is only written at the component's side.
 */
bool CBESrvLoopFunction::DoWriteFunction(CBEImplementationFile * pFile)
{
    if (!IsTargetFile(pFile))
        return false;
    return dynamic_cast<CBEComponent*>(pFile->GetTarget());
}

/** \brief determines the direction, the server loop sends to
 *  \return DIRECTION_OUT
 */
int CBESrvLoopFunction::GetSendDirection()
{
    return DIRECTION_OUT;
}

/** \brief determined the direction the server loop receives from
 *  \return DIRECTION_IN
 */
int CBESrvLoopFunction::GetReceiveDirection()
{
    return DIRECTION_IN;
}

/** \brief adds the message buffer parameter to this function
 *  \param pFEInterface the front-end interface to use as reference

 *  \return true if successful
 */
bool CBESrvLoopFunction::AddMessageBuffer(CFEInterface * pFEInterface)
{
    // get class's message buffer
    CBEClass *pClass = GetClass();
    assert(pClass);
    // get message buffer type
    CBEMsgBufferType *pMsgBuffer = pClass->GetMessageBuffer();
    assert(pMsgBuffer);
    // msg buffer not initialized yet
    pMsgBuffer->InitCounts(pClass);
    // create own message buffer
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    m_pMsgBuffer = pCF->GetNewMessageBufferType(true);
    m_pMsgBuffer->SetParent(this);
    if (!m_pMsgBuffer->CreateBackEnd(pMsgBuffer))
    {
        delete m_pMsgBuffer;
        m_pMsgBuffer = 0;
        VERBOSE("%s failed because message buffer could not be created\n", __PRETTY_FUNCTION__);
        return false;
    }
    // reset the stars of the declarator
    CBEDeclarator *pName = m_pMsgBuffer->GetAlias();
    pName->IncStars(-pName->GetStars());
    return true;
}

/** \brief write the initialization code for the CORBA_Environment
 *  \param pFile the file to write to

 */
void CBESrvLoopFunction::WriteEnvironmentInitialization(CBEFile *pFile)
{
    string sServerParam = CCompiler::GetNameFactory()->GetServerParameterName();
    CBEDeclarator *pDecl = m_pCorbaEnv->GetDeclarator();
    // if (server-param)
    //   corba-env = (CORBA_Env*)server-param;
    pFile->PrintIndent("if (%s)\n", sServerParam.c_str());
    pFile->IncIndent();
    pFile->PrintIndent("%s = (", pDecl->GetName().c_str());
    m_pCorbaEnv->WriteType(pFile);
    pFile->Print("*)%s;\n", sServerParam.c_str());
    pFile->DecIndent();
    // should be set to default environment, but if
    // it is a pointer, we cannot, but have to allocate memory first...
    pFile->PrintIndent("else\n");
    pFile->PrintIndent("{\n");
    pFile->IncIndent();
    // corba-env = (CORBA_Env*)malloc(sizeof(CORBA_Env));
    pFile->PrintIndent("%s = ", pDecl->GetName().c_str());
    m_pCorbaEnv->GetType()->WriteCast(pFile, true);
    pFile->Print("_dice_alloca(sizeof");
    m_pCorbaEnv->GetType()->WriteCast(pFile, false);
    pFile->Print(");\n");
    if (CCompiler::IsBackEndSet(PROGRAM_BE_C))
    {
	pFile->PrintIndent("*%s = ", pDecl->GetName().c_str());
	m_pCorbaEnv->GetType()->WriteCast(pFile, false);
	pFile->Print("%s;\n", "dice_default_server_environment");
    } 
    else if (CCompiler::IsBackEndSet(PROGRAM_BE_CPP))
    {
	*pFile << "\t" << pDecl->GetName() << "->major = 0;\n";
	*pFile << "\t" << pDecl->GetName() << "->repos_id = 0;\n";
	*pFile << "\t" << pDecl->GetName() << "->_p.param = 0;\n";
	*pFile << "\t" << pDecl->GetName() << "->timeout = L4_IPC_TIMEOUT(0, 1, 0, 0, 15, 0);\n";
	*pFile << "\t" << pDecl->GetName() << "->rcv_fpage.fp.grant = 1;\n";
	*pFile << "\t" << pDecl->GetName() << "->rcv_fpage.fp.write = 1;\n";
	*pFile << "\t" << pDecl->GetName() << "->rcv_fpage.fp.size = L4_WHOLE_ADDRESS_SPACE;\n";
	*pFile << "\t" << pDecl->GetName() << "->rcv_fpage.fp.zero = 0;\n";
	*pFile << "\t" << pDecl->GetName() << "->rcv_fpage.fp.page = 0;\n";
	*pFile << "\t" << pDecl->GetName() << "->malloc = malloc_warning;\n";
	*pFile << "\t" << pDecl->GetName() << "->free = free_warning;\n";
	*pFile << "\t" << pDecl->GetName() << "->user_data = 0;\n";
	*pFile << "\tfor (int i=0; i<DICE_PTRS_MAX; i++)\n";
	pFile->IncIndent();
	*pFile << "\t" << pDecl->GetName() << "->ptrs[i] = 0;\n";
	pFile->DecIndent();
	*pFile << "\t" << pDecl->GetName() << "->ptrs_cur = 0;\n";
    }
    pFile->DecIndent();
    pFile->PrintIndent("}\n");
}

/** \brief writes the attributes for the function
 *  \param pFile the file to write to

 *
 * This implementation adds the "noreturn" attribute to the declaration
 */
void CBESrvLoopFunction::WriteFunctionAttributes(CBEFile* pFile)
{
    pFile->Print(" __attribute__((noreturn))");
}

/** \brief writes the return statement of this function
 *  \param pFile the file to write to

 *
 * Since this function is "noreturn" it is not allowed to have a return statement.
 */
void CBESrvLoopFunction::WriteReturn(CBEFile* pFile)
{
    /* empty */
}
