/**
 *    \file    dice/src/be/BESndFunction.cpp
 *    \brief   contains the implementation of the class CBESndFunction
 *
 *    \date    01/14/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/BESndFunction.h"
#include "be/BEContext.h"
#include "be/BEFile.h"
#include "be/BEType.h"
#include "be/BETypedDeclarator.h"
#include "be/BEClient.h"
#include "be/BEComponent.h"
#include "be/BEImplementationFile.h"
#include "be/BEHeaderFile.h"
#include "be/BEMsgBufferType.h"
#include "be/BESizes.h"
#include "be/BEMarshaller.h"
#include "be/BEAttribute.h"

#include "Compiler.h"
#include "TypeSpec-Type.h"
#include "fe/FEOperation.h"

CBESndFunction::CBESndFunction()
{
}

CBESndFunction::CBESndFunction(CBESndFunction & src):CBEOperationFunction(src)
{
}

/** \brief destructor of target class */
CBESndFunction::~CBESndFunction()
{

}

/** \brief writes the variable declarations of this function
 *  \param pFile the file to write to

 *
 * The variable declarations of the call function include the message buffer
 * for send and receive.
 */
void CBESndFunction::WriteVariableDeclaration(CBEFile * pFile)
{
    // define message buffer
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    pMsgBuffer->WriteDefinition(pFile, false);
    // check for temp
    if (HasVariableSizedParameters(GetSendDirection()) ||
        HasArrayParameters(GetSendDirection()))
    {
        string sTmpVar = CCompiler::GetNameFactory()->GetTempOffsetVariable();
        string sOffsetVar = CCompiler::GetNameFactory()->GetOffsetVariable();
        pFile->PrintIndent("unsigned %s __attribute__ ((unused));\n",
	    sTmpVar.c_str());
        pFile->PrintIndent("unsigned %s __attribute__ ((unused));\n",
	    sOffsetVar.c_str());
    }
}

/** \brief writes the variable initializations of this function
 *  \param pFile the file to write to

 *
 * This implementation should initialize the message buffer and the pointers of the out variables.
 */
void CBESndFunction::WriteVariableInitialization(CBEFile * pFile)
{
    // init message buffer
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    pMsgBuffer->WriteInitialization(pFile);
}

/** \brief writes the invocation of the message transfer
 *  \param pFile the file to write to

 *
 * This implementation calls the underlying message trasnfer mechanisms
 */
void CBESndFunction::WriteInvocation(CBEFile * pFile)
{
    pFile->PrintIndent("/* send */\n");
}

/** \brief writes the return statement
 *  \param pFile the file to write to
 *
 * If this function is called, an empty return statement should be written,
 * because it is also used to return in case of an error before doing
 * something else (like tracing).
 */
void CBESndFunction::WriteReturn(CBEFile * pFile)
{
    *pFile << "\treturn;\n";
}

/** \brief creates the back-end send function
 *  \param pFEOperation the corresponding front-end operation

 *  \return true if successful
 */
bool CBESndFunction::CreateBackEnd(CFEOperation * pFEOperation)
{
    // set target file name
    SetTargetFileName(pFEOperation);
    // set name
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sName = pNF->GetFunctionName(pFEOperation, FUNCTION_SEND);

    if (!CBEOperationFunction::CreateBackEnd(pFEOperation))
        return false;

    // set return type
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEType *pReturnType = pCF->GetNewType(TYPE_VOID);
    pReturnType->SetParent(this);
    if (!pReturnType->CreateBackEnd(false, 0, TYPE_VOID))
    {
        delete pReturnType;
        return false;
    }
    CBEType *pOldType = m_pReturnVar->ReplaceType(pReturnType);
    delete pOldType;

    // we have to sort the parameter before we add the message buffer
    SortParameters(0);

    // need a message buffer, don't we?
    if (!AddMessageBuffer(pFEOperation))
        return false;

    return true;
}

/** \brief adds the parameters of a front-end function to this function
 *  \param pFEOperation the front-end function
 *  \return true if successful
 *
 * This implementation adds the return value to the parameter list. The return
 * value is the value returned by the component-function.
 *
 * Since this function is called before the rest of the above CreateBE
 * function is executed, we can assume, that the return variable is still the
 * original function's return variable and not the opcode return variable.
 */
bool 
CBESndFunction::AddParameters(CFEOperation * pFEOperation)
{
    if (!GetReturnType()->IsVoid())
    {
        // create new parameter
        CBETypedDeclarator *pReturnParam = 
	    (CBETypedDeclarator*)(m_pReturnVar->Clone());
	int nDir = GetSendDirection();
	ATTR_TYPE nAttr = (nDir == DIRECTION_IN) ? ATTR_IN : ATTR_OUT;
	if (!pReturnParam->FindAttribute(nAttr))
	{
	    CBEAttribute *pAttr = pReturnParam->FindAttribute(
		(nAttr == ATTR_IN) ? ATTR_OUT : ATTR_IN);
	    pAttr->CreateBackEnd(nAttr);
	}
        CBEFunction::AddParameter(pReturnParam);
        AddSortedParameter(pReturnParam);
    }
    // call base class to add rest
    return CBEOperationFunction::AddParameters(pFEOperation);
}

/** \brief check if this parameter has to be unmarshalled
 *  \param pParameter the parameter to be unmarshalled

 *  \return true if it should be unmarshalled
 *
 * This implementation is empty, because a send function does not receive any
 * parameters and thus does not need any unmarshalling.
 */
bool
CBESndFunction::DoUnmarshalParameter(CBETypedDeclarator * pParameter)
{
    return false;
}

/** \brief test if this function should be written to the target file
 *  \param pFile the file to write to
 *  \return true if successful
 *
 * A send function is written at the client's side if the IN attribute is set,
 * and at the component's side if the OUT attribute is set. And of course, only
 * if the target file is suitable.
 */
bool CBESndFunction::DoWriteFunction(CBEHeaderFile * pFile)
{
    if (!IsTargetFile(pFile))
        return false;
    if (pFile->IsOfFileType(FILETYPE_CLIENT) && (FindAttribute(ATTR_IN)))
        return true;
    if (pFile->IsOfFileType(FILETYPE_COMPONENT) && (FindAttribute(ATTR_OUT)))
        return true;
    return false;
}

/** \brief test if this function should be written to the target file
 *  \param pFile the file to write to
 *  \return true if successful
 *
 * A send function is written at the client's side if the IN attribute is set,
 * and at the component's side if the OUT attribute is set. And of course, only
 * if the target file is suitable.
 */
bool CBESndFunction::DoWriteFunction(CBEImplementationFile * pFile)
{
    if (!IsTargetFile(pFile))
        return false;
    if (pFile->IsOfFileType(FILETYPE_CLIENT) && (FindAttribute(ATTR_IN)))
        return true;
    if (pFile->IsOfFileType(FILETYPE_COMPONENT) && (FindAttribute(ATTR_OUT)))
        return true;
    return false;
}

/** \brief return the direction, which this functions sends to
 *  \return DIRECTION_IN if sending to server, DIRECTION_OUT if sending to client
 */
int CBESndFunction::GetSendDirection()
{
    return IsComponentSide() ? DIRECTION_OUT : DIRECTION_IN;
}

/** \brief get the direction this function receives data from
 *  \return DIRECTION_IN if receiving from client, DIRECTION_OUT if receiving from server
 *
 * Since this function only sends data, the value should be superfluous.
 */
int CBESndFunction::GetReceiveDirection()
{
    return IsComponentSide() ? DIRECTION_IN : DIRECTION_OUT;
}

/** \brief calcualtes the size of this function
 *  \param nDirection the direction to calulate the size for

 *  \return the size of the function's parameters in bytes
 */
int CBESndFunction::GetSize(int nDirection)
{
    // get base class' size
    int nSize = CBEOperationFunction::GetSize(nDirection);
    if ((nDirection & DIRECTION_IN) &&
        !FindAttribute(ATTR_NOOPCODE))
        nSize += CCompiler::GetSizes()->GetOpcodeSize();
    return nSize;
}

/** \brief calculates the size of the fixed sized params of this function
 *  \param nDirection the direction to calc

 *  \return the size of the params in bytes
 */
int CBESndFunction::GetFixedSize(int nDirection)
{
    int nSize = CBEOperationFunction::GetFixedSize(nDirection);
    if ((nDirection & GetSendDirection()) &&
        !FindAttribute(ATTR_NOOPCODE))
        nSize += CCompiler::GetSizes()->GetOpcodeSize();
    return nSize;
}

/** \brief marshals the return value
 *  \param pFile the file to write to
 *  \param nStartOffset the offset at which the return value should be \
 *         marshalled
 *  \param bUseConstOffset true if a constant offset should be used, set it \
 *         to false if not possible
 *  \return the number of bytes used for the return value
 *
 * This function assumes that it is called before the marshalling of the other
 * parameters.
 */
int CBESndFunction::WriteMarshalReturn(CBEFile * pFile, int nStartOffset,
    bool& bUseConstOffset)
{
    VERBOSE("CBESndFunction::WriteMarshalReturn(%s) called\n",
	GetName().c_str());

    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sReturn = pNF->GetReturnVariable();
    CBETypedDeclarator *pReturn = FindParameter(sReturn);
    if (!pReturn)
        return 0;
    if (pReturn->GetType()->IsVoid())
        return 0;
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEMarshaller *pMarshaller = pCF->GetNewMarshaller();
    int nSize = pMarshaller->Marshal(pFile, pReturn, nStartOffset,
	bUseConstOffset, m_vParameters.size() == 0);
    delete pMarshaller;

    VERBOSE("CBESndFunction::WriteMarshalReturn(%s) finished\n",
	GetName().c_str());

    return nSize;
}
