/**
 *    \file    dice/src/be/BEOperationFunction.cpp
 *    \brief   contains the implementation of the class CBEOperationFunction
 *
 *    \date    01/14/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/BEOperationFunction.h"
#include "be/BEContext.h"
#include "be/BEAttribute.h"
#include "be/BEType.h"
#include "be/BETypedDeclarator.h"
#include "be/BEException.h"
#include "be/BERoot.h"
#include "be/BEDeclarator.h"
#include "be/BEClass.h"
#include "be/BEOpcodeType.h"
#include "be/BEMarshaller.h"

#include "fe/FEOperation.h"
#include "fe/FETypedDeclarator.h"
#include "Attribute-Type.h"
#include "fe/FEInterface.h"
#include "fe/FEStringAttribute.h"
#include "Compiler.h"

CBEOperationFunction::CBEOperationFunction()
{
}

CBEOperationFunction::CBEOperationFunction(CBEOperationFunction & src)
 : CBEFunction(src)
{
}

/**  \brief destructor of target class */
CBEOperationFunction::~CBEOperationFunction()
{
}

/** \brief prepares this class for further deployment using the front-end operation
 *  \param pFEOperation the respective front.end operation

 *  \return true if the code generation was succesful
 *
 * This implementation adds the attributes, types, parameters, exception, etc.
 * given by the front-end function to this instance of the back-end function.
 */
bool 
CBEOperationFunction::CreateBackEnd(CFEOperation * pFEOperation)
{
    assert(pFEOperation);
    // basic init
    CBEFunction::CreateBackEnd(pFEOperation);
    // add attributes
    if (!AddAttributes(pFEOperation))
    {
        VERBOSE("%s failed because attributes could not be added\n", 
	    __PRETTY_FUNCTION__);
        return false;
    }
    // add return type
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sReturn = pNF->GetReturnVariable();
    if (!SetReturnVar(pFEOperation->GetReturnType(), sReturn))
    {
        VERBOSE("%s failed because return var could not be set\n",
	    __PRETTY_FUNCTION__);
        return false;
    }
    // add parameters
    if (!AddParameters(pFEOperation))
    {
        VERBOSE("%s failed because parameters could not be added\n", 
	    __PRETTY_FUNCTION__);
        return false;
    }
    // add exceptions
    if (!AddExceptions(pFEOperation))
    {
        VERBOSE("%s failed because exceptions could not be added\n", 
	    __PRETTY_FUNCTION__);
        return false;
    }
    // set opcode name
    m_sOpcodeConstName = pNF->GetOpcodeConst(pFEOperation);
    // set parent
    CBERoot *pRoot = GetSpecificParent<CBERoot>();
    assert(pRoot);
    CFEInterface *pFEInterface = 
	pFEOperation->GetSpecificParent<CFEInterface>();
    assert(pFEInterface);
    m_pClass = pRoot->FindClass(pFEInterface->GetName());
    assert(m_pClass);
    // would like to test for class == parent, but this is not the case for
    // switch case: parent = srv-loop function

    // check if interface has error function and add its name if available
    if (pFEInterface)
    {
        if (pFEInterface->FindAttribute(ATTR_ERROR_FUNCTION))
        {
            CFEStringAttribute *pErrorFunc = static_cast<CFEStringAttribute*>
		(pFEInterface->FindAttribute(ATTR_ERROR_FUNCTION));
            m_sErrorFunction = pErrorFunc->GetString();
        }
        if (pFEInterface->FindAttribute(ATTR_ERROR_FUNCTION_CLIENT) &&
            !IsComponentSide())
        {
            CFEStringAttribute *pErrorFunc = static_cast<CFEStringAttribute*>
		(pFEInterface->FindAttribute(ATTR_ERROR_FUNCTION_CLIENT));
            m_sErrorFunction = pErrorFunc->GetString();
        }
        if (pFEInterface->FindAttribute(ATTR_ERROR_FUNCTION_SERVER) &&
            IsComponentSide())
        {
            CFEStringAttribute *pErrorFunc = static_cast<CFEStringAttribute*>
		(pFEInterface->FindAttribute(ATTR_ERROR_FUNCTION_SERVER));
            m_sErrorFunction = pErrorFunc->GetString();
        }
    }

    return true;
}

/** \brief adds the parameters of a front-end function to this class
 *  \param pFEOperation the front-end function

 *  \return true if successful
 */
bool 
CBEOperationFunction::AddParameters(CFEOperation * pFEOperation)
{
    VERBOSE("%s called for %s\n", __PRETTY_FUNCTION__,
        pFEOperation->GetName().c_str());
    vector<CFETypedDeclarator*>::iterator iter = 
	pFEOperation->GetFirstParameter();
    CFETypedDeclarator *pFEParameter;
    while ((pFEParameter = pFEOperation->GetNextParameter(iter)) != 0)
    {
        if (!AddParameter(pFEParameter))
            return false;
    }
    VERBOSE("%s returns true\n", __PRETTY_FUNCTION__);
    return true;
}

/** \brief adds a single parameter to this class
 *  \param pFEParameter the parameter to add

 *  \return true if successful
 */
bool 
CBEOperationFunction::AddParameter(CFETypedDeclarator * pFEParameter)
{
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBETypedDeclarator *pParameter = pCF->GetNewTypedDeclarator();
    CBEFunction::AddParameter(pParameter);
    if (!pParameter->CreateBackEnd(pFEParameter))
    {
        VERBOSE("%s failed because parameter could not be created\n", 
	    __PRETTY_FUNCTION__);
        RemoveParameter(pParameter);
        delete pParameter;
        return false;
    }
    AddSortedParameter(pParameter);
    return true;
}

/** \brief adds exceptions of a front-end function to this class
 *  \param pFEOperation the front-end function

 *  \return true if successful
 */
bool 
CBEOperationFunction::AddExceptions(CFEOperation * pFEOperation)
{
    VERBOSE("%s called for %s\n", __PRETTY_FUNCTION__,
        pFEOperation->GetName().c_str());
    vector<CFEIdentifier*>::iterator iter = 
	pFEOperation->GetFirstRaisesDeclarator();
    CFEIdentifier *pFEException;
    while ((pFEException = pFEOperation->GetNextRaisesDeclarator(iter)) != 0)
    {
        if (!AddException(pFEException))
            return false;
    }
    VERBOSE("%s returns true\n", __PRETTY_FUNCTION__);
    return true;
}

/** \brief adds a single exception to this class
 *  \param pFEException the exception to add

 *  \return true if successful
 */
bool 
CBEOperationFunction::AddException(CFEIdentifier * pFEException)
{
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEException *pException = pCF->GetNewException();
    CBEFunction::AddException(pException);
    if (!pException->CreateBackEnd(pFEException))
    {
        RemoveException(pException);
        delete pException;
        return false;
    }
    return true;
}

/** \brief adds attributes of a front-end function this this class
 *  \param pFEOperation the front-end operation

 *  \return true if successful
 */
bool 
CBEOperationFunction::AddAttributes(CFEOperation * pFEOperation)
{
    VERBOSE("%s called for %s\n", __PRETTY_FUNCTION__,
        pFEOperation->GetName().c_str());
    vector<CFEAttribute*>::iterator iter = pFEOperation->GetFirstAttribute();
    CFEAttribute *pFEAttribute;
    while ((pFEAttribute = pFEOperation->GetNextAttribute(iter)) != 0)
    {
        if (!AddAttribute(pFEAttribute))
            return false;
    }
    VERBOSE("%s returns true\n", __PRETTY_FUNCTION__);
    return true;
}

/** \brief adds a single attribute to this function
 *  \param pFEAttribute the attribute to add

 *  \return true if successful
 */
bool
CBEOperationFunction::AddAttribute(CFEAttribute * pFEAttribute)
{
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEAttribute *pAttribute = pCF->GetNewAttribute();
    CBEFunction::AddAttribute(pAttribute);
    if (!pAttribute->CreateBackEnd(pFEAttribute))
    {
        RemoveAttribute(pAttribute);
        delete pAttribute;
        return false;
    }
    return true;
}

/** \brief returns a reference to the interface belonging to this function
 *  \return a reference to the interface of this function
 *
 * Because we have a hierarchy of libraries, types, consts and interfaces
 * beside the file and function hierarchy, we have to connect them somewhere.
 * This is done using the m_pInterface member. It will be set differently for
 * Operation-function and interface-functions. Anyhow: if a back-end function
 * needs information about its interface it can obtain a reference to this
 * interface using this function
 */
CBEClass* CBEOperationFunction::GetClass()
{
    return m_pClass;
}

/** \brief marshals the opcode at the specified offset
 *  \param pFile the file to write to
 *  \param nStartOffset the offset to start marshalling from
 *  \param bUseConstOffset true if nStartOffset should be used

 *  \return the size of the marshalled opcode
 *
 * This function assumes that it is called before the other parameters
 */
int 
CBEOperationFunction::WriteMarshalOpcode(CBEFile * pFile, 
    int nStartOffset, 
    bool& bUseConstOffset)
{
    /* if the attribute noopcode is set, do not marshal an opcode */
    if (FindAttribute(ATTR_NOOPCODE))
        return 0;

    int nSize = 0;
    // opcode type
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEOpcodeType *pType = pCF->GetNewOpcodeType();
    pType->SetParent(this);
    if (pType->CreateBackEnd())
    {
        CBETypedDeclarator *pConst = pCF->GetNewTypedDeclarator();
        pConst->SetParent(this);
        if (pConst->CreateBackEnd(pType, m_sOpcodeConstName))
        {
            CBEMarshaller *pMarshaller = pCF->GetNewMarshaller();
            // only if we really did marshal something, set size
            nSize = pMarshaller->Marshal(pFile, pConst, nStartOffset,
                bUseConstOffset, m_vParameters.size() == 0);
            delete pMarshaller;
        }
        delete pConst;
    }
    delete pType;
    return nSize;
}

