/**
 *    \file    dice/src/be/BEMarshaller.h
 *    \brief    contains the declaration of the class CBEMarshaller
 *
 *    \date    05/08/2002
 *    \author    Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

/** preprocessing symbol to check header file */
#ifndef __DICE_BEMARSHALLER_H__
#define __DICE_BEMARSHALLER_H__

#include <be/BEObject.h>
#include <vector>
using namespace std;

#include "be/BEDeclarator.h" // needed for declarator-stack

class CBEFile;
class CBEFunction;
class CBETypedDeclarator;
class CBEType;

/** \class CBEMarshaller
 *  \brief the class contains the marshalling code
 */
class CBEMarshaller : public CBEObject
{
public:
    /** \brief constructor of marshaller */
    CBEMarshaller();
    virtual ~CBEMarshaller();

    virtual int Marshal(CBEFile *pFile, CBEFunction *pFunction, int nStartOffset, bool& bUseConstOffset);
    virtual int Marshal(CBEFile *pFile, CBEFunction * pFunction, int nFEType, int nNumber, int nStartOffset, bool & bUseConstOffset);
    virtual int Marshal(CBEFile *pFile, CBETypedDeclarator *pParameter, int nStartOffset, bool& bUseConstOffset, bool bLastParameter);
    virtual int MarshalValue(CBEFile *pFile, CBEFunction *pFunction, int nBytes, int nValue, int nStartOffset, bool & bUseConstOffset);
    virtual int Unmarshal(CBEFile * pFile, CBEFunction * pFunction, int nStartOffset, bool& bUseConstOffset);
    virtual int Unmarshal(CBEFile * pFile, CBEFunction * pFunction, int nFEType, int nNumber, int nStartOffset, bool& bUseConstOffset);
    virtual int Unmarshal(CBEFile * pFile, CBETypedDeclarator * pParameter, int nStartOffset, bool& bUseConstOffset, bool bLastParameter);
    virtual bool MarshalToPosition(CBEFile *pFile, CBEFunction *pFunction, int nPosition, int nPosSize, int nDirection, bool bWrite);
    virtual bool MarshalToPosition(CBEFile *pFile, CBETypedDeclarator *pParameter,  int nPosition,  int nPosSize,  int nCurSize,  bool bWrite);

    virtual bool TestPositionSize(CBEFunction *pFunction, int nPosSize,
	int nDirection, bool bAllowSmaller, bool bAllowLarger, int nNumber);

protected: // Protected attributes
  /** \var CBEFile *m_pFile
   *  \brief the file to write to
   */
  CBEFile *m_pFile;
  /** \var bool m_bMarshal
   *  \brief true if we are marshalling; false if unmarshalling
   */
  bool m_bMarshal;
  /** \var CBEType *m_pType
   *  \brief the type of the currently marshalled variable
   */
  CBEType *m_pType;
  /** \var vector<CDeclaratorStackLocation*> m_vDeclaratorStack
   *  \brief the declarator stack for the marshalled parameter
   */
  vector<CDeclaratorStackLocation*> m_vDeclaratorStack;
  /** \var CBETypedDeclarator *m_pParameter
   *  \brief a reference to the top parameter (used to find attributes)
   */
  CBETypedDeclarator* m_pParameter;
  /** \var CBEFunction *m_pFunction
   *  \brief a reference to the function
   */
  CBEFunction *m_pFunction;

protected: // Protected methods
    virtual int MarshalDeclarator(CBEType *pType, int nStartOffset, bool& bUseConstOffset, bool bIncOffsetVariable, bool bLastParameter);
    virtual int MarshalUnion(CBEUnionType *pType, int nStartOffset, bool& bUseConstOffset, bool bLastParameter);
    virtual int MarshalCUnion(CBEUnionType *pType, int nStartOffset, bool& bUseConstOffset, bool bLastParameter);
    virtual int MarshalStruct(CBEStructType *pType, int nStartOffset, bool& bUseConstOffset, bool bLastParameter);
    virtual int MarshalArray(CBEType *pType, int nStartOffset, bool& bUseConstOffset, bool bLastParameter);
    virtual int MarshalString(CBEType *pType, int nStartOffset, bool & bUseConstOffset, bool bLastParameter);
    virtual int MarshalConstArray(CBEType *pType,
        int nStartOffset,
        bool & bUseConstOffset,
        bool bLastParameter,
        vector<CBEExpression*>::iterator iter,
        int nLevel);
    virtual int MarshalVariableArray(CBEType * pType, int nStartOffset, bool & bUseConstOffset, bool bLastParameter);
    virtual int MarshalValue(int nBytes, int nValue, int nStartOffset, bool & bUseConstOffset, bool bIncOffsetVariable);
    virtual int MarshalBitfieldStruct(CBEStructType * pType, int nStartOffset, bool & bUseConstOffset);
    virtual void WriteBuffer(CBEType *pType, int nStartOffset, bool& bUseConstOffset, bool bDereferencePosition, bool bCast);
    virtual void WriteBuffer(string sTypeName, int nStartOffset, bool & bUseConstOffset, bool bDereferencePosition, bool bCast);

    virtual bool MarshalDeclaratorToPosition(CBEType *pType, int nStartSize, int nPosition, int nPosSize, bool bWrite);
    virtual bool MarshalArrayToPosition(CBEType *pType, int nStartSize, int nPosition, int nPosSize, bool bWrite);
    virtual bool MarshalConstArrayToPosition(CBEType *pType, int nStartSize, int nPosition, int nPosSize, bool bWrite, vector<CBEExpression*>::iterator iter, int nLevel);
    virtual bool MarshalStructToPosition(CBEStructType *pType, int nStartSize, int nPosition, int nPosSize, bool bWrite);
    virtual bool MarshalUnionToPosition(CBEUnionType *pType, int nStartSize, int nPosition, int nPosSize, bool bWrite);

    virtual bool TestPositionSize(CBETypedDeclarator* pParameter, int nPosSize,
	bool bAllowSmaller, bool bAllowLarger);
    virtual bool TestDeclaratorPositionSize(CBEType* pType, int nPosSize,
	bool bAllowSmaller, bool bAllowLarger);
    virtual bool TestArrayPositionSize(CBEType* pType, int nPosSize,
	bool bAllowSmaller, bool bAllowLarger);
    virtual bool TestConstArrayPositionSize(CBEType* pType, int nPosSize,
	bool bAllowSmaller, bool bAllowLarger,
	vector<CBEExpression*>::iterator iter, int nLevel);
    virtual bool TestStructPositionSize(CBEStructType* pType, int nPosSize,
	bool bAllowSmaller, bool bAllowLarger);
    virtual bool TestUnionPositionSize(CBEUnionType* pType, int nPosSize,
	bool bAllowSmaller, bool bAllowLarger);

    virtual void WriteAssignment(CBEType *pType, int nStartOffset, bool &bUseConstOffset, int nAlignment);
};

#endif  // !__DICE_BEMARSHALLER_H__
