/**
 *    \file    dice/src/be/BEMarshalFunction.cpp
 *    \brief   contains the implementation of the class CBEMarshalFunction
 *
 *    \date    10/09/2003
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */
#include "be/BEMarshalFunction.h"
#include "be/BEContext.h"
#include "be/BEType.h"
#include "be/BETypedDeclarator.h"
#include "be/BEOpcodeType.h"
#include "be/BEMsgBufferType.h"
#include "be/BEUserDefinedType.h"
#include "be/BEHeaderFile.h"
#include "be/BEImplementationFile.h"
#include "be/BEComponent.h"
#include "be/BESizes.h"
#include "be/BETrace.h"
#include "Compiler.h"
#include "TypeSpec-Type.h"
#include "fe/FEInterface.h"
#include "fe/FEOperation.h"
#include "fe/FETypedDeclarator.h"

CBEMarshalFunction::CBEMarshalFunction()
 : CBEOperationFunction()
{
}

CBEMarshalFunction::CBEMarshalFunction(CBEMarshalFunction & src)
: CBEOperationFunction(src)
{
}

/** \brief destructor of target class */
CBEMarshalFunction::~CBEMarshalFunction()
{
}

/** \brief creates the back-end marshal function
 *  \param pFEOperation the corresponding front-end operation
 *  \return true if successful
 *
 * This function should only contain OUT parameters if it is on the
 * component's side an IN parameters if it is on the client's side.
 */
bool 
CBEMarshalFunction::CreateBackEnd(CFEOperation * pFEOperation)
{
    // set target file name
    SetTargetFileName(pFEOperation);
    // set name
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sName = pNF->GetFunctionName(pFEOperation, FUNCTION_MARSHAL);

    if (!CBEOperationFunction::CreateBackEnd(pFEOperation))
    {
        VERBOSE("%s failed because base function could not be created\n",
	    __PRETTY_FUNCTION__);
        return false;
    }

    // add parameters
    if (!AddMessageBuffer(pFEOperation->GetSpecificParent<CFEInterface>()))
    {
        VERBOSE("%s failed because message buffer could not be created\n", 
	    __PRETTY_FUNCTION__);
        return false;
    }

    // set return type
    if (IsComponentSide())
    {
	CBEClassFactory *pCF = CCompiler::GetClassFactory();
        CBEType *pReturnType = pCF->GetNewType(TYPE_VOID);
        pReturnType->SetParent(this);
        if (!pReturnType->CreateBackEnd(false, 0, TYPE_VOID))
        {
            VERBOSE("%s failed because return var could not be created\n", 
		__PRETTY_FUNCTION__);
            delete pReturnType;
            return false;
        }
        CBEType *pOldType = m_pReturnVar->ReplaceType(pReturnType);
        delete pOldType;
    }

    return true;
}

/** \brief writes the variable declarations of this function
 *  \param pFile the file to write to
 *
 * The variable declarations contains the return variable if needed. And a
 * temporary variable if we have any variable sized arrays. No message buffer
 * definition (its an parameter).
 */
void 
CBEMarshalFunction::WriteVariableDeclaration(CBEFile * pFile)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    // do NOT declare return variable
    // check for temp
    if (HasVariableSizedParameters(GetSendDirection()) || HasArrayParameters())
    {
        string sTmpVar = pNF->GetTempOffsetVariable();
        string sOffsetVar = pNF->GetOffsetVariable();
	*pFile << "\tunsigned " << sTmpVar << " __attribute__ ((unused));\n";
	*pFile << "\tunsigned " << sOffsetVar << " __attribute__ ((unused));\n";
    }
    // declare local exception variable if at component's side
    if (IsComponentSide())
        WriteExceptionWordDeclaration(pFile, true);
}

/** \brief writes the variable initializations of this function
 *  \param pFile the file to write to
 *
 * This implementation should initialize the pointers of the out variables. Do
 * not initialize the message buffer - this may overwrite the values we try to
 * unmarshal.
 */
void 
CBEMarshalFunction::WriteVariableInitialization(CBEFile * pFile)
{
}

/** \brief writes the invocation of the message transfer
 *  \param pFile the file to write to
 *
 * This implementation does nothing, because the unmarshalling does not
 * contain a message transfer.
 */
void 
CBEMarshalFunction::WriteInvocation(CBEFile * pFile)
{
}

/** \brief writes the unmarshalling of the message
 *  \param pFile the file to write to
 *  \param nStartOffset the offset where to start unmarshalling
 *  \param bUseConstOffset true if a constant offset should be used, set it \
 *         to false if not possible
 *
 * This implementation iterates over the parameters (except the message buffer
 * itself) and unmarshals them.
 */
void 
CBEMarshalFunction::WriteMarshalling(CBEFile * pFile, 
    int nStartOffset, 
    bool& bUseConstOffset)
{
    assert(m_pTrace);
    bool bLocalTrace = false;
    if (!m_bTraceOn)
    {
	m_pTrace->BeforeMarshalling(pFile, this);
	m_bTraceOn = bLocalTrace = true;
    }
    
    if (IsComponentSide())
    {
        // start after exception
        nStartOffset += WriteMarshalException(pFile, nStartOffset, 
	    bUseConstOffset);
    }
    else
    {
        // start after opcode
	CBEClassFactory *pCF = CCompiler::GetClassFactory();
        CBEOpcodeType *pOpcodeType = pCF->GetNewOpcodeType();
        pOpcodeType->SetParent(this);
        if (pOpcodeType->CreateBackEnd())
            nStartOffset += pOpcodeType->GetSize();
        delete pOpcodeType;
    }
    // now unmarshal rest
    CBEOperationFunction::WriteMarshalling(pFile, nStartOffset, 
	bUseConstOffset);

    if (bLocalTrace)
    {
	m_pTrace->AfterMarshalling(pFile, this);
	m_bTraceOn = false;
    }
}

/** \brief adds the parameters of a front-end function to this function
 *  \param pFEOperation the front-end function
 *  \return true if successful
 *
 * This implementation adds the return value to the parameter list. The return
 * value is the value returned by the component-function.
 *
 * Since this function is called before the rest of the above CreateBE
 * function is executed, we can assume, that the return variable is still the
 * original function's return variable and not the opcode return variable.
 */
bool 
CBEMarshalFunction::AddParameters(CFEOperation * pFEOperation)
{
    if (!GetReturnType()->IsVoid())
    {
        // create new parameter
        CBETypedDeclarator *pReturnParam = 
	    (CBETypedDeclarator*)(m_pReturnVar->Clone());
        CBEFunction::AddParameter(pReturnParam);
        AddSortedParameter(pReturnParam);
    }
    // call base class to add rest
    return CBEOperationFunction::AddParameters(pFEOperation);
}

/** \brief adds a single parameter to this function
 *  \param pFEParameter the parameter to add
 *  \return true if successful
 *
 * This function decides, which parameters to add and which not. The
 * parameters to marshal are for client-to-component transfer the IN
 * parameters and for component-to-client transfer the OUT and return
 * parameters. We depend on the information set in m_bComponentSide.
 */
bool 
CBEMarshalFunction::AddParameter(CFETypedDeclarator * pFEParameter)
{
    if (IsComponentSide())
    {
        // we transfer from the component to the client
        if (!(pFEParameter->FindAttribute(ATTR_OUT)))
            // skip adding a parameter if it has no OUT
            return true;
    }
    else
    {
        if (!(pFEParameter->FindAttribute(ATTR_IN)))
            // skip parameter if it has no IN
            return true;
    }
    return CBEOperationFunction::AddParameter(pFEParameter);
}

/** \brief checks if this parameter should be marshalled or not
 *  \param pParameter the parameter to check
 *  \return true if this parameter should be marshalled
 *
 * Return true if the at component's side, the parameter has an OUT attribute,
 * or if at client's side the parameter has an IN attribute.
 */
bool 
CBEMarshalFunction::DoMarshalParameter(CBETypedDeclarator * pParameter)
{
    if (IsComponentSide())
    {
        if (pParameter->FindAttribute(ATTR_OUT))
            return true;
    }
    else
    {
        if (pParameter->FindAttribute(ATTR_IN))
            return true;
    }
    return false;
}

/** \brief checks if this parameter should be unmarshalled or not
 *  \param pParameter the parameter to check
 *  \return true if this parameter should be unmarshalled
 */
bool
CBEMarshalFunction::DoUnmarshalParameter(CBETypedDeclarator * pParameter)
{
    return false;
}

/** \brief writes the message buffer parameter
 *  \param pFile the file to write to
 *  \param bComma true if a comma has to be written before the parameter
 */
void
CBEMarshalFunction::WriteAfterParameters(CBEFile * pFile, 
    bool bComma)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (bComma)
        *pFile << ",\n\t";
    WriteParameter(pFile, pMsgBuffer, false /* no const msgbuf */);
    CBEOperationFunction::WriteAfterParameters(pFile, true);
}

/** \brief writes the message buffer call parameter
 *  \param pFile the file to write to
 *  \param bComma true if a comma has to be written before the declarators
 *
 * This is also where the environment variable is written. If the server has a
 * parameter of type Corba-Environment, it is a pointer in the server loop and
 * when calling the unmarshal function, needs no reference making '&'.
 */
void
CBEMarshalFunction::WriteCallAfterParameters(CBEFile * pFile, 
    bool bComma)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    assert(pMsgBuffer);
    if (bComma)
        *pFile << ",\n\t";
    if (m_bCastMsgBufferOnCall)
        pMsgBuffer->GetType()->WriteCast(pFile, true);
    WriteCallParameter(pFile, pMsgBuffer);
    CBEOperationFunction::WriteCallAfterParameters(pFile, true);
}

/** \brief tries to find a parameter by its type
 *  \param sTypeName the name of the type
 *  \return a reference to the found parameter
 */
CBETypedDeclarator*
CBEMarshalFunction::FindParameterType(string sTypeName)
{
    CBEMsgBufferType *pMsgBuffer = GetMessageBuffer();
    if (pMsgBuffer)
    {
        CBEType *pType = pMsgBuffer->GetType();
        if (dynamic_cast<CBEUserDefinedType*>(pType))
        {
            if (((CBEUserDefinedType*)pType)->GetName() == sTypeName)
                return pMsgBuffer;
        }
        if (pType->HasTag(sTypeName))
            return pMsgBuffer;
    }
    return CBEOperationFunction::FindParameterType(sTypeName);
}

/** \brief adds the specific message buffer parameter for this function
 *  \param pFEInterface the respective front-end interface to use as reference
 *  \return true if the create process was successful
 *
 * Instead of creating a whole new message buffer type, we use the existing type
 * of the class as a user defined type.
 */
bool 
CBEMarshalFunction::AddMessageBuffer(CFEInterface * pFEInterface)
{
    // get class's message buffer
    CBEClass *pClass = GetClass();
    assert(pClass);
    // get message buffer type
    CBEMsgBufferType *pMsgBuffer = pClass->GetMessageBuffer();
    assert(pMsgBuffer);
    // we have to sort parameters before the message buffer extracts the sizes
    SortParameters(0);
    // msg buffer not yet initialized
    pMsgBuffer->InitCounts(pClass);
    // create own message buffer
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    m_pMsgBuffer = pCF->GetNewMessageBufferType(true);
    m_pMsgBuffer->SetParent(this);
    if (!m_pMsgBuffer->CreateBackEnd(pMsgBuffer))
    {
        delete m_pMsgBuffer;
        m_pMsgBuffer = 0;
        VERBOSE("%s failed because message buffer could not be created\n", 
	    __PRETTY_FUNCTION__);
        return false;
    }
    // since we reply to a specific message, we have to set the correct counts
    m_pMsgBuffer->ZeroCounts(GetSendDirection());
    // since InitCounts uses MAX to determine counts, the receive direction
    // will have no effect
    m_pMsgBuffer->InitCounts(this);
    return true;
}

/** \brief gets the direction, which the marshal-parameters have
 *  \return if at client's side DIRECTION_IN, if at server's side DIRECTION_OUT
 *
 * Since this function ignores marshalling parameter this value should be
 * irrelevant
 */
int CBEMarshalFunction::GetSendDirection()
{
    return IsComponentSide() ? DIRECTION_OUT : DIRECTION_IN;
}

/** \brief gets the direction of the unmarshal-parameters
 *  \return if at client's side DIRECTION_OUT, if at server's side DIRECTION_IN
 */
int CBEMarshalFunction::GetReceiveDirection()
{
    return IsComponentSide() ? DIRECTION_IN : DIRECTION_OUT;
}

/** \brief calculates the size of the function's parameters
 *  \param nDirection the direction to count
 *  \return the size of the parameters
 *
 * The reply function has the return value as a parameter. The base class'
 * GetSize function adds the size of the return type to the sum of the
 * parameters. Thus the status code for the IPC is counted even though it
 * shouldn't. We have to subtract it from the calculated size.
 */
int CBEMarshalFunction::GetSize(int nDirection)
{
    int nSize = CBEOperationFunction::GetSize(nDirection);
    if ((nDirection & DIRECTION_OUT) &&
        !FindAttribute(ATTR_NOEXCEPTIONS))
        nSize += CCompiler::GetSizes()->GetExceptionSize();
    return nSize;
}

/** \brief calculates the size of the function's parameters
 *  \param nDirection the direction to count
 *  \return the size of the parameters
 */
int CBEMarshalFunction::GetFixedSize(int nDirection)
{
    int nSize = CBEOperationFunction::GetFixedSize(nDirection);
    if ((nDirection & DIRECTION_OUT) &&
        !FindAttribute(ATTR_NOEXCEPTIONS))
        nSize += CCompiler::GetSizes()->GetExceptionSize();
    return nSize;
}

/** \brief do not count this function's return var
 *  \param nDirection the direction to count
 *  \return zero
 *
 * The return var (or the variable in m_pReturnValue is the IPC reply code,
 * which should not be counted as a parameter.
 */
int CBEMarshalFunction::GetMaxReturnSize(int)
{
    return 0;
}

/** \brief do not count this function's return var
 *  \param nDirection the direction to count
 *  \return zero
 *
 * The return var (or the variable in m_pReturnValue is the IPC reply code,
 * which should not be counted as a parameter.
 */
int CBEMarshalFunction::GetFixedReturnSize(int)
{
    return 0;
}

/** \brief do not count this function's return var
 *  \param nDirection the direction to count
 *  \return zero
 *
 * The return var (or the variable in m_pReturnValue is the IPC reply code,
 * which should not be counted as a parameter.
 */
int CBEMarshalFunction::GetReturnSize(int nDirection)
{
    return 0;
}

/** \brief writes the return statement of the function
 *  \param pFile the file to write to
 *
 * The Marshal function has nor return value.
 */
void CBEMarshalFunction::WriteReturn(CBEFile * pFile)
{
    *pFile << "\treturn;\n";
}

/** \brief test if this function should be written
 *  \param pFile the file to write to
 *  \return true if should be written
 *
 * A marshal function is written if client's side and IN or if component's side
 * and one of the parameters has an OUT or we have an exception to transmit.
 */
bool
CBEMarshalFunction::DoWriteFunction(CBEHeaderFile * pFile)
{
    if (!IsTargetFile(pFile))
        return false;
    if (pFile->IsOfFileType(FILETYPE_CLIENT) &&
        (FindAttribute(ATTR_IN)))
        return true;
    if (dynamic_cast<CBEComponent*>(pFile->GetTarget()))
    {
        /* look for an OUT parameter */
        vector<CBETypedDeclarator*>::iterator iter = GetFirstParameter();
        CBETypedDeclarator *pParameter;
        while ((pParameter = GetNextParameter(iter)) != 0)
        {
            if (pParameter->FindAttribute(ATTR_OUT))
                return true;
        }
        /* look for return type */
        if (GetReturnType() &&
            !GetReturnType()->IsVoid())
            return true;
        /* check for exceptions */
        if (!FindAttribute(ATTR_NOEXCEPTIONS))
            return true;
    }
    return false;
}

/** \brief test if this function should be written
 *  \param pFile the file to write to
 *  \return true if should be written
 *
 * A marshal function is written if client's side and IN or if component's side
 * and one of the parameters has an OUT or we have an exception to transmit.
 */
bool
CBEMarshalFunction::DoWriteFunction(CBEImplementationFile * pFile)
{
    if (!IsTargetFile((CBEImplementationFile*)pFile))
        return false;
    if (pFile->IsOfFileType(FILETYPE_CLIENT) &&
        (FindAttribute(ATTR_IN)))
        return true;
    if (dynamic_cast<CBEComponent*>(pFile->GetTarget()))
    {
        /* look for an OUT parameter */
        vector<CBETypedDeclarator*>::iterator iter = GetFirstParameter();
        CBETypedDeclarator *pParameter;
        while ((pParameter = GetNextParameter(iter)) != 0)
        {
            if (pParameter->FindAttribute(ATTR_OUT))
                return true;
        }
        /* look for return type */
        if (GetReturnType() &&
            !GetReturnType()->IsVoid())
            return true;
        /* check for exceptions */
        if (!FindAttribute(ATTR_NOEXCEPTIONS))
            return true;
    }
    return false;
}
