/**
 *    \file    dice/src/be/BEFile.h
 *    \brief   contains the declaration of the class CBEFile
 *
 *    \date    01/10/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

/** preprocessing symbol to check header file */
#ifndef __DICE_BEFILE_H__
#define __DICE_BEFILE_H__

#include "File.h"
#include <vector>
using namespace std;

class CBEFunction;
class CBETarget;
class CBEClass;
class CBENameSpace;

class CFEOperation;
class CFEInterface;
class CFELibrary;
class CFEFile;
class CIncludeStatement;

/**    \class CBEFile
 *    \ingroup backend
 *    \brief the output file class for the back-end classes
 *
 * This class unifies the common properties of the class CBEHeaderFile and
 * CBEImplementationFile. These common properties are that both contain
 * functions and include files.
 */
class CBEFile : public CFile
{
// Constructor
public:
    /**    \brief constructor
     */
    CBEFile();
    virtual ~CBEFile();

protected:
    /**    \brief copy constructor */
    CBEFile(CBEFile &src);

public:
    virtual CBETarget* GetTarget();

    virtual void Write() = 0;

    virtual bool CreateBackEnd(CFEOperation *pFEOperation, int nFileType) = 0;
    virtual bool CreateBackEnd(CFEInterface *pFEInterface,  int nFileType) = 0;
    virtual bool CreateBackEnd(CFELibrary *pFELibrary, int nFileType) = 0;
    virtual bool CreateBackEnd(CFEFile *pFEFile, int nFileType) = 0;

    CBEClass* FindClass(string sClassName);
    CBEClass* GetNextClass(vector<CBEClass*>::iterator &iter);
    vector<CBEClass*>::iterator GetFirstClass();
    void RemoveClass(CBEClass *pClass);
    void AddClass(CBEClass *pClass);

    CBENameSpace* FindNameSpace(string sNameSpaceName);
    CBENameSpace* GetNextNameSpace(vector<CBENameSpace*>::iterator &iter);
    vector<CBENameSpace*>::iterator GetFirstNameSpace();
    void RemoveNameSpace(CBENameSpace *pNameSpace);
    void AddNameSpace(CBENameSpace *pNameSpace);

    virtual void AddIncludedFileName(string sFileName, bool bIDLFile, 
	bool bIsStandardInclude, CObject* pRefObj = 0);
    virtual vector<CIncludeStatement*>::iterator GetFirstIncludeStatement();
    virtual CIncludeStatement* GetNextIncludeStatement(
	vector<CIncludeStatement*>::iterator & iter);

    CBEFunction* FindFunction(string sFunctionName);
    CBEFunction* GetNextFunction(vector<CBEFunction*>::iterator &iter);
    vector<CBEFunction*>::iterator GetFirstFunction();
    void RemoveFunction(CBEFunction *pFunction);
    void AddFunction(CBEFunction *pFunction);

    virtual bool IsOfFileType(int nFileType);
    virtual bool HasFunctionWithUserType(string sTypeName);

    virtual int GetSourceLineEnd();

protected:
    virtual void WriteFunction(CBEFunction *pFunction) = 0;
    virtual void WriteDefaultIncludes();
    virtual void WriteInclude(CIncludeStatement* pInclude);
    virtual void WriteNameSpace(CBENameSpace *pNameSpace) = 0;
    virtual void WriteClass(CBEClass *pClass) = 0;
    virtual void WriteIntro();
    virtual int GetFunctionCount(void);

    virtual void WriteHelperFunctions();

    virtual void CreateOrderedElementList(void);
    void InsertOrderedElement(CObject *pObj);

protected:
    /** \var vector<CIncludeStatement*> m_vIncludedFiles
     *  \brief contains the names of the included files
     *
     * This is an array of strings, because a file may not only include
     * CBEFile-represented files, but also other files. For instance C-header
     * files. Therefore it is more convenient to store the names of the files,
     * than to create CBEFiles for the C-header files as well.
     */
    vector<CIncludeStatement*> m_vIncludedFiles;
    /** \var vector<CBEClass*> m_vClasses
     *  \brief contains all classes, which belong to this file
     */
    vector<CBEClass*> m_vClasses;
    /** \var vector<CBENameSpace*> m_vNameSpaces
     *  \brief contains all namespaces, which belong to this file
     */
    vector<CBENameSpace*> m_vNameSpaces;
    /** \var vector<CBEFunction*> m_vFunctions
     *  \brief contains the functions, which belong to this file
     */
    vector<CBEFunction*> m_vFunctions;
    /** \var int m_nFileType
     *  \brief contains the type of the file
     */
    int m_nFileType;
    /** \var vector<CObject*> m_vOrderedElements
     *  \brief contains ordered list of elements
     */
    vector<CObject*> m_vOrderedElements;
};

#endif // !__DICE_BEFILE_H__
