/**
 * \file   dice/src/be/BEContext.cpp
 * \brief  contains the implementation of the class CBEContext
 *
 * \date   01/10/2002
 * \author Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/BEContext.h"
#include "be/BEFunction.h"
#include "be/BETypedDeclarator.h"
#include "be/BEDeclarator.h"
#include "Compiler.h"
#include <string.h>

CBEContext::CBEContext()
{
}

/** CBEContext destructor */
CBEContext::~CBEContext()
{
}

/** \brief writes the actually used malloc function
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 *
 * We use CORBA_alloc if it is explicetly forced, otherwise we use
 * the malloc member of the environment. If environments malloc use is
 * enforced it overrides other options.
 */
void CBEContext::WriteMalloc(CBEFile* pFile, CBEFunction* pFunction)
{
    bool bUseMalloc = !CCompiler::IsOptionSet(PROGRAM_FORCE_CORBA_ALLOC) |
	CCompiler::IsOptionSet(PROGRAM_FORCE_ENV_MALLOC);
    assert(pFunction);
    string sFuncName = pFunction->GetName();
    if (bUseMalloc)
    {
        CBETypedDeclarator* pEnv = pFunction->GetEnvironment();
        CBEDeclarator *pDecl = (pEnv) ? pEnv->GetDeclarator() : 0;
        if (pDecl)
        {
	    string sMalloc = "(" + pDecl->GetName();
            if (pDecl->GetStars())
		sMalloc += "->malloc)";
            else
		sMalloc += ".malloc)";
	    *pFile << sMalloc;
	    // if at component side we rely on the malloc member being set.
            if (CCompiler::IsWarningSet(PROGRAM_WARNING_PREALLOC))
                CCompiler::Warning("CORBA_Environment.malloc is used to set receive buffer in %s.",
                    sFuncName.c_str());
        }
        else
        {
            if (CCompiler::IsOptionSet(PROGRAM_FORCE_ENV_MALLOC))
                CCompiler::Warning("Using CORBA_alloc because function %s has no environment.",
                    sFuncName.c_str());
            if (CCompiler::IsWarningSet(PROGRAM_WARNING_PREALLOC))
                CCompiler::Warning("CORBA_alloc is used to set receive buffer in %s.",
                    sFuncName.c_str());
	    *pFile << "CORBA_alloc";
        }
    }
    else
    {
        if (CCompiler::IsWarningSet(PROGRAM_WARNING_PREALLOC))
            CCompiler::Warning("CORBA_alloc is used to set receive buffer in %s.", 
		sFuncName.c_str());
	*pFile << "CORBA_alloc";
    }
}

/** \brief writes the actual used free function
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 */
void CBEContext::WriteFree(CBEFile* pFile, CBEFunction* pFunction)
{
    bool bUseFree = !CCompiler::IsOptionSet(PROGRAM_FORCE_CORBA_ALLOC) |
	CCompiler::IsOptionSet(PROGRAM_FORCE_ENV_MALLOC);
    assert(pFunction);
    string sFuncName = pFunction->GetName();
    if (bUseFree)
    {
        CBETypedDeclarator* pEnv = pFunction->GetEnvironment();
        CBEDeclarator *pDecl = (pEnv) ? pEnv->GetDeclarator() : 0;
        if (pDecl)
        {
	    string sFree = "(" + pDecl->GetName();
            if (pDecl->GetStars())
                sFree += "->free)";
            else
		sFree += ".free)";
	    *pFile << sFree;
	    // at the component side we rely on the free member to be set.
            if (CCompiler::IsWarningSet(PROGRAM_WARNING_PREALLOC))
                CCompiler::Warning("CORBA_Environment.free is used to set receive buffer in %s.",
                    sFuncName.c_str());
        }
        else
        {
            if (CCompiler::IsOptionSet(PROGRAM_FORCE_ENV_MALLOC))
                CCompiler::Warning("Using CORBA_free because function %s has no environment.",
                    sFuncName.c_str());
            if (CCompiler::IsWarningSet(PROGRAM_WARNING_PREALLOC))
                CCompiler::Warning("CORBA_free is used to set receive buffer in %s.",
                    sFuncName.c_str());
            *pFile << "CORBA_free";
        }
    }
    else
    {
        if (CCompiler::IsWarningSet(PROGRAM_WARNING_PREALLOC))
            CCompiler::Warning("CORBA_free is used to set receive buffer in %s.",
		sFuncName.c_str());
        *pFile << "CORBA_free";
    }
}
