/**
 *    \file    dice/src/Compiler.h
 *  \brief   contains the declaration of the class CCompiler
 *
 *    \date    01/31/2001
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

/** preprocessing symbol to check header file */
#ifndef __DICE_COMPILER_H__
#define __DICE_COMPILER_H__

#include <stdio.h>
#include "defines.h"
#include <string>
#include <vector>
using namespace std;

#include "ProgramOptions.h" // needed for ProgramOptionType
#include "be/BENameFactory.h"
#include "be/BEClassFactory.h"

class CBERoot;
class CFEBase;
class CFEFile;
class CFELibrary;
class CFEInterface;
class CFEOperation;
class CBESizes;

#define USE_FE_NONE        0x0    /**< defines that no front-end is used */
#define USE_FE_DCE        0x1    /**< defines that the DCE front-end is used */
#define USE_FE_CORBA    0x2    /**< defines that the CORBA front-end is used */

/**
 *  \class CCompiler
 *  \brief is the place where all the problems start ;)
 *
 * This class contains the "logic" of the compiler. The main function creates
 * an object of this class and calls its methods.
 */
class CCompiler
{
public:
    /** creates a compiler object */
    CCompiler();
    ~ CCompiler();

    // Operations
public:
    void Write();
    void PrepareWrite();
    void ShowHelp(bool bShort = false);
    void ShowCopyright();
    void ParseArguments(int argc, char *argv[]);
    static void Error(const char *sMsg, ...);
    static void Warning(const char *sMsg, ...);
    static void GccError(CFEBase * pFEObject, int nLinenb, 
	const char *sMsg, ...);
    static void GccErrorVL(CFEBase * pFEObject, int nLinenb, 
	const char *sMsg, va_list vl);
    static void GccWarning(CFEBase * pFEObject, int nLinenb,
	const char *sMsg, ...);
    static void GccWarningVL(CFEBase * pFEObject, int nLinenb,
	const char *sMsg, va_list vl);
    void Parse();

    static bool IsOptionSet(unsigned int nRawOption);
    static bool IsOptionSet(ProgramOptionType nOption);
    static void ModifyOptions(unsigned int nAdd, unsigned int nRemove);
    static bool IsVerbose();
    static bool IsBackEndSet(unsigned long nOption);
    static bool IsWarningSet(unsigned long nLevel);
    static string GetIncludePrefix();
    static string GetFilePrefix();
    static string GetTraceClientFunc();
    static string GetTraceServerFunc();
    static string GetTraceMsgBufFunc();
    static string GetTraceLib();
    static int GetTraceMsgBufDwords();
    static string GetInitRcvStringFunc();
    static string GetOutputDir();
    static CBENameFactory *GetNameFactory();
    static CBEClassFactory *GetClassFactory();
    static CBESizes* GetSizes();
    static bool HasSymbol(string sSymbol);

protected:
    void Verbose(const char *sMsg, ...);
    void ShowVersion();
    void PrintDependencyTree(FILE * output, CFEFile * pFEFile);
    void PrintDependencies();
    void PrintGeneratedFiles(FILE * output, CFEFile * pFEFile);
    void PrintGeneratedFiles4File(FILE * output, CFEFile * pFEFile);
    void PrintGeneratedFiles4Library(FILE * output, CFEFile * pFEFile);
    void PrintGeneratedFiles4Library(FILE * output, CFELibrary * pFELibrary);
    void PrintGeneratedFiles4Library(FILE * output, CFEInterface * pFEInterface);
    void PrintGeneratedFiles4Interface(FILE * output, CFEFile * pFEFile);
    void PrintGeneratedFiles4Interface(FILE * output, 
	CFELibrary * pFEpFELibrary);
    void PrintGeneratedFiles4Interface(FILE * output, 
	CFEInterface * pFEInterface);
    void PrintGeneratedFiles4Operation(FILE * output, CFEFile * pFEFile);
    void PrintGeneratedFiles4Operation(FILE * output, CFELibrary * pFELibrary);
    void PrintGeneratedFiles4Operation(FILE * output, 
	CFEInterface * pFEInterface);
    void PrintGeneratedFiles4Operation(FILE * output, 
	CFEOperation * pFEOperation);
    void PrintDependentFile(FILE * output, string sFileName);

    void InitTraceLib(int argc, char *argv[]);
    
    static void AddSymbol(string sNewSymbol);
    static void SetOption(unsigned int nRawOption);
    static void SetOption(ProgramOptionType nOption);
    static void UnsetOption(unsigned int nRawOption);
    static void ModifyBackEnd(unsigned long nAdd, unsigned long nRemove);
    static void SetWarningLevel(unsigned long nLevel);
    static void UnsetWarningLevel(unsigned long nLevel);
    static void SetIncludePrefix(string sPrefix);
    static void SetFilePrefix(string sPrefix);
    static void SetTraceClientFunc(string sName);
    static void SetTraceServerFunc(string sName);
    static void SetTraceMsgBufFunc(string sName);
    static void SetTraceMsgBufDwords(int nDwords);
    static void SetTraceLib(string sName);
    static void SetInitRcvStringFunc(string sName);
    static void SetOutputDir(string sOutputDir);
    static void SetNameFactory(CBENameFactory *pNF);
    static void SetClassFactory(CBEClassFactory *pCF);
    static void SetSizes(CBESizes *pSizes);

    // Attributes
protected:
    /** \var int m_nVerboseLevel
     *  \brief the verbose level
     */
    int m_nVerboseLevel;
    /** \var unsigned long m_nWarningLevel
     *  \brief the warning level
     */
    static unsigned long m_nWarningLevel;
    /** \var int m_nUseFrontEnd
     *  \brief contains the front-end to use
     */
    int m_nUseFrontEnd;
    /** \var CBERoot* m_pRootBE
     *  \brief the root point of the back end
     */
    CBERoot *m_pRootBE;
    /** \var CFEFile *m_pRootFE
     *  \brief a reference to the top most file of the front-end
     */
    CFEFile *m_pRootFE;
    /** \var string m_sIncludePrefix
     *  \brief the include prefix, which is used to prefix each include statement
     */
    static string m_sIncludePrefix;
    /** \var string m_sFilePrefix
     *  \brief the file prefix, which all files are supposed to receive
     */
    static string m_sFilePrefix;
    /** \var string m_sOutFileName
     *  \brief the name of the output file
     */
    string m_sOutFileName;
    /** \var string m_sInFileName
     *  \brief the name of the input file, NULL if stdin is used
     */
    string m_sInFileName;
    /** \var unsigned int m_nOptions[PROGRAM_OPTION_GROUPS]
     *  \brief the options which are specified with the compiler call
     */
    static unsigned int m_nOptions[PROGRAM_OPTION_GROUPS];
    /** \var unsigned m_nBackEnd
     *  \brief the back-end, which are specified with the compiler call
     */
    static unsigned m_nBackEnd;
    /** \var CBESizes *m_pSizes
     *  \brief contains a reference to the sizes class of a traget architecture
     */
    static CBESizes *m_pSizes;
    /** \var int m_nOpcodeSize
     *  \brief contains the size of the opcode type in bytes
     */
    int m_nOpcodeSize;
    /** \var bool m_bVerbose
     *  \brief is the same as (m_nOptions & PROGRAM_VERBOSE)
     */
    bool m_bVerbose;
    /** \var int m_nCurCol
     *  \brief is the current column for dependency output
     */
    int m_nCurCol;
    /** \var string m_sIncludePaths[MAX_INCLUDE_PATHS]
     *  \brief contains the include paths
     */
    string m_sIncludePaths[MAX_INCLUDE_PATHS];
    /** \var string m_sInitRcvStringFunc
     *  \brief contains the name of the init-rcv-string function
     */
    static string m_sInitRcvStringFunc;
    /** \var string m_sTraceClientFunc
     *  \brief contains the name of the Trace function for the client
     */
    static string m_sTraceClientFunc;
    /** \var string m_sTraceServerFunc
     *  \brief contains the name of the Trace function for the server
     */
    static string m_sTraceServerFunc;
    /** \var string m_sTraceMsgBuf
     *  \brief contains the name of the Trace function for the message buffer
     */
    static string m_sTraceMsgBufFunc;
    /** \var string m_sTraceLib
     *  \brief contains the file name of the tracing library
     */
    static string m_sTraceLib;
    /** \var string m_sOutputDir
     *  \brief determines the output directory
     */
    static string m_sOutputDir;
    /** \var vector<string> m_vSymbols
     *  \brief list of symbols defined when calling dice
     */
    static vector<string> m_vSymbols;
    /** \var int m_nDumpMsgBufDwords
     *  \brief contains the number of dwords to dump
     */
    static int m_nDumpMsgBufDwords;
    /** \var CBEClassFactory *m_pClassFactory
     *  \brief a reference to the class factory
     */
    static CBEClassFactory *m_pClassFactory;
    /** \var CBENameFactory *m_pNameFactory
     *  \brief a reference to the name factory
     */
    static CBENameFactory *m_pNameFactory;
};

/** \brief set the option
 *  \param nRawOption the option in raw format
 */
inline 
void 
CCompiler::SetOption(unsigned int nRawOption)
{
    m_nOptions[PROGRAM_OPTION_GROUP_INDEX(nRawOption)] |=
        PROGRAM_OPTION_OPTION(nRawOption);
}

/** \brief set the option
 *  \param nOption the option in ProgramOptionType format
 */
inline
void 
CCompiler::SetOption(ProgramOptionType nOption)
{
    m_nOptions[nOption._s.group] |= nOption._s.option;
}

/** \brief unset an option
 *  \param nRawOption the option in raw format
 */
inline
void
CCompiler::UnsetOption(unsigned int nRawOption)
{
    m_nOptions[PROGRAM_OPTION_GROUP_INDEX(nRawOption)] &=
        ~PROGRAM_OPTION_OPTION(nRawOption);
}

/** \brief tests if an option is set
 *  \param nRawOption the option in raw format
 */
inline
bool
CCompiler::IsOptionSet(unsigned int nRawOption)
{
    return m_nOptions[PROGRAM_OPTION_GROUP_INDEX(nRawOption)] &
        PROGRAM_OPTION_OPTION(nRawOption);
}

/** \brief check if a specific option is set
 *  \param nOption the value of the option to check
 *  \return true if the option is set, false if not
 */
inline
bool
CCompiler::IsOptionSet(ProgramOptionType nOption)
{
    return m_nOptions[nOption._s.group] & nOption._s.option;
}

/** \brief modifies the options
 *  \param nAdd the options to add
 *  \param nRemove the options to remove
 */
inline
void
CCompiler::ModifyOptions(unsigned int nAdd,
    unsigned int nRemove)
{
    UnsetOption(nRemove);
    SetOption(nAdd);
}

/** \brief check if the verbose option is set
 *  \return true if verbose is set
 */
inline
bool
CCompiler::IsVerbose()
{
    return IsOptionSet(PROGRAM_VERBOSE);
}

/** \brief modifies the back-end option
 *  \param nAdd the options to add
 *  \param nRemove the optiosn to remove
 */
inline
void
CCompiler::ModifyBackEnd(unsigned long nAdd, unsigned long nRemove)
{
    m_nBackEnd = ((m_nBackEnd & ~nRemove) | nAdd);
}

/** \brief test the back-end option
 *  \param nOption the option to test
 *  \return true if option is set
 */
inline
bool
CCompiler::IsBackEndSet(unsigned long nOption)
{
    return (m_nBackEnd & nOption) > 0;
}

/** \brief set a warning level
 *  \param nLevel the new warning level to set
 */
inline
void
CCompiler::SetWarningLevel(unsigned long nLevel)
{
    m_nWarningLevel |= nLevel;
}

/** \brief unset a warning level
 *  \param nLevel the warning level to delete
 */
inline
void
CCompiler::UnsetWarningLevel(unsigned long nLevel)
{
    m_nWarningLevel &= ~nLevel;
}

/** \brief test for a specific warning level
 *  \param the level to test for
 *  \return true if level is set
 */
inline
bool
CCompiler::IsWarningSet(unsigned long nLevel)
{
    return (m_nWarningLevel & nLevel) > 0;
}

/** \brief retrieve include prefix
 *  \return include prefix
 */
inline
string
CCompiler::GetIncludePrefix()
{
    return m_sIncludePrefix;
}

/** \brief set include prefix
 *  \param sPrefix the new include prefix
 */
inline
void
CCompiler::SetIncludePrefix(string sPrefix)
{
    m_sIncludePrefix = sPrefix;
}

/** \brief retrieve file prefix
 *  \return file prefix
 */
inline
string
CCompiler::GetFilePrefix()
{
    return m_sFilePrefix;
}

/** \brief set file prefix
 *  \param sPrefix the new file prefix
 */
inline
void
CCompiler::SetFilePrefix(string sPrefix)
{
    m_sFilePrefix = sPrefix;
}

/** \brief sets the name of the trace lib
 *  \param sName the name
 */
inline
void 
CCompiler::SetTraceLib(string sName)
{
    m_sTraceLib = sName;
}

/** \brief retrieves the name of the Trace lib
 *  \return the name
 */
inline
string
CCompiler::GetTraceLib()
{
    return m_sTraceLib;
}

/** \brief sets the name of the trace function
 *  \param sName the name
 */
inline
void 
CCompiler::SetTraceClientFunc(string sName)
{
    m_sTraceClientFunc = sName;
}

/** \brief retrieves the name of the Trace function
 *  \return the name
 */
inline
string
CCompiler::GetTraceClientFunc()
{
    return m_sTraceClientFunc;
}

/** \brief sets the name of the trace function
 *  \param sName the name
 */
inline
void
CCompiler::SetTraceServerFunc(string sName)
{
    m_sTraceServerFunc = sName;
}

/** \brief retrieves the name of the Trace function
 *  \return the name
 */
inline
string
CCompiler::GetTraceServerFunc()
{
    return m_sTraceServerFunc;
}

/** \brief sets the name of the trace function
 *  \param sName the name
 */
inline
void
CCompiler::SetTraceMsgBufFunc(string sName)
{
    m_sTraceMsgBufFunc = sName;
}

/** \brief retrieves the name of the Trace function
 *  \return the name
 */
inline
string
CCompiler::GetTraceMsgBufFunc()
{
    return m_sTraceMsgBufFunc;
}

/** \brief sets the number of dwords to dump from the message buffer
 *  \param nDwords the number of dwords
 */
inline
void
CCompiler::SetTraceMsgBufDwords(int nDwords)
{
    m_nDumpMsgBufDwords = nDwords;
}

/** \brief get the number of dumpable dwords
 *  \return number of dwords
 */
inline
int
CCompiler::GetTraceMsgBufDwords()
{
    return m_nDumpMsgBufDwords;
}

/** \brief sets the name of the init-rcv-string function
 *  \param sName the name
 */
inline
void
CCompiler::SetInitRcvStringFunc(string sName)
{
    m_sInitRcvStringFunc = sName;
}

/** \brief retrieves the name of the init-rcv-string function
 *  \return the name
 */
inline
string
CCompiler::GetInitRcvStringFunc()
{
    return m_sInitRcvStringFunc;
}

/** \brief returns the output directory
 *  \return the output directory
 */
inline
string
CCompiler::GetOutputDir()
{
    return m_sOutputDir;
}

/** \brief set the output directory
 *  \param sOutputDir the directory
 */
inline
void 
CCompiler::SetOutputDir(string sOutputDir)
{
    m_sOutputDir = sOutputDir;
}

/** \brief returns a reference to the class factory
 *  \return the reference to the class-factory
 */
inline
CBEClassFactory*
CCompiler::GetClassFactory()
{
    return m_pClassFactory;
}

/** \brief set the class factory member
 *  \param pCF the new class factory variable
 *
 * Previously set class factory member is deleted.
 */
inline
void 
CCompiler::SetClassFactory(CBEClassFactory *pCF)
{
    if (m_pClassFactory)
	delete m_pClassFactory;
    m_pClassFactory = pCF;
}

/** \brief returns a reference to the name factory
 *  \return the reference to the name factory
 */
inline
CBENameFactory*
CCompiler::GetNameFactory()
{
    return m_pNameFactory;
}

/** \brief sets the name factory member
 *  \param pNF the new name factory variable
 *
 * previously set name factory member is deleted.
 */
inline
void 
CCompiler::SetNameFactory(CBENameFactory *pNF)
{
    if (m_pNameFactory)
	delete m_pNameFactory;
    m_pNameFactory = pNF;
}

/** \brief restrieves a reference to the sizes class
 *  \return a reference to the sizes class
 */
inline
CBESizes*
CCompiler::GetSizes()
{
    return m_pSizes;
}

/** \brief sets the sizes member
 *  \param pSizes the new sizes member
 */
inline
void
CCompiler::SetSizes(CBESizes *pSizes)
{
    m_pSizes = pSizes;
}

#endif                // __DICE_COMPILER_H__
