#include <l4/sys/ipc.h>
#include <l4/sys/types.h>
#include <l4/sys/syscalls.h>
#include <l4/sys/compiler.h>

#undef EXPECT_TRUE
#undef EXPECT_FALSE

#include <l4_types.h>
#include <syscall-user.h>
#include <v2cs.h>
#include <l4_types.h>
#include <stdio.h>
#include "debug_glue.h"


const Cap_id TASK_SELF = 0;

void panic(const char *msg)
{
  puts(msg);
  asm("int3");
  while(1);
}

extern "C"
void
outnstring(const char *value, int count)
{
        __libc_backend_outs(value,count);
}

Address v2emu_ep() { return V2_cs::get_cap
    (V2_cs::Thread_cap_open_ep, V2_cs::V2emu_task_num, V2_cs::V2emu_thread_num);
}

Fpage_Xe kmem() { return V2_cs::get_kmem(); }



static inline void tid2thread_task(Mword id, Mword *task, Mword *thread)
{
  assert(thread);
  assert(task);

  *task    = (id >>7) & 0x7ff;
  *thread = id & 0x7f;
}

static inline l4_threadid_t tid2id(Mword id)
{
  l4_threadid_t tmp = (l4_threadid_t){{0, 0}};
  tmp.id.task = (id >>7) & 0x7ff;  
  tmp.id.lthread = id & 0x7f;
  return tmp;
}

static inline Address tid2cap(Mword id, Mword type)
{
  Address thread, task;

  tid2thread_task(id, &task, &thread);
  //  return V2_cs::get_cap(V2_cs::Thread_cap_thread, task, thread);
  return V2_cs::get_cap(type, task, thread);
}

static inline Address get_own_cap(Mword type)
{

  Mword id;
  if (Syscall_user::ex_regs_get_tid(&id))
    panic("wrapper: Could not get TID");

  return tid2cap(id, type);   
}


#if 0
static inline l4_threadid_t cap2id(Address cap, unsigned type)
{
  Address thread, task;

  tid2thread_task(id, &task, &thread);
  //  return V2_cs::get_cap(V2_cs::Thread_cap_thread, task, thread);
  return V2_cs::get_cap(type, task, thread);
}
#endif

// dont handle grant
static inline Fpage_Xe fpagev2_to_fpagexe(l4_fpage_t page)
{
  printf("\nfpage addr %#x size: %d write: %s\n", 
	 page.fp.page, page.fp.size, page.fp.write ? "yes" : "no");
  
  return Fpage_Xe(page.fp.page << L4_PAGESHIFT, page.fp.size, Fpage_Xe::Type_mem, 
		  page.fp.write ? Capability::Write : 0 |
		  Capability::Read | Capability::Execute);
  asm("int3");
}

static inline Fpage_Xe fpagev2_to_fpagexe(Mword raw_)
{
  return fpagev2_to_fpagexe((l4_fpage_t){raw:raw_});
}




static inline unsigned convert_xe_error_code(Error::Errors res)
{
  // XXX use an translation table
  // XXX Xe needs much better and finer grained errorcodes
  switch (res) {
  case Error::No:
    return 0;
  case Error::Exist:
    return L4_IPC_ENOT_EXISTENT;
  case Error::Cancel:
    return L4_IPC_SECANCELED;
  case Error::Timeout:
    return L4_IPC_SETIMEOUT;

  case Error::Exist_rcv:
    return L4_IPC_ENOT_EXISTENT;
  case Error::Cancel_rcv:
    return L4_IPC_RECANCELED;
  case Error::Timeout_rcv:
    return L4_IPC_RETIMEOUT;

  default:
    printf("untranslatable ipc error %d\n", res);
    panic("untranslatable ipc error");
  }
  return 0;
}

static inline l4_msgdope_t ipc_v2_msg_dope(unsigned rcv_fpage,
						  unsigned error_code,
						  unsigned strings,
						  unsigned dwords)
{
  return ( (l4_msgdope_t) {md: {0, rcv_fpage, 0, 0, 0, error_code, strings, dwords }});  
}

static inline l4_msgdope_t convert_ipc_result(IPC_result res)
{
  return ipc_v2_msg_dope(res.rcv_typed() ? 1 : 0, convert_xe_error_code(res.error()),
			 0, 2);
}




static inline void back_pt(Address start, Address end)
{
  printf("back_pt from %#lx to %#lx\n", start, end);
  start = start >> L4_SUPERPAGESHIFT;
  end   = end >> L4_SUPERPAGESHIFT;
  
  for(;start <= end; start++)
    {

#if 0
      printf("Syscall_user::map(%lx,%lx,%lx,%lx,%x,%x)", TASK_SELF,
	  kmem().raw(), TASK_SELF, 
	  Fpage_Xe(start << L4_SUPERPAGESHIFT,
		   L4_SUPERPAGESHIFT,0,15).raw(), 1, 0);
#endif
      Mword res;
      if ((res = Syscall_user::map(TASK_SELF, kmem(), 
				   TASK_SELF,
				   Fpage_Xe(start << L4_SUPERPAGESHIFT,
					    L4_SUPERPAGESHIFT,0,15),
				   1)))
	{
	  log(1,"Pager: pagetable backing for %lx failed with %ld", 
	      start<< L4_SUPERPAGESHIFT, res);
	  panic("failed with pagetable backing");
	  assert(0);
	}
    }
}

static inline void back_pt(Fpage_Xe addr)
{
  assert(addr.type() == Fpage_Xe::Type_mem);
  back_pt(addr.page(), addr.page() + (1 << addr.size()));
}


static void sync_capspace(Fpage_Xe addr)
{
  assert(addr.type() == Fpage_Xe::Type_cap);

  Mword m0, m1, m2;
  unsigned map = 1;
  Cap_id rcv_ep = get_own_cap(V2_cs::Thread_cap_close_ep);

  IPC_result res = Syscall_user::short_ipc
    (v2emu_ep(), V2_cs::Magic_request_mask | V2_cs::Sync_CapSpace, addr.raw(), 0,
     rcv_ep, &m0, &m1, &m2,
     0 /* badge */, 0 /* snd_map */, &map);

  if(res.error())
    {
      printf("error call to v2emu res: %ld\n", res.raw());
      panic("ipc error");
    }
}



l4_threadid_t
l4_myself(void) 
{
  Mword id;
  
  if (Syscall_user::ex_regs_get_tid(&id))
    panic("Hello: Could not get TID");
 
  return tid2id(id);
}

void
__do_l4_thread_ex_regs(l4_umword_t val0,
                       l4_umword_t eip,
                       l4_umword_t esp, 
                       l4_threadid_t *preempter,
                       l4_threadid_t *pager,
                       l4_umword_t *old_eflags,
                       l4_umword_t *old_eip,
                       l4_umword_t *old_esp)
{
  
  printf("\nwrapper: ex_regs %lu %#lx %#lx\n", val0, eip, esp);

  Mword local_cap = (V2_cs::get_local_cap(V2_cs::Thread_cap_thread, val0));
  Mword res;
  Address old_pager_id;

  res = Syscall_user::get_thread_ctrl(local_cap, 1,
				      Number::TCR_pager_snd, &old_pager_id);  

  // no mapping exists, ask v2emu for syncing the cap space
  // argh buggy cap space impl.
  if(res == Error::Exist || res == Error::Type)
    {

      printf("\nwrapper::request: ex_regs %lu (%#lx) %#lx %#lx\n", val0, local_cap, eip, esp);
      Mword m0, m1, m2;

      Address rcv_ep = get_own_cap(V2_cs::Thread_cap_close_ep);

      IPC_result res = Syscall_user::short_ipc(v2emu_ep(), 
					       V2_cs::Magic_request_mask | V2_cs::Create_thread, val0, 0,
					       rcv_ep, &m0, &m1, &m2);

      printf("\nvwrapper:.request end %ld\n", res.raw());
    }

  // XXX if the pager is not mapped, request in!!
  // 
  if(!l4_is_invalid_id(*pager))
    { 
      Address pager_id = V2_cs::get_cap(V2_cs::Thread_cap_open_ep, (*pager).id.task, (*pager).id.lthread);
      printf("setting pager snd ep to %#lx  %x.%x\n", pager_id, (*pager).id.task, (*pager).id.lthread );
      res = Syscall_user::set_thread_ctrl(local_cap, 1,
					  Number::TCR_pager_snd, pager_id);
    }
  printf("\nwrapper: old pager id %#lx\n", old_pager_id);
  
  if(old_pager_id)
    *pager = tid2id(V2_cs::get_id(old_pager_id, V2_cs::Thread_cap_open_ep));
  else
    *pager = L4_NIL_ID;

  *preempter = L4_NIL_ID;
  
  Mword old_tid;
  Syscall_user::ex_regs(local_cap, 
			(Address) eip, (Address) esp, (Mword) ~0UL, (Mword) ~0UL,
			(Address*) old_eip, (Address*) old_esp, (Mword*) old_eflags, &old_tid, 
			//			(val0 == 0) || (val0 == 1) ? false : true);
			false);

  //  asm("int3");
}



static void prepare_receive(void *rcv_msg, l4_timeout_t timeout, 
			    Mword &rcv_id, unsigned &rcv_to, unsigned &rcv_count, unsigned &rcv_map)
{

  Mword id;

  if (Syscall_user::ex_regs_get_tid(&id))
    panic("wrapper: Could not get TID");

  Cap_id thread_cap = tid2cap(id, V2_cs::Thread_cap_thread);

  Mword rcv_desc = ((Mword) rcv_msg) & ~L4_IPC_OPEN_IPC;

  if(rcv_desc == 0)
    {
      rcv_map = 0;
      rcv_count = 3;
    }
  else if(rcv_desc & (Mword) L4_IPC_SHORT_FPAGE)
    {
      rcv_map = 1;
      rcv_count = 3;
      
      Fpage_Xe rcv_window = fpagev2_to_fpagexe((Mword) rcv_msg);      
      
      back_pt(rcv_window);
      // set receive window
      Syscall_user::set_thread_ctrl(thread_cap, 1,
				    Number::TCR_mem_rcv_window, rcv_window.raw());
    }
  //XXX combine timeout and rcv-window to one syscall, use self
  unsigned to = 0;
  if(timeout.to.rcv_exp)
    {
      Syscall_user::set_thread_ctrl(thread_cap, 1,
				    Number::TCR_timeout0, 
				    ((15-timeout.to.rcv_exp+1 + 4) <<26) 
				    + (timeout.to.rcv_man<<16));
      to = 2;
    }

  rcv_id = id;  
  rcv_to = to;

}

int
l4_ipc_send(l4_threadid_t dest,
            const void *snd_msg,
            l4_umword_t snd_dword0,
            l4_umword_t snd_dword1,
            l4_timeout_t timeout,
            l4_msgdope_t *result)
{

  Address snd_cap = V2_cs::get_cap(V2_cs::Thread_cap_open_ep, dest.id.task, dest.id.lthread);
  printf("\nwrapper::ipc_send to %lx (%x.%x)\n", snd_cap, dest.id.task, dest.id.lthread);

  Mword res = Syscall_user::send_short_ipc(snd_cap, 0, snd_dword0, snd_dword1);
  
  printf("\nwrapper::ipc_send res: %ld\n", res);
  *result = L4_IPC_DOPE(0, 0);
  
  //  asm("int3");

  return 0;
}


int
l4_ipc_receive(l4_threadid_t src,
               void *rcv_msg,
               l4_umword_t *rcv_dword0,
               l4_umword_t *rcv_dword1,
               l4_timeout_t timeout,
               l4_msgdope_t *result)
{

  Mword id;
  unsigned to;
  unsigned map;
  unsigned count;

  prepare_receive(rcv_msg, timeout, id, to, count, map);
  
  Mword dummy;
  Badge recv_badge;
  IPC_result res = Syscall_user::recv_short_ipc(tid2cap(id, V2_cs::Thread_cap_close_ep), 
					   &dummy, (Mword *)rcv_dword0, (Mword *) rcv_dword1, &recv_badge, 0 ,to);

  l4_threadid_t recv_id = tid2id(recv_badge.raw());
  printf("\nwrapper::ipc recv from %#lx (%x.%x) res %lx\n", 
	 recv_badge.raw(), recv_id.id.task, recv_id.id.lthread, res.raw());

  if (!res.error())
    assert(recv_id.id.task == src.id.task && recv_id.id.lthread ==  src.id.lthread);

  *result = convert_ipc_result(res);
  return L4_IPC_ERROR(*result);
}


int
l4_ipc_wait(l4_threadid_t *src,
            void *rcv_msg,
            l4_umword_t *rcv_dword0,
            l4_umword_t *rcv_dword1,
            l4_timeout_t timeout,
            l4_msgdope_t *result)
{

  printf("\nwrapper::ipc_wait\n");  

  Mword id;
  unsigned to;
  unsigned map;
  unsigned count;
  
  prepare_receive(rcv_msg, timeout, id, to, count, map);

  Mword dummy;
  Badge recv_badge;
  
  IPC_result res = Syscall_user::recv_short_ipc(tid2cap(id, V2_cs::Thread_cap_open_ep), 
					   &dummy, (Mword *)rcv_dword0, (Mword *) rcv_dword1, &recv_badge, 0 ,to);

  l4_threadid_t recv_id = tid2id(recv_badge.raw());
  printf("\nwrapper::ipc wait from %#lx (%x.%x) res %lx\n", 
	 recv_badge.raw(), recv_id.id.task, recv_id.id.lthread, res.raw());

  *src = recv_id;
  *result = convert_ipc_result(res);

  return L4_IPC_ERROR(*result);
}

int
l4_ipc_reply_and_wait(l4_threadid_t dest,
                      const void *snd_msg,
                      l4_umword_t snd_dword0,
                      l4_umword_t snd_dword1,
                      l4_threadid_t *src,
                      void *rcv_msg,
                      l4_umword_t *rcv_dword0,
                      l4_umword_t *rcv_dword1,
                      l4_timeout_t timeout,
                      l4_msgdope_t *result)
{
  printf("\nwrapper::ipc_reply_wait call to %x.%x w0: %#lx w1: %#lx\n",
	 dest.id.task, dest.id.lthread, snd_dword0, snd_dword1);

  Address snd_cap = V2_cs::get_cap(V2_cs::Thread_cap_open_ep, dest.id.task, dest.id.lthread);

  Mword id;
  unsigned to;
  unsigned map;
  unsigned count;
  
  prepare_receive(rcv_msg, timeout, id, to, count, map);

  Mword dummy;
  Badge recv_badge;

  // XXX fr rcv EP waere es echt suess, wenn man dort ein TCR angeben kann
  // spart syscalls
  IPC_result res = Syscall_user::short_ipc(snd_cap, 0, snd_dword0, snd_dword1,
					   tid2cap(id, V2_cs::Thread_cap_open_ep), 
					   &dummy, (Mword *)rcv_dword0, (Mword *) rcv_dword1,
					   &recv_badge, 0 , &map, to);

  l4_threadid_t recv_id = tid2id(recv_badge.raw());

  printf("\nwrapper::reply_wait from %#lx (%x.%x) res %lx\n", 
	 recv_badge.raw(), recv_id.id.task, recv_id.id.lthread, res.raw());

  *src = recv_id;
  *result = convert_ipc_result(res);

  return L4_IPC_ERROR(*result);
}


int
l4_ipc_call(l4_threadid_t dest,
            const void *snd_msg,
            l4_umword_t snd_dword0,
            l4_umword_t snd_dword1,
            void *rcv_msg,
            l4_umword_t *rcv_dword0,
            l4_umword_t *rcv_dword1,
            l4_timeout_t timeout,
            l4_msgdope_t *result)
{
  printf("\nwrapper::ipc_call to %x.%x w0: %#lx w1: %#lx\n",
	 dest.id.task, dest.id.lthread, snd_dword0, snd_dword1);

  Address snd_cap = V2_cs::get_cap(V2_cs::Thread_cap_open_ep, dest.id.task, dest.id.lthread);

  Mword id;
  unsigned to;
  unsigned map;
  unsigned count;

  prepare_receive(rcv_msg, timeout, id, to, count, map);

  Mword dummy;
  Badge recv_badge;  

  IPC_result res = Syscall_user::short_ipc(snd_cap, V2_cs::Magic_request_mask | V2_cs::V2_Sigma0_Request,
					   snd_dword0, snd_dword1,
					   tid2cap(id, V2_cs::Thread_cap_close_ep), 
					   &dummy, (Mword *)rcv_dword0, (Mword *) rcv_dword1,
					   &recv_badge, 0 , &map, to);

  l4_threadid_t recv_id = tid2id(recv_badge.raw());

  printf("\nwrapper::call from %#lx (%x.%x) res %lx\n", 
	 recv_badge.raw(), recv_id.id.task, recv_id.id.lthread, res.raw());

  assert(recv_id.id.task == dest.id.task && recv_id.id.lthread ==  dest.id.lthread);

  *result = convert_ipc_result(res);
  asm("int3");
  return L4_IPC_ERROR(*result);
}

l4_cpu_time_t
l4_thread_schedule(l4_threadid_t dest,
		   l4_sched_param_t param,
		   l4_threadid_t *ext_preempter,
		   l4_threadid_t *partner,
		   l4_sched_param_t *old_param)
{
  printf("\nwrapper::schedule from %x.%x\n",
	 dest.id.task, dest.id.lthread);
  asm("int3");
  return 0;
}

l4_taskid_t
l4_task_new(l4_taskid_t destination,
	    l4_umword_t mcp_or_new_chief,
	    l4_umword_t esp,
	    l4_umword_t eip,
	    l4_threadid_t pager)
{
  printf("\nwrapper::task_new %x\n",destination.id.task);
  asm("int3");
  
  return L4_NIL_ID;
}


void
l4_thread_switch(l4_threadid_t destination)
{
  printf("\nwrapper::switch from %x.%x\n",
	 destination.id.task, destination.id.lthread);
  asm("int3");
}


