/* $Id: startup.c,v 1.54 2006/02/24 16:45:47 adam Exp $ */
/**
 * \file	bootstrap/server/src/startup.c
 * \brief	Main functions
 * 
 * \date	09/2004
 * \author	Torsten Frenzel <frenzel@os.inf.tu-dresden.de>,
 * 		Frank Mehnert <fm3@os.inf.tu-dresden.de>,
 * 		Adam Lackorzynski <adam@os.inf.tu-dresden.de> */

/* (c) 2005 Technische Universitaet Dresden
 * This file is part of DROPS, which is distributed under the terms of the
 * GNU General Public License 2. Please see the COPYING file for details. */

/* LibC stuff */
#define _GNU_SOURCE
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>

/* L4 stuff */
#include <l4/sys/compiler.h>
#include <l4/sys/kernel.h>
#include <l4/sys/kdebug.h>
#include <l4/util/mb_info.h>

#ifndef USE_OSKIT
#include <l4/env_support/panic.h>
#endif

/* local stuff */
#include "exec.h"
#include "macros.h"
#include "version.h"
#include "region.h"
#include "module.h"
#include "startup.h"
#include "init.h"
#include "patch.h"
#if defined (ARCH_x86) || defined(ARCH_amd64)
#include "ARCH-x86/serial.h"
#endif

#if defined(REALMODE_LOADING) || defined(XEN) || defined(ARCH_arm)
#include "loader_mbi.h"
#endif

#undef getchar


#ifndef MEMORY
#define MEMORY 64
#endif

extern int _start;	/* begin of image -- defined in crt0.S */
extern int _stack;	/* begin of stack -- defined by crt0.S */
extern int _end;	/* end   of image -- defined by bootstrap.ld */

/* temporary buffer for multiboot info */
static l4util_mb_info_t mb_info;
static char mb_cmdline[CMDLINE_MAX];
       l4util_mb_mod_t mb_mod[MODS_MAX];
static char mb_mod_names[4096];
static l4util_mb_vbe_ctrl_t mb_vbe_ctrl;
static l4util_mb_vbe_mode_t mb_vbe_mode;

const l4_addr_t _mod_addr = MODADDR;

/* hardware config variables */
int hercules = 0; /* hercules monitor attached */

/* kernel stuff variables */
static int l4_version = 0; /* L4 version number */
static int abi_version = 0; /* ABI version number/for v4 kip initialisation */

/* modules to load by bootstrap */
static const int kernel = 1; /* we have at least a kernel */
#ifdef XEN
static int sigma0 = 0; /* we need sigma0 */
static int roottask = 0; /* we need a roottask */
#else
static int sigma0 = 1; /* we need sigma0 */
static int roottask = 1; /* we need a roottask */
#endif

enum {
  kernel_module,
  sigma0_module,
  roottask_module,
};

/* RAM memory size */
static l4_addr_t mem_lower = 0;	/* lower bound of physical memory */
static l4_addr_t mem_upper = 0;	/* upper bound of physical memory */
static l4_addr_t mem_high = 0;	/* highest usable address */

static l4_addr_t mod_range_start = 0;
static l4_addr_t mod_range_end = 0;

static l4_addr_t bios_area_start = 0;
static l4_addr_t bios_area_end = 0;

static l4_addr_t bootstrap_start = 0;
static l4_addr_t bootstrap_end = 0;

/* kernel memmory*/
       l4_addr_t kernel_low = 0;
       l4_addr_t kernel_high = 0;
static l4_addr_t kernel_start = 0;

/* sigma0 memmory */
static l4_addr_t sigma0_low = 0;
static l4_addr_t sigma0_high = 0;
static l4_addr_t sigma0_start = 0;
static l4_addr_t sigma0_stack = 0;

/* roottask memmory */
static l4_addr_t roottask_low = 0;
static l4_addr_t roottask_high = 0;
static l4_addr_t roottask_start = 0;
static l4_addr_t roottask_stack = 0;

/* we define a small stack for sigma0 and roottask here -- it is used by L4 
 * for parameter passing. however, sigma0 and roottask must switch to its
 * own private stack as soon as it has initialized itself because this memory
 * region is later recycled in init.c */
static char roottask_init_stack[64]; /* XXX hardcoded */
static char sigma0_init_stack[64]; /* XXX hardcoded */

/* entry point */
void startup(l4util_mb_info_t *, l4_umword_t, void *);

static exec_read_func_t l4_exec_read;
static exec_read_exec_func_t l4_exec_read_exec;

/**
 * \brief  Detect L4 Version
 *
 * \return L4 Version:
 *         - #VERSION_L4_V2    old GMD L4 version
 *         - #VERSION_L4_IBM   IBM LN/X version
 *         - #VERSION_FIASCO   Fiasco
 *         - #VERSION_L4_KA    L4-KA/Hazelnut
 */
static int
check_l4_version(unsigned char *start, unsigned char *end, int *abi_version)
{
  unsigned char *p;

  printf("  Kernel at %08lx-%08lx\n", (l4_addr_t)start, (l4_addr_t)end);

  for (p = start; p < end; p++)
    {
      if (memcmp(p, "L4/486 \346-Kernel", 15) == 0)
	{
	  printf("  Detected L4/486\n");
	  /* remove old 4M event */
	  for (p=start; p<end; p++)
	    {
	      if (memcmp(p, "\314\353\0024M", 5) == 0)
		{
		  p[0] = 0x90; /* replace "int $3" by "nop" */
		  return VERSION_L4_V2;
		}
	    }
	  printf("  4M sequence not found in L4/486 -- that's OK.\n");
	  return VERSION_L4_V2;
	}

      if (memcmp(p, "L4/Pentium \346-Kernel", 19) == 0)
	{
	  /* it's Jochen's Pentium version */
	  printf("  Detected L4/Pentium\n");
	  return VERSION_L4_V2;
	}

      if (memcmp(p, "Lava Nucleus (Pentium)", 22) == 0)
	{
	  /* it's the IBM version */
	  printf("  Detected IBM LN/Pentium\n");
	  return VERSION_L4_IBM;
	}

      if (memcmp(p, "L4-X Nucleus (x86)", 18) == 0)
	{
	  /* it's the IBM X version */
	  printf("  Detected IBM X/Pentium\n");
#if !defined(L4_API_L4X0) && !defined(L4API_l4x0)
	  /* X.0 API, but not adaption enabled, fail */
	  panic("L4-X uses 32 Bit TIDs, but not adaption enabled");
#endif
	  return VERSION_L4_IBM;
	}

      if (memcmp(p, "DD-L4(v2)/x86 microkernel", 25) == 0 ||
	  memcmp(p, "DD-L4/x86 microkernel", 21) == 0)
	{
	  /* it's the Dresden version */
	  printf("  Detected new-style DD-L4(v2)/Fiasco x86\n");
#if defined(L4_API_L4X0) || defined(L4API_l4x0)
	  panic("Fiasco uses 64 Bit TIDs, but 32 Bit TID adaption enabled");
#endif
	  return VERSION_FIASCO;
	}

      if (memcmp(p, "DD-L4(x0)/x86 microkernel", 25) == 0)
	{
	  /* it's the Dresden version */
	  printf("  Detected new-style DD-L4(x0)/Fiasco x86\n");
#if !defined(L4_API_L4X0) && !defined(L4API_l4x0)
	  panic("Fiasco(x0) uses 32 Bit TIDs, but 64 Bit TID are used");
#endif
	  return VERSION_FIASCO;
	}

      if (memcmp(p, "DD-L4(v4)/x86 microkernel", 25) == 0)
	{
	  /* it's the Dresden version */
	  printf("  Detected very-new-style DD-L4(v4)/Fiasco\n");
	  *abi_version = VERSION_ABI_V4;
	  return VERSION_FIASCO;
	}

      if (memcmp(p, "DD-L4(x0)/arm microkernel", 25) == 0)
	{
	  printf("  Detected DD-L4(x0)/Fiasco ARM\n");
	  return VERSION_FIASCO;
	}

      if (memcmp(p, "L4/KA", 5) == 0 ||
	  memcmp(p, "L4Ka/Hazelnut", 13) == 0)
        {
	  /* it's the Karlsruhe version */
	  printf("  Detected L4Ka/Hazelnut\n");
#if !defined(L4_API_L4X0) && !defined(L4API_l4x0)
	  /* X.0 API, but not adaption enabled, fail */
	  panic("Hazelnut uses 32 Bit TIDs, but not adaption enabled");
#endif
	  return VERSION_L4_KA;
        }

      if (memcmp(p, "L4Ka::Pistachio ", 16) == 0)
        {
	  printf("  Detected L4Ka::Pistachio\n");
	  *abi_version = VERSION_ABI_V4;
	  return VERSION_L4_KA;
	}
    }

  printf("  could not detect version of L4 -- assuming Jochen.\n");
  return VERSION_L4_V2;
}

/**
 * \brief  Init Fiasco
 */
static void
init_fiasco(l4_kernel_info_t ** l4i)
{
  static char cmdline[256];
  char str_proto[32] = "";
  const char *c = get_args_module(kernel_module);
  const char *f = (const char *)((l4_addr_t)mb_mod[kernel_module].cmdline);
  const char *a = strchr(f, ' ');

  /* use an info page prototype allocated from our address space;
     this will be copied later into the kernel's address space */
  static char proto[L4_PAGESIZE] __attribute__((aligned(L4_PAGESIZE)));

  if (!a)
    a = f+strlen(f);

  memset(proto, 0, L4_PAGESIZE);
  *l4i = (l4_kernel_info_t *) proto;

#if defined(ARCH_x86) || defined(ARCH_amd64)
  sprintf(str_proto, " proto=0x%lx", (l4_addr_t) proto);
#endif

  /* append address of kernel info prototype to kernel's command line */
  snprintf(cmdline, sizeof(cmdline), "%.*s %s%s",
           (unsigned)(a-f), f, c ? c : a, str_proto);

  mb_mod[kernel_module].cmdline = (l4_addr_t)cmdline;
}

#if defined(ARCH_x86) || defined(ARCH_amd64)
/**
 * @brief Initialize KIP prototype for Fiasco/v4.
 */
static void
init_v4_kip (l4_kernel_info_t * l4i)
{
  l4_umword_t *p         = (l4_umword_t*) ((char*) l4i + V4KIP_MEM_DESC);
  l4_umword_t *p_meminfo = (l4_umword_t*) ((char*) l4i + V4KIP_MEM_INFO);

  // reserve sigma0 memory
  *(p++) = l4i->sigma0_memory.start | 3;
  *(p++) = l4i->sigma0_memory.end;

  // reserve roottask memory
  *(p++) = l4i->root_memory.start | 3;
  *(p++) = l4i->root_memory.end;

  // set the mem_info variable: offset and count of mem descriptors
  *p_meminfo = (((l4_umword_t) V4KIP_MEM_DESC) << 16)   // offset
    | 2;                                                // count
}
#endif

/**
 * setup memory range
 */
static void
setup_memory_range(l4util_mb_info_t *mbi)
{
  char *c;

  if (mbi->flags & L4UTIL_MB_MEMORY)
    {
      mem_upper = mbi->mem_upper;
      mem_lower = mbi->mem_lower;
    }

  /* maxmem= parameter? */
  if ((mbi->flags & L4UTIL_MB_CMDLINE) &&
      (c = strstr(L4_CHAR_PTR(mbi->cmdline), " -maxmem=")))
    mem_upper = 1024 * (strtoul(c + 9, NULL, 10) - 1);

  if (mbi->mem_upper != mem_upper)
    printf("  Limiting memory to %ld MB (%ld KB)\n",
	mem_upper/1024+1, mem_upper+1024);

  mem_high = (mb_info.mem_upper + 1024) << 10;

  /* ensure that we have a valid multiboot info structure */
  mbi->mem_upper = mem_upper;
  mbi->mem_lower = mem_lower;
}

/**
 * init memory map
 */
static void
init_memmap(l4util_mb_info_t *mbi)
{
#if defined(ARCH_x86) || defined(ARCH_amd64)
  bios_area_start = 0x09f000;
  bios_area_end   = 0x100000;
#endif

  bootstrap_start = (l4_addr_t)&_start;
  bootstrap_end   = (l4_addr_t)&_end;

  mod_range_start = (L4_MB_MOD_PTR(mbi->mods_addr))[0].mod_start;
  mod_range_end   = (L4_MB_MOD_PTR(mbi->mods_addr))[mbi->mods_count-1].mod_end;
}

/**
 * init region map
 */
static void
init_region(l4util_mb_info_t *mbi)
{
  region_add(bios_area_start, bios_area_end, ".BIOS area");
  region_add(bootstrap_start, bootstrap_end, ".bootstrap");
  region_add(mod_range_start, mod_range_end, ".Modules Memory");
  region_add(0,               RAM_BASE,      ".Deep Space");
#ifndef XEN
  region_add(RAM_BASE + mem_high, ~0U,       ".Deep Space");
#endif
}

#if defined(ARCH_x86) || defined(ARCH_amd64)
/**
 * setup Kernel Info Page
 */
static void
init_kip(l4_kernel_info_t *l4i, l4util_mb_info_t *mbi)
{
  l4i->root_esp = (l4_umword_t) &_stack;

  if (l4_version != VERSION_FIASCO)
    {
      l4i->main_memory.start  = 0;
      l4i->main_memory.end    = (mb_info.mem_upper + 1024) << 10;
      l4i->dedicated[0].start = mb_info.mem_lower          << 10;
      l4i->dedicated[0].end   = 1                          << 20;
    }
  else
    {
      /* don't add kernel, sigma0, and roottask to dedicated[1] because
       * these modules are already extracted once the kernel is running */
      int first_module = !!kernel + !!sigma0 + !!roottask;
      l4i->dedicated[1].start =
	(L4_MB_MOD_PTR(mbi->mods_addr))[first_module].mod_start;
      l4i->dedicated[1].end =
	(L4_MB_MOD_PTR(mbi->mods_addr))[mbi->mods_count-1].mod_end;
    }

  /* set up sigma0 info */
  if (sigma0)
    {
      l4i->sigma0_eip = sigma0_start;
      l4i->sigma0_memory.start = sigma0_low;
      l4i->sigma0_memory.end   = sigma0_high;

      /* XXX UGLY HACK: Jochen's kernel can't pass args on a sigma0
         stack not within the L4 kernel itself -- therefore we use the
         kernel info page itself as the stack for sigma0.  the field
         we use is the task descriptor for the unused "ktest2" task */
      switch (l4_version)
	{
	case VERSION_L4_V2:
	case VERSION_L4_IBM:
	  l4i->sigma0_esp = 0x1060;
	  break;
	case VERSION_FIASCO:
	  l4i->sigma0_esp = sigma0_stack;
	  break;
	}
      printf("  Sigma0 config    eip:%08lx esp:%08lx [%08lx-%08lx]\n",
	     l4i->sigma0_eip, l4i->sigma0_esp,
	     l4i->sigma0_memory.start, l4i->sigma0_memory.end);
    }

  /* set up roottask info */
  if (roottask)
    {
      l4i->root_eip = roottask_start;
      l4i->root_memory.start = roottask_low;
      l4i->root_memory.end   = roottask_high;
      printf("  Roottask config  eip:%08lx esp:%08lx [%08lx-%08lx]\n", 
	     l4i->root_eip, l4i->root_esp,
	     l4i->root_memory.start, l4i->root_memory.end);
    }
}
#endif

/**
 * load module
 */
static l4_addr_t
load_module(l4_umword_t module, l4_addr_t *begin, l4_addr_t *end)
{
  exec_task_t exec_task;
  l4_addr_t entry;
  int r;
  const char *error_msg;

  exec_task.begin = 0xffffffff;
  exec_task.end   = 0;

  exec_task.mod_start = L4_VOID_PTR(mb_mod[module].mod_start);
  exec_task.task_no   = module;
  exec_task.mod_no    = module;

  r = exec_load_elf(l4_exec_read, l4_exec_read_exec, &exec_task, 
                    &error_msg, &entry);

  /* clear the image for debugging and security reasons */
  memset(L4_VOID_PTR(mb_mod[module].mod_start), 0,
         l4_round_page(mb_mod[module].mod_end) - mb_mod[module].mod_start);

  if (r)
    printf("  => can't load module (%s)\n", error_msg);

  *begin = exec_task.begin;
  *end   = exec_task.end;

  /* Only add region if non-elf module */
  if (r)
    region_add(exec_task.begin, exec_task.end, 
	       L4_CHAR_PTR(mb_mod[module].cmdline));

  return entry;
}

/**
 * copy mbi to save place for ourself
 */
static void
copy_mbi_for_ourself(l4util_mb_info_t *dst_mbi, l4util_mb_info_t *src_mbi)
{
  int i;
  char *mod_names;

  /* copy (extended) multiboot info structure */
  memcpy(dst_mbi, src_mbi, sizeof(l4util_mb_info_t));

  /* copy command line, if available */
  if (dst_mbi->flags & L4UTIL_MB_CMDLINE)
    {
      snprintf(mb_cmdline, sizeof(mb_cmdline), "%s",
			  L4_CHAR_PTR(dst_mbi->cmdline));
      dst_mbi->cmdline = (l4_addr_t) mb_cmdline;
    }

  /* copy extended VIDEO information, if available */
  if (dst_mbi->flags & L4UTIL_MB_VIDEO_INFO)
    {
      if (src_mbi->vbe_mode_info)
	{
	  memcpy(&mb_vbe_mode, L4_VOID_PTR(src_mbi->vbe_mode_info),
		 sizeof(l4util_mb_vbe_mode_t));
	  dst_mbi->vbe_mode_info = (l4_addr_t) &mb_vbe_mode;
	}

      /* copy VBE controller info structure */
      if (src_mbi->vbe_ctrl_info)
	{
	  memcpy(&mb_vbe_ctrl, L4_VOID_PTR(src_mbi->vbe_ctrl_info),
		 sizeof(l4util_mb_vbe_ctrl_t));
	  dst_mbi->vbe_ctrl_info = (l4_addr_t) &mb_vbe_ctrl;
	}
    }

  /* copy module descriptions */
  memcpy(mb_mod, L4_VOID_PTR(dst_mbi->mods_addr),
	 dst_mbi->mods_count * sizeof (l4util_mb_mod_t));
  dst_mbi->mods_addr = (l4_addr_t) mb_mod;

  /* copy command lines of modules */
  mod_names = mb_mod_names;
  for (i = 0; i < dst_mbi->mods_count; i++)
    {
      char *name = L4_CHAR_PTR(mb_mod[i].cmdline);
      char *newstring = name ? name : "[unknown]";
      int len = strlen(newstring) + 1;

      if (mod_names - mb_mod_names + len > sizeof(mb_mod_names))
	panic("Bootstrap: mbi-cmdline overflow!");

      strcpy(mod_names, newstring);
      if (name)
	mb_mod[i].cmdline = (l4_addr_t)mod_names;
      mod_names += len;
    }
}


/**
 * copy mbi from bootstrap to roottask
 * also add the l4_version variable to command-line
 * don't copy kernel, sigma0 and roottask module
 */
static inline void*
lin_alloc(l4_size_t size, l4_uint8_t **ptr)
{
  void *ret = *ptr;
  *ptr += (size + 3) & ~3;;
  return ret;
}

static void
copy_mbi_for_roottask(l4_umword_t imagecopy_addr)
{
  int                  i;
  char                 *new_mb_cmdline;
  l4util_mb_vbe_ctrl_t *new_mb_vbe_ctrl;
  l4util_mb_vbe_mode_t *new_mb_vbe_mode;
  l4util_mb_info_t     *new_mb_info;
  l4util_mb_mod_t      *new_mb_mods;
  char                 *new_mb_modnames;
  l4_ssize_t           size_names;

  /* XXX hacky, use first 4 pages to save mbi for roottask */
  l4_uint8_t *p   = (l4_uint8_t*)(roottask_start - 0x4000);

  memset(p, 0, 0x4000);

  new_mb_info     = lin_alloc(sizeof(l4util_mb_info_t), &p);
  new_mb_cmdline  = lin_alloc(sizeof(mb_cmdline), &p);
  new_mb_vbe_mode = lin_alloc(sizeof(l4util_mb_vbe_mode_t), &p);
  new_mb_vbe_ctrl = lin_alloc(sizeof(l4util_mb_vbe_ctrl_t), &p);
  new_mb_mods     = lin_alloc(sizeof(l4util_mb_mod_t)*mb_info.mods_count-3, &p);
  new_mb_modnames = lin_alloc(/* dummy */0x1000, &p);

  size_names      = (l4_addr_t)roottask_start - (l4_addr_t)new_mb_modnames;

  if (size_names < 0)
    panic("Bootstrap: Not enough space for module info");

  /* copy multiboot info structure */
  memcpy(new_mb_info, &mb_info, sizeof(l4util_mb_info_t));

  /* copy roottask command line, if available */
  if (((L4_MB_MOD_PTR(mb_info.mods_addr)+roottask_module)->cmdline))
    {
      /* Place l4_version as the first argument to not disturb the
       * configuration parser in the roottask */
      char *cl = L4_CHAR_PTR 
	         ((L4_MB_MOD_PTR(mb_info.mods_addr)+roottask_module)->cmdline);
      char *p = strchr(cl, ' ');

      snprintf(new_mb_cmdline, sizeof(mb_cmdline), "%.*s l4_version=%d %s",
	       p ? (unsigned) (p-cl) : (unsigned) sizeof(mb_cmdline),
	       cl, l4_version, p ? p : "");

      new_mb_info->cmdline = (l4_addr_t)new_mb_cmdline;
      new_mb_info->flags  |= L4UTIL_MB_CMDLINE;
    }

  /* copy extended VIDEO information, if available */
  if (mb_info.flags & L4UTIL_MB_VIDEO_INFO)
    {
      if (mb_info.vbe_mode_info)
	{
	  memcpy(new_mb_vbe_mode, L4_VOID_PTR(mb_info.vbe_mode_info),
		 sizeof(l4util_mb_vbe_mode_t));
	  new_mb_info->vbe_mode_info = (l4_addr_t) new_mb_vbe_mode;
	}

      if (mb_info.vbe_ctrl_info)
	{
	  memcpy(new_mb_vbe_ctrl, L4_VOID_PTR(mb_info.vbe_ctrl_info),
		 sizeof(l4util_mb_vbe_ctrl_t));
	  new_mb_info->vbe_ctrl_info = (l4_addr_t) new_mb_vbe_ctrl;
	}
    }

  /* Tell roottask where to find its unmodified image copy */
  new_mb_info->syms.e.addr = imagecopy_addr;

  /* Copy module description. Clean the gap between end-of-module and
   * beginning of the next page (mostly for debugging). */
  for (i = 0; i < mb_info.mods_count - 3; i++)
    {
      memcpy(&new_mb_mods[i], L4_MB_MOD_PTR(mb_info.mods_addr) + i + 3,
	     sizeof(l4util_mb_mod_t));
      memset(L4_VOID_PTR(new_mb_mods[i].mod_end), 0,
	     l4_round_page(new_mb_mods[i].mod_end) - new_mb_mods[i].mod_end);
    }
  new_mb_info->mods_addr  = (l4_addr_t) new_mb_mods;
  new_mb_info->mods_count = mb_info.mods_count - 3;

  /* Copy command lines of modules. */
  for (i = 0; i < mb_info.mods_count - 3; i++)
    {
      const char *cmdline  = L4_CHAR_PTR(mb_mod[i + 3].cmdline);
      const char *new_args = get_args_module(i + 3);
      int sz;

      new_mb_mods[i].cmdline = (l4_addr_t)new_mb_modnames;
      if (new_args)
	{
	  const char *org_args = strchr(cmdline, ' ');
	  /* Replace old command line by new one */
	  sz = snprintf(new_mb_modnames, size_names, "%.*s %s",
			org_args ? (unsigned)(org_args-cmdline) 
			    : MOD_NAME_MAX,
	                        cmdline ? cmdline : "[unknown]",
				new_args) + 1;
	  printf("Replaced \n \"%s\" by \n \"%s\"\n", 
	      cmdline, L4_CHAR_PTR(new_mb_mods[i].cmdline));
	}
      else
	{
	  sz = snprintf(new_mb_modnames, size_names, "%s",
			cmdline ? cmdline : "[unknown]") + 1;
	}

      if (sz >= size_names)
	panic("Bootstrap: Not enough space for modules cmdlines");

      new_mb_modnames += sz;
      size_names      -= sz;
    }
}

static void
roottask_save_symbols_lines(l4_umword_t *e_addr)
{
  /* To transfer the symbol and lines info to the roottask with the minimal
   * effort in bootstrap, we just copy the whole roottask image to a new
   * location a just set the syms.e.addr field in the mbi. Roottask can then
   * figure out the rest itself, load the symbols/lines info and release
   * the memory. */

  int size = mb_mod[roottask_module].mod_end 
           - mb_mod[roottask_module].mod_start;
  l4_addr_t start = l4_round_page(mod_range_end);

  region_add(start, start + size, ".Roottask syms/lines (copy)");

  memcpy((void *)start, L4_VOID_PTR(mb_mod[roottask_module].mod_start), size);

  *e_addr  = start;
}


#ifdef IMAGE_MODE

#ifdef COMPRESS
#include "uncompress.h"
#endif

typedef struct
{
  l4_uint32_t  start;
  l4_uint32_t  size;
  l4_uint32_t  size_uncompressed;
  l4_uint32_t  name;
} mod_info;

extern mod_info _module_info_start[];
extern mod_info _module_info_end[];

extern l4util_mb_mod_t _modules_mbi_start[];
extern l4util_mb_mod_t _modules_mbi_end[];

static void
construct_mbi(l4util_mb_info_t *mbi)
{
  int i;
  l4util_mb_mod_t *mods = _modules_mbi_start;

  mbi->mods_count  = _module_info_end - _module_info_start;
  mbi->flags      |= L4UTIL_MB_MODS;
  mbi->mods_addr   = (l4_addr_t)mods;

  assert(mbi->mods_count >= 2);

  for (i = 0; i < mbi->mods_count; i++)
    {
#ifdef COMPRESS
      l4_addr_t image =
	 (l4_addr_t)decompress(L4_CONST_CHAR_PTR(_module_info_start[i].name),
                               L4_VOID_PTR(_module_info_start[i].start),
                               _module_info_start[i].size,
                               _module_info_start[i].size_uncompressed);
#else
      l4_addr_t image = _module_info_start[i].start;
#endif
      mods[i].mod_start = image;
      mods[i].mod_end   = image + _module_info_start[i].size_uncompressed - 1;
      printf("mod%02u: %08x-%08x: %s\n",
	     i, mods[i].mod_start, mods[i].mod_end,
	     L4_CHAR_PTR(_module_info_start[i].name));
    }
}
#endif /* IMAGE_MODE */

#ifdef REALMODE_LOADING
l4_addr_t _mod_end;

static void
move_modules_to_modaddr(void)
{
  int i, count = _module_info_end - _module_info_start;
  l4_uint8_t *a = (l4_uint8_t *)_mod_addr;

  /* patch module_info structure with new start values */
  for (i = 0; i < count; i++)
    {
      memcpy(a, L4_VOID_PTR(_module_info_start[i].start),
	    _module_info_start[i].size);
      _module_info_start[i].start = (l4_uint32_t)(l4_addr_t)a;
      a += l4_round_page(_module_info_start[i].size);
    }
  _mod_end = l4_round_page(a);
}
#else
extern char _module_data_start;
extern char _module_data_end;
l4_addr_t _mod_end   = (l4_addr_t)&_module_data_end;
#endif /* REALMODE_LOADING */

/**
 * \brief  Startup, started from crt0.S
 */
void
startup(l4util_mb_info_t *mbi, l4_umword_t flag, 
	void *realmode_or_xen_si_pointer)
{
  l4_kernel_info_t * l4i;
#if defined(ARCH_x86) || defined(ARCH_amd64)
  l4util_mb_info_t kernel_mbi;
#endif

#if defined(ARCH_x86) || defined(ARCH_amd64)
#ifdef REALMODE_LOADING
  mbi = init_loader_mbi(realmode_or_xen_si_pointer, 0);
  move_modules_to_modaddr();
#else
# ifdef XEN
  mbi = init_loader_mbi(NULL, MEMORY << 10);
# else
  assert(flag == L4UTIL_MB_VALID); /* we need to be multiboot-booted */
# endif
#endif
#else /* arm */
#ifdef ARCH_arm
  mbi = init_loader_mbi(NULL, MEMORY << 10);
#else
#error Unknown arch!
#endif
#endif

  /* parse command line */
  if (mbi->flags & L4UTIL_MB_CMDLINE)
    {
#if defined(ARCH_x86) || defined(ARCH_amd64)
      int comport = 1;
      const char *s;

      if ((s = strstr(L4_CHAR_PTR(mbi->cmdline), " -comport=")))
	comport = strtoul(s+10,0,0);

      if (strstr(L4_CHAR_PTR(mbi->cmdline), " -serial"))
	com_cons_init(comport);

      if (strstr(L4_CHAR_PTR(mbi->cmdline), " -hercules") && have_hercules())
        hercules = 1;
#endif

      if (strstr(L4_CHAR_PTR(mbi->cmdline), " -sigma0"))
	sigma0 = 1;

      if (strstr(L4_CHAR_PTR(mbi->cmdline), " -roottask"))
	roottask = 1;
    }

#ifdef IMAGE_MODE
  construct_mbi(mbi);
#endif

  /* We need at least two boot modules */
  assert(mbi->flags & L4UTIL_MB_MODS);
  assert(mbi->flags & L4UTIL_MB_MEMORY);
  /* We have at least the L4 kernel and the first user task */
  assert(mbi->mods_count >= 2);
  assert(mbi->mods_count <= MODS_MAX);

  /* we're not an L4 task yet -- we're just a GRUB-booted kernel! */
  puts("\nL4 Bootstrapper");

#ifdef ARCH_x86
  /* Limit memory, we cannot really handle more right now. In fact, the
   * problem is roottask. It maps as many superpages/pages as it gets.
   * After that, the remaining pages are mapped using l4sigma0_map_anypage()
   * with a receive window of L4_WHOLE_ADDRESS_SPACE. In response Sigma0
   * could deliver pages beyond the 3GB user space limit. */
  if (mbi->mem_upper << 10 > 3048UL << 20)
    mbi->mem_upper = 3048UL << 10;
#endif

  /* copy Multiboot data structures, we still need to a safe place
   * before playing with memory we don't own and starting L4 */
  copy_mbi_for_ourself(&mb_info, mbi);

  /* --- Shouldn't touch original Multiboot parameters after here. -- */

  /* Guess the amount of installed memory from the bootloader. We already
   * need that value because (1) we have to initialize the memmap for module
   * loading and (2) we have to check for a valid L4 kernel. Later (in init.c),
   * mem_high is determined from the sigma0 server. */
  setup_memory_range(&mb_info);

  /* patch modules with content given at command line */
  if (mb_info.flags & L4UTIL_MB_CMDLINE)
    {
      const char *s = L4_CONST_CHAR_PTR(mb_info.cmdline);
      while ((s = strstr(s, " -patch=")))
	patch_module(&s, &mb_info);

      s = L4_CONST_CHAR_PTR(mb_info.cmdline);
      while ((s = strstr(s, " -arg=")))
	args_module(&s, &mb_info);
    }

  /* We initialize the memmap here to observe loading tasks before starting
   * the L4 kernel. Later in init.c, we re-initialize the memmap */
  init_memmap(&mb_info);

  /* We initialize the region map here to find overlapping regions and
   * report useful error messages. roottask will use its own region map */
  init_region(&mb_info);

  /* setup kernel PART ONE */
  if (kernel)
    {
      printf("  Loading ");
      print_module_name(L4_CONST_CHAR_PTR(mb_mod[kernel_module].cmdline),
			"[KERNEL]");
      putchar('\n');

      kernel_start = load_module(kernel_module, &kernel_low, &kernel_high);

      /* check for L4 version */
      l4_version = check_l4_version((unsigned char *)kernel_low,
				    (unsigned char *)kernel_high,
				    &abi_version);
    }

  /* setup sigma0 */
  if (sigma0)
    {
      printf("  Loading ");
      print_module_name(L4_CONST_CHAR_PTR(mb_mod[sigma0_module].cmdline),
			 "[SIGMA0]");
      putchar('\n');

      sigma0_start = load_module(sigma0_module, &sigma0_low, &sigma0_high);
      sigma0_stack =
	(l4_addr_t)(sigma0_init_stack + sizeof(sigma0_init_stack));
    }

  /* setup roottask */
  if (roottask)
    {
      l4_umword_t imagecopy_addr;

      printf("  Loading ");
      print_module_name(L4_CONST_CHAR_PTR(mb_mod[roottask_module].cmdline),
			 "[ROOTTASK]");
      putchar('\n');

      /* Save symbol/lines information */
      roottask_save_symbols_lines(&imagecopy_addr);

      roottask_start = load_module(roottask_module,
				   &roottask_low, &roottask_high);
      roottask_stack = (l4_addr_t)roottask_init_stack +
				   sizeof(roottask_init_stack);
      copy_mbi_for_roottask(imagecopy_addr);
    }

  /* setup kernel PART TWO (special kernel initialization) */
  switch (l4_version)
    {
#if defined(ARCH_x86)
    case VERSION_L4_V2:
      /* init old GMD-Version */
      l4i = (l4_kernel_info_t *) kernel_start;
      {
        l4_addr_t dummy;
        init_l4_gmd(l4i, &dummy);
      }
      break;

    case VERSION_L4_IBM:
      l4i = (l4_kernel_info_t *) kernel_start;
      init_ibm_nucleus(l4i);
      break;
#endif

    case VERSION_FIASCO:
      init_fiasco(&l4i);
      break;

#ifdef ARCH_x86
    case VERSION_L4_KA:
      init_hazelnut(&l4i);
      break;
#endif

    default:
      l4i = 0;
      printf("  Could not identify version of the kernel\n");
    }

#if defined(ARCH_x86) || defined(ARCH_amd64)
  /* setup multi boot info structure for kernel */
  kernel_mbi = mb_info;
  kernel_mbi.flags = L4UTIL_MB_MEMORY;
    if (mb_mod[kernel_module].cmdline)
      {
	kernel_mbi.cmdline = mb_mod[kernel_module].cmdline;
	kernel_mbi.flags  |= L4UTIL_MB_CMDLINE;
      }

    /* setup the L4 kernel info page before booting the L4 microkernel:
     * patch ourselves into the booter task addresses */
    init_kip(l4i, &mb_info);

    if (abi_version == VERSION_ABI_V4)
      {
	printf ("  Initializing v4 KIP\n");
	init_v4_kip (l4i);
      }
#endif

  regions_dump();

  printf("  Starting kernel ");
  print_module_name(L4_CONST_CHAR_PTR(mb_mod[kernel_module].cmdline), 
		    "[KERNEL]");
  printf(" at 0x%08lx\n", kernel_start);

#if defined(ARCH_x86)
  asm volatile
    ("pushl $exit ; jmp *%3"
     :
     : "a" (L4UTIL_MB_VALID),
       "b" (&kernel_mbi),
       "S" (realmode_or_xen_si_pointer),
       "r" (kernel_start));

#elif defined(ARCH_amd64)

  asm volatile
    ("push $exit; jmp *%2"
     :
     : "S" (L4UTIL_MB_VALID), "D" (&kernel_mbi), "r" (kernel_start));

#elif defined(ARCH_arm)

  {
    typedef void (*startup_func)(void);
    startup_func f = (startup_func)kernel_start;
    l4_addr_t p = (l4_addr_t)kernel_start;
    int i;

    p &= 0xfffff000; /* 4k align */

    /* Search for the KIP in the kernel image */
    l4_kernel_info_t *kip = NULL;

    for (i = 1; i <= 5; i++)
      {
	l4_kernel_info_t *k = (l4_kernel_info_t *)(p + i * 0x1000);
	if (k->magic == L4_KERNEL_INFO_MAGIC)
	  {
	    kip = k;
	    break;
	  }
      }

    if (!kip)
      panic("No kernel info page found in kernel binary image!");

    printf("KIP is at %p\n", kip);

    kip->sigma0_eip = sigma0_start;
    kip->root_eip   = roottask_start;

    l4_kernel_info_mem_desc_t *md = l4_kernel_info_get_mem_descs(kip);
    l4_kernel_info_set_mem_desc(md, RAM_BASE, RAM_BASE + ((MEMORY << 20) - 1),
				l4_mem_type_conventional, 0, 0);
    l4_kernel_info_set_mem_desc(++md, kernel_low, kernel_high - 1,
				l4_mem_type_reserved, 0, 0);
    l4_kernel_info_set_mem_desc(++md, sigma0_low, sigma0_high - 1,
				l4_mem_type_reserved, 0, 0);
    l4_kernel_info_set_mem_desc(++md, roottask_low, roottask_high - 1,
				l4_mem_type_reserved, 0, 0);

    if (_module_data_start < _module_data_end)
      {
	l4_kernel_info_set_mem_desc(++md,
	                            (l4_umword_t)_module_data_start,
				    (l4_umword_t)_module_data_end - 1,
				    l4_mem_type_bootloader, 0, 0);
      }

    printf("Starting kernel... (%x)\n", 0);
    f();
    panic("Returned from kernel?!");
  }

#else

#error "How to enter the kernel?"

#endif

  /*NORETURN*/
}

static int
l4_exec_read(void *handle, l4_addr_t file_ofs, void *buf, l4_size_t size,
	     l4_size_t *out_actual)
{
  exec_task_t *e = (exec_task_t *) handle;
  memcpy(buf, (char*)(e->mod_start) + file_ofs, size);

  *out_actual = size;
  return 0;
}

static int
l4_exec_read_exec(void * handle,
		  l4_addr_t file_ofs, l4_size_t file_size,
		  l4_addr_t mem_addr, l4_size_t mem_size,
		  exec_sectype_t section_type)
{
  exec_task_t *exec_task = (exec_task_t*)handle;

  if (! (section_type & EXEC_SECTYPE_ALLOC))
    return 0;

  if (! (section_type & (EXEC_SECTYPE_ALLOC|EXEC_SECTYPE_LOAD)))
    return 0;

  if (mem_addr < exec_task->begin)
    exec_task->begin = mem_addr;
  if (mem_addr + mem_size > exec_task->end)
    exec_task->end = mem_addr + mem_size;

  region_add(mem_addr, mem_addr + mem_size,
             mb_mod[exec_task->mod_no].cmdline
	       ? L4_CONST_CHAR_PTR(mb_mod[exec_task->mod_no].cmdline)
	       :  ".[Unknown]");

  memcpy((void *) mem_addr,
         (char*)(exec_task->mod_start) + file_ofs, file_size);
  if (file_size < mem_size)
    memset((void *) (mem_addr + file_size), 0, mem_size - file_size);

  return 0;
}
