#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include <l4/sys/kdebug.h>
#include <l4/util/cpu.h>
#include <l4/util/port_io.h>
#include <l4/util/reboot.h>

#include "base_critical.h"
#include "ARCH-x86/serial.h"
#include "version.h"

extern int hercules;	/* provided by startup.c */

static void direct_cons_putchar(unsigned char c, int to_hercules);
static int  direct_cons_try_getchar(void);
       int  have_hercules(void);
       void reboot(void);

#ifndef XEN
int
putchar(int c)
{
  direct_cons_putchar(c, hercules);
  com_cons_putchar(c);

  return c;
}
#endif

int
getchar(void)
{
  int c;
  do
    {
      c = com_cons_try_getchar();
      if (c == -1)
	c = direct_cons_try_getchar();
    } while (c == -1);
  return c;
}

void __attribute__((noreturn))
reboot(void)
{
  extern void l4util_reboot_arch(void) __attribute__((noreturn));
  l4util_reboot_arch();
}

void
_exit(int fd)
{
  extern void l4util_reboot_arch(void);
  printf("\n\033[1mReturn reboots...\033[m\n");
  getchar();
  printf("Rebooting.\n");
  reboot();
}

static void
direct_cons_putchar(unsigned char c, int to_hercules)
{
  static int ofs  = -1;
  static int esc;
  static int esc_val;
  static int attr = 0x07;
  unsigned char *vidbase = (unsigned char*)(to_hercules ? 0xb0000 : 0xb8000);

  base_critical_enter();

  if (ofs < 0)
    {
      /* Called for the first time - initialize.  */
      ofs = 80*2*24;
      direct_cons_putchar('\n', to_hercules);
    }

  switch (esc)
    {
    case 1:
      if (c == '[')
	{
	  esc++;
	  goto done;
	}
      esc = 0;
      break;

    case 2:
      if (c >= '0' && c <= '9')
	{
	  esc_val = 10*esc_val + c - '0';
	  goto done;
	}
      if (c == 'm')
	{
	  attr = esc_val ? 0x0f : 0x07;
	  goto done;
	}
      esc = 0;
      break;
    }

  switch (c)
    {
    case '\n':
      bcopy(vidbase+80*2, vidbase, 80*2*24);
      bzero(vidbase+80*2*24, 80*2);
      /* fall through... */
    case '\r':
      ofs = 0;
      break;

    case '\t':
      ofs = (ofs + 8) & ~7;
      break;

    case '\033':
      esc = 1;
      esc_val = 0;
      break;

    default:
      /* Wrap if we reach the end of a line.  */
      if (ofs >= 80)
	direct_cons_putchar('\n', to_hercules);

      /* Stuff the character into the video buffer. */
	{
	  volatile unsigned char *p = vidbase + 80*2*24 + ofs*2;
	  p[0] = c;
	  p[1] = attr;
	  ofs++;
	}
      break;
    }

done:
  base_critical_leave();
}

/** check if hercules console is present */
int
have_hercules(void)
{
  unsigned short *p, p_save;
  int count = 0;
  unsigned long delay;

  /* check for video memory */
  p = (unsigned short*)0xb0000;
  p_save = *p;
  *p = 0xaa55; if (*p == 0xaa55) count++;
  *p = 0x55aa; if (*p == 0x55aa) count++;
  *p = p_save;
  if (count != 2)
    return 0;

  /* check for I/O ports (HW cursor) */
  l4util_out8(0x0f, 0x3b4);
  l4util_iodelay();
  l4util_out8(0x66, 0x3b5);
  for (delay=0; delay<0x100000UL; delay++)
    asm volatile ("nop" : :);
  if (l4util_in8(0x3b5) != 0x66)
    return 0;

  l4util_iodelay();
  l4util_out8(0x0f, 0x3b4);
  l4util_iodelay();
  l4util_out8(0x99, 0x3b5);

  for (delay=0; delay<0x10000; delay++)
    l4util_iodelay();
  if (l4util_in8(0x3b5) != 0x99)
    return 0;

  /* reset position */
  l4util_out8(0xf, 0x3b4);
  l4util_iodelay();
  l4util_out8(0x0, 0x3b5);
  l4util_iodelay();
  return 1;
}

/** poor man's getchar, only returns raw scan code */
int
direct_cons_try_getchar(void)
{
  unsigned status, scan_code;

  base_critical_enter();

  l4util_cpu_pause();

  /* Wait until a scan code is ready and read it. */
  status = l4util_in8(0x64);
  if ((status & 0x01) == 0)
    {
      base_critical_leave();
      return -1;
    }
  scan_code = l4util_in8(0x60);

  /* Drop mouse events */
  if ((status & 0x20) != 0)
    {
      base_critical_leave();
      return -1;
    }

  base_critical_leave();
  return scan_code;
}

void *memmove(void *dest, const void *src, size_t n);
void *memmove(void *dest, const void *src, size_t n)
{
  bcopy(src, dest, n);
  return dest;
}

void *memmem(const void* haystack, size_t hl, const void* needle, size_t nl);
void *memmem(const void* haystack, size_t hl, const void* needle, size_t nl)
{
  int i;
  if (nl>hl)
    return 0;
  for (i=hl-nl+1; i; --i)
    {
      if (!memcmp(haystack,needle,nl))
	return (char*)haystack;
      ++haystack;
    }
  return 0;
}

int memcmp(const void *dst, const void *src, size_t count);
int memcmp(const void *dst, const void *src, size_t count)
{
  int r;
  const unsigned char *d=dst;
  const unsigned char *s=src;
  ++count;
  while (--count)
    {
      if ((r=(*d - *s)))
	return r;
      ++d;
      ++s;
    }
  return 0;
}

