INTERFACE [amd64]:

EXTENSION class Space
{
private:
  void switch_ldt();
};

IMPLEMENTATION[amd64]:

#include <cassert>
#include "l4_types.h"
#include "kmem.h"
#include "mem_unit.h"
#include "cpu_lock.h"
#include "lock_guard.h"
#include "paging.h"

#include <cstring>
#include "config.h"
#include "kmem.h"

IMPLEMENT inline NEEDS ["cpu.h", "kmem.h"]
Space *
Space::current()
{
  return reinterpret_cast<Space*>(Kmem::phys_to_virt(Cpu::get_pdbr()));
}

IMPLEMENT inline NEEDS ["cpu.h", "kmem.h"]
void
Space::make_current()
{
  Cpu::set_pdbr((Mem_layout::pmem_to_phys(this)));
}

//---------------------------------------------------------------------------
IMPLEMENTATION [amd64-!smas]:

#include "cpu.h"
#include "kmem.h"
#include "logdefs.h"

PUBLIC inline
void
Space::ldt_addr(void *addr)
{
  Pd_entry *p = _pml4.lookup(Mem_layout::Ldt_addr)
    		  ->pdp()->lookup(Mem_layout::Ldt_addr)
		    ->pdir()->lookup(Mem_layout::Ldt_addr);
  assert(!p->valid());
  assert(!((Address)addr & 1));

  *p = (Address)addr & ~1;
}

PUBLIC inline
Address
Space::ldt_addr() const
{
  Pd_entry p = _pml4.lookup(Mem_layout::Ldt_addr)
  		 ->pdp()->lookup(Mem_layout::Ldt_addr)
		   ->pdir()->entry(Mem_layout::Ldt_addr);
  assert(!p.valid());

  return p.raw();
}

PUBLIC inline
void
Space::ldt_size(Mword size)
{
  Pd_entry *p = _pml4.lookup(Mem_layout::Ldt_size)
    		  ->pdp()->lookup(Mem_layout::Ldt_size)
		    ->pdir()->lookup(Mem_layout::Ldt_size);
  assert(!p->valid());
  assert(!(size & 1));

  *p = size & ~1;
}

PUBLIC inline
Mword
Space::ldt_size() 
{
  Pd_entry p = _pml4.lookup(Mem_layout::Ldt_size)
    		 ->pdp()->lookup(Mem_layout::Ldt_size)
		   ->pdir()->entry(Mem_layout::Ldt_size);
  assert(!p.valid());

  return p.raw();
}

IMPLEMENT inline NEEDS["kmem.h","logdefs.h"]
void
Space::switchin_context()
{
  // never switch to kernel space (context of the idle thread)
  if (this == (Space*)Kmem::dir())
    return;

  bool need_flush_tlb = false;
  Pdir *_pdir	      = _pml4.lookup(Kmem::ipc_window(0))->pdp()
    				->lookup(Kmem::ipc_window(0))->pdir();
  unsigned index      = Pdir::virt_to_idx(Kmem::ipc_window(0));

  if ((*_pdir)[index] || (*_pdir)[index + 1])
    {
      (*_pdir)[index]     = 0;
      (*_pdir)[index + 1] = 0;
      need_flush_tlb  = true;
    }

  index = Pdir::virt_to_idx(Kmem::ipc_window(1));

  if ((*_pdir)[index] || (*_pdir)[index + 1])
    {
      (*_pdir)[index]     = 0;
      (*_pdir)[index + 1] = 0;
      need_flush_tlb  = true;
    }

  if (need_flush_tlb || this != Space::current())
    {
      switchin_lipc_kip_pointer();
      CNT_ADDR_SPACE_SWITCH;
      make_current();
      switch_ldt();
    }
}

//------------------------------------------------------------------------
IMPLEMENTATION [amd64-!segments-!smas]:

IMPLEMENT inline
void
Space::switch_ldt()
{
}

IMPLEMENTATION [amd64]:
/** Constructor.  Creates a new address space and registers it with
  * Space_index.
  *
  * Registration may fail (if a task with the given number already
  * exists, or if another thread creates an address space for the same
  * task number concurrently).  In this case, the newly-created
  * address space should be deleted again.
  *
  * @param new_number Task number of the new address space
  */
PROTECTED
Space::Space(unsigned number)
{
  _pml4.clear();	// initialize to zero
  Kmem::dir_init(&_pml4);	// copy current shared kernel page directory

  // Scribble task/chief numbers into an unused part of the page directory.
  // Assure that the page directory entry is not valid!
  
  (_pml4[Mem_layout::Space_index]) = number << 8;
  (_pml4[Mem_layout::Chief_index]) = Space_index(number).chief() << 8;
  // register in task table
  Space_index::add(this, number);
}

/*
 * The following functions are all no-ops on native amd64.
 * Pages appear in an address space when the corresponding PTE is made
 * ... unlike Fiasco-UX which needs these special tricks
 */

IMPLEMENT inline
void
Space::page_map (Address, Address, Address, unsigned)
{}

IMPLEMENT inline
void
Space::page_protect (Address, Address, unsigned)
{}

IMPLEMENT inline
void
Space::page_unmap (Address, Address)
{}

IMPLEMENT inline NEEDS ["kmem.h"]
void Space::kmem_update (void *addr)
{
  Unsigned64 attr = Pd_entry::Valid | Pd_entry::Writable |
		    Pd_entry::Referenced;

  Pdir *kpdir = Kmem::dir()->lookup((Address)addr)
		->pdp()->lookup((Address)addr)
		  ->pdir();
  unsigned i = kpdir->virt_to_idx((Address)addr);
  
  Pdir *_pdir = _pml4.lookup((Address)addr)
		        ->alloc_pdp(Mapped_allocator::allocator(), attr)
		          ->lookup((Address)addr)
		            ->alloc_pdir(Mapped_allocator::allocator(), attr);
  
  (*_pdir)[i] = (*kpdir)[i];
}

/**
 * Copy multiple PDEs between two address spaces.
 * Be aware that some PDEs are reserved and contain thread-local information
 * like space_index, chief number, IO bitmap which must not be overwritten.
 * Callers are expected to know which slots are safe to copy.
 */
IMPLEMENT inline NEEDS [<cassert>, "mem_unit.h", "paging.h"]
void
Space::remote_update (const Address loc_addr, const Space *rem,
                      const Address rem_addr, size_t n)
{
  bool tlb_flush    = false;

  Pdir *ldir = _pml4.lookup(loc_addr)
		        ->pdp()
		          ->lookup(loc_addr)
		            ->pdir();

  Pdir *rdir = rem->pml4()->lookup(rem_addr)
    			->pdp()
			  ->lookup(rem_addr)
			    ->pdir();

  unsigned loc_slot = Pdir::virt_to_idx(loc_addr),
           rem_slot = Pdir::virt_to_idx(rem_addr);

  // PDE boundary check
  assert (loc_slot + n <= 512); // XXX make constants
  assert (rem_slot + n <= 512); // XXX make constants
  
  while (n--)
    {
      if ((*ldir)[loc_slot])
        tlb_flush = true;
      
      // XXX: If we copy a PDE and it maps either a superpage with PDE_GLOBAL
      // or a pagetable with one or multiple PTE_GLOBAL entries, these won't
      // be flushed on a TLB flush. Suggested solution: When mapping a page
      // with global bit, a PDE_NO_LONGIPC bit should be set in the
      // corresponding PDE and this bit should be checked here before copying.

      // This loop seems unnecessary, but remote_update is also used for
      // updating the long IPC window.
      // Now consider following scenario with super pages:
      // Sender A makes long IPC to receiver B.
      // A setups the IPC window by reading the pagedir slot from B in an 
      // temporary register. Now the sender is preempted by C. Then C unmaps 
      // the corresponding super page from B. C switch to A back, using 
      // switch_to, which clears the IPC window pde slots from A. BUT then A 
      // write the  content of the temporary register, which contain the now 
      // invalid pde slot, in his own page directory and starts the long IPC.
      // Because no pagefault will happen, A will write to now invalid memory.
      // So we compare after storing the pde slot, if the copy is still
      // valid. And this solution is much faster than grabbing the cpu lock,
      // when updating the ipc window.
      for(;;) {

	const volatile Pd_entry *val = rdir->index(rem_slot);
	(*ldir)[loc_slot] =  val->raw();
	
	if(EXPECT_TRUE((*ldir)[loc_slot] == val->raw()))
	  break;
      }
      
      loc_slot++;
      rem_slot++;

    }

  if (tlb_flush)
    Mem_unit::tlb_flush();
}

// --------------------------------------------------------------------
IMPLEMENTATION[{ia32,amd64}-segments]:

PRIVATE inline
void
Space::free_ldt_memory()
{
  if (ldt_addr())
    Mapped_allocator::allocator()->free(Config::PAGE_SHIFT,
				  reinterpret_cast<void*>(ldt_addr()));
}

// --------------------------------------------------------------------
IMPLEMENTATION[!smas]:

IMPLEMENT inline
void
Space::update_small(Address, bool)
{}

