/**
 *    \file    dice/src/be/BEResourceRequestFunction.cpp
 *    \brief    contains the implementation of the class CBEResourceRequestFunction
 *
 *    \date    04/25/2004
 *    \author    Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/* Copyright (C) 2001-2003 by
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */
#include "BEResourceRequestFunction.h"
#include "BEContext.h"
#include "BEHeaderFile.h"
#include "BEImplementationFile.h"
#include "BEComponent.h"
#include "BEType.h"
#include "fe/FEResourceDeclarator.h"
#include "fe/FEResourceProperty.h"
#include "fe/FEInterface.h"

#include "ProgramOptions.h"
#include "TypeSpec-Type.h"

IMPLEMENT_DYNAMIC(CBEResourceRequestFunction);

CBEResourceRequestFunction::CBEResourceRequestFunction()
 : CBEInterfaceFunction()
{
    IMPLEMENT_DYNAMIC_BASE(CBEResourceRequestFunction, CBEInterfaceFunction);
}

CBEResourceRequestFunction::CBEResourceRequestFunction(CBEResourceRequestFunction &src)
 : CBEInterfaceFunction(src)
{
    IMPLEMENT_DYNAMIC_BASE(CBEResourceRequestFunction, CBEInterfaceFunction);
}

/** destroys the function object */
CBEResourceRequestFunction::~CBEResourceRequestFunction()
{
}

/**    \brief creates the function from a given frontend interface
 *    \param pFEInterface the interface to extract information from
 *    \param pContext the context of the creation
 *    \return true if creation was successful
 */
bool CBEResourceRequestFunction::CreateBackEnd(CFEInterface *pFEInterface,
        CBEContext *pContext)
{
    pContext->SetFunctionType(FUNCTION_RESOURCE_REQUEST);
    // set target file name
    SetTargetFileName(pFEInterface, pContext);
    // set name
    m_sName = pContext->GetNameFactory()->GetFunctionName(pFEInterface, pContext);

    if (!CBEInterfaceFunction::CreateBackEnd(pFEInterface, pContext))
        return false;

    /* a request has the form
     *
     * <reservation>
     *  <alternative>
     *   <resource>
     *    <name></name>
     *    <properties>
     *     <property>
     *      <name></name>
     *      <value></value>
     *     </property>
     *    </properties>
     *   </resource>
     *  </alternative>
     * </reservation>
     *
     * Since currently we cannot parse alternative resource descriptions,
     * we have only one alternative here.
     * First try to append the resource strings, this way we can later
     * easily check if there were resources at all.
     */
    m_sRequest.Empty();

    VectorElement *pIter = pFEInterface->GetFirstResource();
    CFEResourceDeclarator *pResource;
    while ((pResource = pFEInterface->GetNextResource(pIter)) != 0)
    {
        if (!AppendResource(pResource, pContext))
            return false;
    }
    if (!m_sRequest.IsEmpty())
    {
        String s = "<reservation><alternative>" + m_sRequest +
            "</alternative></reservation>";
        m_sRequest = s;
    }

    // return type -> set to IPC reply code
    if (!SetReturnVar(false, 1, TYPE_CHAR_ASTERISK, String("handles"), pContext))
    {
        VERBOSE("CBEDispatchFunction::CreateBE failed because return var could not be set\n");
        return false;
    }

     return true;
}

/**    \brief append a resource description to the request string
 *    \param pFEResource the resource declarator describing the resource
 *    \param pContext the context of the creation
 *    \return true if successful
 */
bool CBEResourceRequestFunction::AppendResource(CFEResourceDeclarator *pFEResource,
    CBEContext *pContext)
{
    m_sRequest += "<resource><name>" + pFEResource->GetName() + "</name>";
    m_sRequest += "<properties>";

    VectorElement *pIter = pFEResource->GetFirstProperty();
    CFEResourceProperty *pProperty;
    while ((pProperty = pFEResource->GetNextProperty(pIter)) != 0)
    {
        m_sRequest += "<property><name>";
        m_sRequest += pProperty->GetName();
        m_sRequest += "</name><value>";
        m_sRequest += pProperty->GetValue();
        m_sRequest += "</value></property>";
    }

    m_sRequest += "</properties></resource>";
    return true;
}

/** \brief test fi this function should be written
 *  \param pFile the file to write to
 *  \param pContext the context of the write operation
 *  \return true if this function should be written
 *
 * A server loop is only written at the component's side.
 */
bool CBEResourceRequestFunction::DoWriteFunction(CBEFile * pFile, CBEContext * pContext)
{
    if (!pContext->IsOptionSet(PROGRAM_GENERATE_RESOURCE_REQUEST))
        return false;
    if (pFile->IsKindOf(RUNTIME_CLASS(CBEHeaderFile)))
        if (!IsTargetFile((CBEHeaderFile*)pFile))
            return false;
    if (pFile->IsKindOf(RUNTIME_CLASS(CBEImplementationFile)))
        if (!IsTargetFile((CBEImplementationFile*)pFile))
            return false;
    return pFile->GetTarget()->IsKindOf(RUNTIME_CLASS(CBEComponent));
}

/** \brief determines the direction, the server loop sends to
 *  \return DIRECTION_OUT
 */
int CBEResourceRequestFunction::GetSendDirection()
{
    return DIRECTION_OUT;
}

/** \brief determined the direction the server loop receives from
 *  \return DIRECTION_IN
 */
int CBEResourceRequestFunction::GetReceiveDirection()
{
    return DIRECTION_IN;
}
