/*
 * Copyright (C) 2024 Kernkonzept GmbH.
 * Author(s): Frank Mehnert <frank.mehnert@kernkonzept.com>
 *
 * License: see LICENSE.spdx (in this directory or the directories above)
 */
/**
 * \file
 */

#include <mutex>

#include <l4/cxx/list_alloc>
#include <l4/drivers/hw_mmio_register_block>
#include <l4/mbox-bcm2835/mbox.h>
#include <l4/re/mmio_space>
#include <l4/sys/cxx/consts>
#include <l4/vbus/vbus>
#include <l4/sys/kdebug.h>

#include "device_factory.h"
#include "guest.h"
#include "irq_dt.h"
#include "irq_svr.h"
#include "mmio_device.h"
#include "vm_ram.h"

namespace {

using namespace Vdev;

/**
 * Device for connecting to the bcm2835 mailbox service.
 *
 * This device emulation proxies guest accesses to this device to the bcm2835
 * mailbox to the 'mbox' service. A device tree entry needs to look like this:
 *
 * \code{.dtb}
 *   mailbox: mailbox@7e00b880 {
 *       compatible = "brcm,bcm2835-mbox";
 *       reg = <0x7e00b880 0x40>;
 *       interrupts = <0 1>;
 *       #mbox-cells = <0>;
 *   };
 * \endcode
 *
 * This is the same layout as used by native Linux.
 */

static Dbg trace(Dbg::Dev, Dbg::Trace, "bcm2835");
static Dbg warn(Dbg::Dev, Dbg::Warn, "bcm2835");

struct Io_irq_svr : public Irq_svr
{
  using Irq_svr::Irq_svr;
};

class Bcm2835_mbox_dev
: public Vdev::Device,
  public Vmm::Mmio_device_t<Bcm2835_mbox_dev>,
  public Bcm2835_mbox
{
  struct alignas(16) Shm_block
  {
    l4_uint32_t phys;
    l4_uint32_t _align[3];
    l4_uint32_t msg_size;
  };

public:
  Bcm2835_mbox_dev(L4::Cap<L4vbus::Vbus> vbus, Vmm::Vm_ram *ram,
               Vdev::Device_lookup *devs, Vdev::Dt_node const &node)
  : Bcm2835_mbox(vbus, warn),
    _ram(ram)
  {
    L4vbus::Icu icu_dev;
    L4Re::chksys(vbus->root().device_by_hid(&icu_dev, "L40009"), "Request ICU");
    auto icu = L4Re::chkcap(L4Re::Util::cap_alloc.alloc<L4::Icu>(),
                            "Allocate ICU cap");
    L4Re::chksys(icu_dev.vicu(icu), "Request ICU cap");

    // Match the interrupt resource provided by the IO vbus device with the
    // corresponding device tree entry. The other resources were already handled
    // by Bcm2835_mbox().
    for (unsigned i = 0; i < _mbox_num_resources; ++i)
      {
        l4vbus_resource_t res;
        L4Re::chksys(_mbox.get_resource(i, &res), "Get mbox device info");
        if (res.type == L4VBUS_RESOURCE_IRQ)
          {
            Vdev::Irq_dt_iterator it(devs, node);
            if (it.next(devs) < 0)
              L4Re::throw_error(
                -L4_EINVAL, "No device tree entry for interrupt resource");
            if (!it.ic_is_virt())
              L4Re::throw_error(
                -L4_EINVAL, "Bcm2835 requires a virtual interrupt controller");
            int io_irq = res.start;
            int dt_irq = it.irq();
            cxx::Ref_ptr<Gic::Ic> const &ic = it.ic();
            auto *irq_source = ic->get_irq_src_handler(dt_irq);
            if (irq_source)
              L4Re::throw_error_fmt(
                -L4_EINVAL, "IO device '%s': irq 0x%x -> 0x%x already registered",
                node.get_name(), io_irq, dt_irq);
            // forward interrupt generated by IO device at pin `io_irq` to the
            // guest at pin `dt_irq`
            auto irq_svr =
              cxx::make_ref_obj<Io_irq_svr>(devs->vmm()->registry(),
                                            icu, io_irq, ic, dt_irq);
            irq_svr->eoi();
            _irq = std::move(irq_svr);
          }
      }

    detect_dma_offset();
  }

  virtual ~Bcm2835_mbox_dev() = default;

  char const *dev_name() const override { return "Bcm2835_mbox"; }

  /**
   * Detect the DMA offset as required by the device.
   *
   * This value is important because guest Linux makes use of it.
   */
  void detect_dma_offset()
  {
    Soc_rev board_rev = get_board_rev();
    if (board_rev.new_style())
      {
        switch (board_rev.type())
          {
          case 0x11: // 4B
            if (board_rev.revision() <= 2)
              break;
            _dma_offset = 0ULL;
            break;
          case 0x13: // 400
            _dma_offset = 0ULL;
            break;
          }
      }

    warn.printf("DMA offset = %08llx\n", _dma_offset);
  }

  l4_uint32_t read(unsigned reg, char size, unsigned /* cpu_id */)
  {
    trace.printf("\033[32mBcm2835::read offset=%02x size=%u\033[m\n", reg, size);
    if (size != 2)
      L4Re::throw_error(-L4_EINVAL, "Can only handle 32-bit reads");

    l4_uint32_t value = _regs[reg];
    if (reg == Mbox0_read && value != 0)
      {
        // We trust the service that it returned us the pointer to a message
        // which we submitted in the write() function below.
        l4_uint32_t shared_offs = (value & ~Channel_mask) - 16;
        l4_uint32_t channel = value & Channel_mask;
        l4_uint8_t *shared_addr = _data_virt + shared_offs;
        l4_uint32_t *shared_words = reinterpret_cast<l4_uint32_t *>(shared_addr);
        l4_uint32_t msg_gst_phys_addr = shared_words[0] - _dma_offset;
        Vmm::Guest_addr msg_gst_phys{msg_gst_phys_addr};
        auto *msg_gst = _ram->guest2host<l4_uint32_t *>(msg_gst_phys);
        auto *msg = reinterpret_cast<Bcm2835_mbox::Message<> *>(msg_gst);
        // Copy letter content from shared area to guest RAM; actual message
        // starts at 4th word.
        l4_uint32_t msg_size = shared_words[1];
        msg->from_mbox_after_dma(shared_words + 4, shared_words[2]);
        l4_cache_flush_data(reinterpret_cast<l4_addr_t>(msg_gst),
                            reinterpret_cast<l4_addr_t>(msg_gst) + msg_size);
        value = shared_words[0] & channel;
        // Free the region allocated in the write() function.
        free_shared_data(shared_offs, msg_size + 16);
      }

    return value;
  }

  void write(unsigned reg, char size, l4_umword_t value, unsigned /* cpu_id */)
  {
    trace.printf("\033[32mBcm2835::write offset=%02x size=%u value=%0lx\033[m\n",
                 reg, size, value);
    if (size != 2)
      L4Re::throw_error(-L4_EINVAL, "Can only handle 32-bit writes");

    if (reg == Mbox1_write)
      {
        l4_uint32_t msg_gst_phys_addr =
          (static_cast<l4_uint32_t>(value) & ~Channel_mask) - _dma_offset;
        l4_uint32_t channel = value & Channel_mask;
        Vmm::Guest_addr msg_gst_phys{msg_gst_phys_addr};
        auto const *msg_gst = _ram->guest2host<l4_uint32_t *>(msg_gst_phys);
        auto const *msg = reinterpret_cast<Bcm2835_mbox::Message<> const *>(msg_gst);
        l4_uint32_t msg_size = msg->hdr.bytes;
        // Allocate a region on the shared data for this request. The guest can
        // performs multiple requests in parallel. Freeing is done in the read()
        // function.
        // The +16 is for storing the physical letter address before the letter!
        l4_size_t shared_offs = alloc_shared_data(msg_size + 16);
        l4_uint8_t *shared_addr =
          static_cast<l4_uint8_t*>(_data_virt + shared_offs);
        l4_uint32_t *shared_words = reinterpret_cast<l4_uint32_t*>(shared_addr);
        // Remember physical letter address, entire size and "out size".
        shared_words[0] = value & ~Channel_mask;
        shared_words[1] = msg_size;
        shared_words[2] = msg->tag.size_out;
        // Copy letter content from guest RAM to shared area; actual message
        // starts at 4th word.
        msg->to_mbox_before_dma(shared_words + 4);
        value = (shared_offs + 16) | channel;
      }

    _regs[reg] = value;
  }

private:
  // Need to access mailbox data from the guest address space.
  Vmm::Vm_ram *_ram;
  // IRQ server to forward the IRQ from the external mbox device to the guest.
  cxx::Ref_ptr<Io_irq_svr> _irq;
  // DMA offset
  l4_uint64_t _dma_offset = 0xc0000000ULL;
};

struct F : Vdev::Factory
{
  cxx::Ref_ptr<Vdev::Device> create(Vdev::Device_lookup *devs,
                                    Vdev::Dt_node const &node) override
  {
    // Use a dedicated vbus capability because the mbox service implements the
    // vbus protocol -- like the IO server.
    auto cap = Vdev::get_cap<L4vbus::Vbus>(node, "l4vmm,vbus-dev");
    if (!cap)
      return nullptr;

    auto c = Vdev::make_device<Bcm2835_mbox_dev>(cap, devs->ram().get(), devs, node);
    devs->vmm()->register_mmio_device(c, Vmm::Region_type::Virtual, node, 0);
    return c;
  }
};

static F f;
static Vdev::Device_type t = { "brcm,bcm2835-mbox", nullptr, &f };

} // namespace
