/**
 * \file
 * Scheduler object functions.
 */
/*
 * (c) 2008-2009 Adam Lackorzynski <adam@os.inf.tu-dresden.de>,
 *               Alexander Warg <warg@os.inf.tu-dresden.de>
 *     economic rights: Technische Universität Dresden (Germany)
 *
 * License: see LICENSE.spdx (in this directory or the directories above)
 */
#pragma once

#include <l4/sys/kernel_object.h>
#include <l4/sys/ipc.h>

/**
 * \defgroup l4_scheduler_api Scheduler
 * \ingroup  l4_kernel_object_api
 * C interface of the Scheduler kernel object, see L4::Scheduler for the C++
 * interface.
 *
 * The Scheduler interface allows a client to manage CPU resources. The API
 * provides functions to query scheduler information, check the online state
 * of CPUs, query CPU idle time and to start threads on defined CPU sets.
 *
 * The scheduler offers a virtual device IRQ which triggers when the number of
 * online cores changes, e.g. due to hotplug events. In contrast to hardware
 * IRQs, this IRQ implements a limited functionality:
 *  - Only IRQ line 0 is supported, no MSI vectors.
 *  - The IRQ is edge-triggered and the IRQ mode cannot be changed.
 *  - As the IRQ is edge-triggered, it does not have to be explicitly unmasked.
 *
 * It depends on the platform, which hotplug events actually trigger the IRQ.
 * Many platforms only support triggering the IRQ when a CPU core different
 * from the boot CPU goes online.
 *
 * \includefile{l4/sys/scheduler.h}
 */

/**
 * Supported scheduler classes.
 * \ingroup l4_scheduler_api
 * \hideinitializer
 */
enum L4_scheduler_classes
{
  /** Fixed-priority scheduler */
  L4_SCHEDULER_CLASS_FIXED_PRIO = 1UL << 1,
  /** Weighted fair queuing scheduler */
  L4_SCHEDULER_CLASS_WFQ        = 1UL << 2,
};

/**
 * CPU sets.
 * \ingroup l4_scheduler_api
 */
typedef struct l4_sched_cpu_set_t
{
  /**
   * Combination of granularity and offset.
   *
   * The granularity defines how many CPUs each bit in map describes. And the
   * offset is the number of the first CPU described by the first bit in the
   * bitmap.
   * \pre offset must be a multiple of 2^granularity.
   *
   * | MSB              |                 LSB |
   * |:-----------------|--------------------:|
   * | 8bit granularity | 24bit offset ..     |
   */
  l4_umword_t gran_offset;

  /**
   * Bitmap of CPUs.
   */
  l4_umword_t map;

#ifdef __cplusplus
  /// \return Get granularity value
  unsigned char granularity() const { return gran_offset >> 24; }
  /// \return Get offset value
  unsigned offset() const { return gran_offset & 0x00ffffff; }
  /**
   * Set offset and granularity.
   *
   * \param granularity  Granularity in log2 notation.
   * \param offset       Offset. Must be a multiple of 2^granularity.
   */
  void set(unsigned char granularity, unsigned offset)
  {
    gran_offset = (static_cast<l4_umword_t>(granularity) << 24)
      | (offset & 0x00ffffff);
  }
#endif
} l4_sched_cpu_set_t;

/**
 *
 * \ingroup l4_scheduler_api
 *
 * \param offset       Offset. Must be a multiple of 2^granularity.
 * \param granularity  Granularity in log2 notation.
 * \param map          Bitmap of CPUs, defaults to 1 in C++.
 *
 * \return CPU set.
 */
L4_INLINE l4_sched_cpu_set_t
l4_sched_cpu_set(l4_umword_t offset, unsigned char granularity,
                 l4_umword_t map L4_DEFAULT_PARAM(1)) L4_NOTHROW;

/**
 * \ingroup l4_scheduler_api
 * \copybrief L4::Scheduler::info
 *
 * \param         scheduler  Scheduler object.
 * \param[out]    cpu_max    Maximum number of CPUs ever available. Optional,
 *                           can be NULL.
 * \param[in,out] cpus       \a cpus.offset is first CPU of interest.
 *                           \a cpus.granularity (see l4_sched_cpu_set_t).
 *                           \a cpus.map Bitmap of online CPUs. Must not be
 *                           NULL.
 *
 * \retval 0           Success.
 * \retval -L4_ERANGE  The given CPU offset is larger than the maximum number
 *                     of CPUs.
 */
L4_INLINE l4_msgtag_t
l4_scheduler_info(l4_cap_idx_t scheduler, l4_umword_t *cpu_max,
                  l4_sched_cpu_set_t *cpus)
                 L4_NOTHROW __attribute__((nonnull (3)));

/**
 * \ingroup l4_scheduler_api
 * \copybrief L4::Scheduler::info
 *
 * \param         scheduler  Scheduler object.
 * \param[out]    cpu_max    Maximum number of CPUs ever available. Optional,
 *                           can be NULL.
 * \param[in,out] cpus       \a cpus.offset is first CPU of interest.
 *                           \a cpus.granularity (see l4_sched_cpu_set_t).
 *                           \a cpus.map Bitmap of online CPUs. Must not be
 *                           NULL.
 * \param[out]    sched_classes  A bitmap of available scheduling classes (see
 *                               L4_scheduler_classes). Optional, can be NULL.
 *
 * \retval 0           Success.
 * \retval -L4_ERANGE  The given CPU offset is larger than the maximum number
 *                     of CPUs.
 *
 * This function delivers the same information as #l4_scheduler_info plus the
 * available scheduler classes (see #L4_scheduler_classes).
 */
L4_INLINE l4_msgtag_t
l4_scheduler_info_with_classes(l4_cap_idx_t scheduler, l4_umword_t *cpu_max,
                               l4_sched_cpu_set_t *cpus,
                               l4_umword_t *sched_classes)
                              L4_NOTHROW __attribute__((nonnull (3)));

/**
 * \internal
 */
L4_INLINE l4_msgtag_t
l4_scheduler_info_u(l4_cap_idx_t scheduler, l4_umword_t *cpu_max,
                    l4_sched_cpu_set_t *cpus, l4_umword_t *sched_classes,
                    l4_utcb_t *utcb) L4_NOTHROW __attribute__((nonnull (3, 5)));


/**
 * Scheduler parameter set.
 * \ingroup l4_scheduler_api
 */
typedef struct l4_sched_param_t
{
  /** CPU affinity. */
  l4_sched_cpu_set_t affinity;
  /**
   * Priority for scheduling.
   * The kernel supports priorities for userland threads in the range of 1..255.
   * Priority 0 is reserved for the kernel.
   */
  l4_umword_t        prio;
  /** Timeslice in micro seconds. */
  l4_umword_t        quantum;
} l4_sched_param_t;

/**
 * Construct scheduler parameter.
 * \ingroup l4_scheduler_api
 * \param prio     Thread priority (1-255).
 * \param quantum  Timeslice in micro seconds.
 *
 * The l4_sched_param_t::affinity of the returned value contains all CPUs.
 */
L4_INLINE l4_sched_param_t
l4_sched_param(unsigned prio,
               l4_umword_t quantum L4_DEFAULT_PARAM(0)) L4_NOTHROW;

/**
 * \ingroup l4_scheduler_api
 * \copybrief L4::Scheduler::run_thread
 *
 * \param scheduler  Scheduler object.
 * \copydetails L4::Scheduler::run_thread
 */
L4_INLINE l4_msgtag_t
l4_scheduler_run_thread(l4_cap_idx_t scheduler,
                        l4_cap_idx_t thread, l4_sched_param_t const *sp)
                       L4_NOTHROW __attribute__((nonnull));

/**
 * \internal
 */
L4_INLINE l4_msgtag_t
l4_scheduler_run_thread_u(l4_cap_idx_t scheduler, l4_cap_idx_t thread,
                          l4_sched_param_t const *sp, l4_utcb_t *utcb)
                         L4_NOTHROW __attribute__((nonnull));

/**
 * \ingroup l4_scheduler_api
 * \copybrief L4::Scheduler::idle_time
 *
 * \param scheduler   Scheduler object.
 * \copydetails L4::Scheduler::idle_time
 */
L4_INLINE l4_msgtag_t
l4_scheduler_idle_time(l4_cap_idx_t scheduler, l4_sched_cpu_set_t const *cpus,
                       l4_kernel_clock_t *us)
                      L4_NOTHROW __attribute__((nonnull));

/**
 * \internal
 */
L4_INLINE l4_msgtag_t
l4_scheduler_idle_time_u(l4_cap_idx_t scheduler, l4_sched_cpu_set_t const *cpus,
                         l4_kernel_clock_t *us, l4_utcb_t *utcb)
                        L4_NOTHROW __attribute__((nonnull));



/**
 * \ingroup l4_scheduler_api
 * \copybrief L4::Scheduler::is_online
 *
 * \param scheduler  Scheduler object.
 * \param cpu        CPU number whose online status should be queried.
 *
 * \retval true   The CPU is online.
 * \retval false  The CPU is offline
 */
L4_INLINE int
l4_scheduler_is_online(l4_cap_idx_t scheduler, l4_umword_t cpu) L4_NOTHROW;

/**
 * \internal
 */
L4_INLINE int
l4_scheduler_is_online_u(l4_cap_idx_t scheduler, l4_umword_t cpu,
                         l4_utcb_t *utcb) L4_NOTHROW __attribute__((nonnull));



/**
 * Operations on the Scheduler object.
 * \ingroup l4_scheduler_api
 * \hideinitializer
 * \internal
 */
enum L4_scheduler_ops
{
  L4_SCHEDULER_INFO_OP       = 0UL, /**< Query infos about the scheduler */
  L4_SCHEDULER_RUN_THREAD_OP = 1UL, /**< Run a thread on this scheduler */
  L4_SCHEDULER_IDLE_TIME_OP  = 2UL, /**< Query idle time for the scheduler */
};

/*************** Implementations *******************/

L4_INLINE l4_sched_cpu_set_t
l4_sched_cpu_set(l4_umword_t offset, unsigned char granularity,
                 l4_umword_t map) L4_NOTHROW
{
  l4_sched_cpu_set_t cs;
  cs.gran_offset = ((l4_umword_t)granularity << 24) | (offset & 0x00ffffff);
  cs.map         = map;
  return cs;
}

L4_INLINE l4_sched_param_t
l4_sched_param(unsigned prio, l4_umword_t quantum) L4_NOTHROW
{
  l4_sched_param_t sp;
  sp.prio     = prio;
  sp.quantum  = quantum;
  sp.affinity = l4_sched_cpu_set(0, ~0, 1);
  return sp;
}


L4_INLINE l4_msgtag_t
l4_scheduler_info_u(l4_cap_idx_t scheduler, l4_umword_t *cpu_max,
                    l4_sched_cpu_set_t *cpus, l4_umword_t *sched_classes,
                    l4_utcb_t *utcb) L4_NOTHROW
{
  l4_msg_regs_t *m = l4_utcb_mr_u(utcb);
  l4_msgtag_t res;

  m->mr[0] = L4_SCHEDULER_INFO_OP;
  m->mr[1] = cpus->gran_offset;

  res = l4_ipc_call(scheduler, utcb, l4_msgtag(L4_PROTO_SCHEDULER, 2, 0, 0), L4_IPC_NEVER);

  if (l4_msgtag_has_error(res))
    return res;

  cpus->map = m->mr[0];

  if (cpu_max)
    *cpu_max = m->mr[1];

  if (sched_classes)
    *sched_classes = m->mr[2];

  return res;
}

L4_INLINE l4_msgtag_t
l4_scheduler_run_thread_u(l4_cap_idx_t scheduler, l4_cap_idx_t thread,
                          l4_sched_param_t const *sp, l4_utcb_t *utcb) L4_NOTHROW
{
  l4_msg_regs_t *m = l4_utcb_mr_u(utcb);
  m->mr[0] = L4_SCHEDULER_RUN_THREAD_OP;
  m->mr[1] = sp->affinity.gran_offset;
  m->mr[2] = sp->affinity.map;
  m->mr[3] = sp->prio;
  m->mr[4] = sp->quantum;
  m->mr[5] = l4_map_obj_control(0, 0);
  m->mr[6] = l4_obj_fpage(thread, 0, L4_CAP_FPAGE_RWS).raw;

  return l4_ipc_call(scheduler, utcb, l4_msgtag(L4_PROTO_SCHEDULER, 5, 1, 0), L4_IPC_NEVER);
}

L4_INLINE l4_msgtag_t
l4_scheduler_idle_time_u(l4_cap_idx_t scheduler, l4_sched_cpu_set_t const *cpus,
                         l4_kernel_clock_t *us, l4_utcb_t *utcb) L4_NOTHROW
{
  l4_msg_regs_t *v = l4_utcb_mr_u(utcb);
  l4_msgtag_t res;

  v->mr[0] = L4_SCHEDULER_IDLE_TIME_OP;
  v->mr[1] = cpus->gran_offset;
  v->mr[2] = cpus->map;

  res = l4_ipc_call(scheduler, utcb,
                    l4_msgtag(L4_PROTO_SCHEDULER, 3, 0, 0), L4_IPC_NEVER);

  if (l4_msgtag_has_error(res))
    return res;

  *us = v->mr64[l4_utcb_mr64_idx(0)];

  return res;
}


L4_INLINE int
l4_scheduler_is_online_u(l4_cap_idx_t scheduler, l4_umword_t cpu,
                         l4_utcb_t *utcb) L4_NOTHROW
{
  l4_sched_cpu_set_t s;
  l4_msgtag_t r;
  s.gran_offset = cpu;
  r = l4_scheduler_info_u(scheduler, NULL, &s, NULL, utcb);
  if (l4_msgtag_has_error(r) || l4_msgtag_label(r) < 0)
    return 0;

  return s.map & 1;
}


L4_INLINE l4_msgtag_t
l4_scheduler_info(l4_cap_idx_t scheduler, l4_umword_t *cpu_max,
                  l4_sched_cpu_set_t *cpus) L4_NOTHROW
{
  return l4_scheduler_info_u(scheduler, cpu_max, cpus, NULL, l4_utcb());
}

L4_INLINE l4_msgtag_t
l4_scheduler_info_with_classes(l4_cap_idx_t scheduler, l4_umword_t *cpu_max,
                               l4_sched_cpu_set_t *cpus,
                               l4_umword_t *sched_classes) L4_NOTHROW
{
  return l4_scheduler_info_u(scheduler, cpu_max, cpus, sched_classes, l4_utcb());
}

L4_INLINE l4_msgtag_t
l4_scheduler_run_thread(l4_cap_idx_t scheduler,
                        l4_cap_idx_t thread, l4_sched_param_t const *sp) L4_NOTHROW
{
  return l4_scheduler_run_thread_u(scheduler, thread, sp, l4_utcb());
}

L4_INLINE l4_msgtag_t
l4_scheduler_idle_time(l4_cap_idx_t scheduler, l4_sched_cpu_set_t const *cpus,
                       l4_kernel_clock_t *us) L4_NOTHROW
{
  return l4_scheduler_idle_time_u(scheduler, cpus, us, l4_utcb());
}

L4_INLINE int
l4_scheduler_is_online(l4_cap_idx_t scheduler, l4_umword_t cpu) L4_NOTHROW
{
  return l4_scheduler_is_online_u(scheduler, cpu, l4_utcb());
}
