/*
 * Copyright (C) 2018 Kernkonzept GmbH.
 * Author(s): Philipp Eppelt <philipp.eppelt@kernkonzept.com>
 *
 * This file is distributed under the terms of the GNU General Public
 * License, version 2.  Please see the COPYING-GPL-2 file for details.
 */
#include "debug.h"
#include "device.h"
#include "pci_bus.h"
#include "device_factory.h"
#include "guest.h"

namespace Vdev {

void
Pci_bus_bridge::init_bus_range(Dt_node const &node)
{
  int sz;
  auto bus_range = node.get_prop<fdt32_t>("bus-range", &sz);
  if (sz != 2)
    {
      Err().printf("Bus range property of Pci_host_bridge has invalid size\n");
      return;
    }

  trace().printf("Init host bridge: Found 'bus-range' 0x%x - 0x%x\n",
                 fdt32_to_cpu(bus_range[0]), fdt32_to_cpu(bus_range[1]));

  auto *const hdr = header();
  hdr->secondary_bus_num = (l4_uint8_t)fdt32_to_cpu(bus_range[0]);
  hdr->subordinate_bus_num = (l4_uint8_t)fdt32_to_cpu(bus_range[1]);
}

}; // namespace Vdev

namespace {

using namespace Vdev;

struct F : Factory
{
  static Dbg info() { return Dbg(Dbg::Dev, Dbg::Info, "PCI bus"); }

  cxx::Ref_ptr<Device> create(Device_lookup *devs, Dt_node const &node) override
  {
    info().printf("Creating host bridge\n");

    if (!node.has_prop("bus-range"))
      {
        info().printf(
          "Bus range not specified in device tree. Device not created.\n");
        return nullptr;
      }

    auto dev = make_device<Pci_bus_bridge>(devs->vbus());
    dev->init_bus_range(node);
    // XXX add vBus-device proxies to memmap & iomap; future change.

    // If the Vbus provides a PCI bus with a host bridge, we don't need the
    // virtual one.
    if (!dev->is_io_pci_host_bridge_present())
      dev->register_device(dev);

    auto io_cfg_connector = make_device<Pci_bus_cfg_io>(dev);
    devs->vmm()->register_io_device(Vmm::Io_region(0xcf8, 0xcff), io_cfg_connector);

    info().printf("Created & Registered the PCI host bridge\n");
    return dev;
  }
}; // struct F

static F f;
static Device_type t = {"virt-pci-bridge", nullptr, &f};

}; // namespace
